/*****************************************************************************
*
* File: OsmoseCore.cpp
*
* Project: Osmose emulator.
*
* Description: This file contains Osmose main loop, handling keyboard, SDL
* event, and hardware emulation.
*
* Author: Vedder Bruno
* Date: 23/01/2005, 14h13
*
* URL: http://bcz.emu-france.com/
*****************************************************************************/
#include "OsmoseCore.h"


SN76489           *p;

// This method is not from OsmoseCore
void sndCallback(void *ud, unsigned char *s, int len);

/*--------------------------------------------------------------------*/
/* This method is the OsmoseCore constructor.			      */
/*--------------------------------------------------------------------*/
OsmoseCore::OsmoseCore(const char *rom_f)
{
    nmi		   = false;
    sound_shot_toggle = false;
    snd_started = false;
    screenshotNbr  = 0;
    tileshotNbr	   = 0;
    soundshotNbr   = 0;
    saveStateSlotNbr = minSaveStateSlot;
    rom_filename   = rom_f;
    gain = 0.00f;
	sdl_window_width = 256;
	sdl_window_height = 192;

    // Setup default parameters.
    if (emu_opt.default_config == true)
    {
        oc = new OsmoseConfiguration();
    }
    else
    {
        oc = new OsmoseConfiguration(emu_opt.ini_file);
    }

    /* Instanciate TextWriter. */
    tw = new TextWriter();

    /* Note: After instantiation, opt.MachineType contains SMS or GAMEGEAR type. */
    mem  = new MemoryMapper(rom_filename, oc->getUserHomeFolder().c_str());
    if (opt.mapperType == CodemasterMapper) mem->setMapperType(CodemasterMapper);
    if (opt.mapperType == KoreanMapper) mem->setMapperType(KoreanMapper);

	env  = new SmsEnvironment();
    cpu  = new Z80(*env);

    p    = new SN76489(3579545, 22050);

    if (opt.MachineType == SMS)
    {
        v    = new VDP(cpu, opt.ntsc);   // Instanciate ntsc or pal SMS VDP.
        iom  = new IOMapper(*v, *p);
    }
    else
    {
        // Instanciate ntsc (always) GAMEGEAR VDP. Force emulator option too.
        v = new VDP_GG(cpu, true);
        opt.ntsc = true;
        iom  = new IOMapper_GG(*v, *p);
    }

    env->setIOMapper(iom);
    env->setMemoryMapper(mem);
    env->setVDP(v);
    env->setCPU(cpu);
    wavW = NULL;
    game_name = mem->getROMName();

#ifdef BUILT_IN_DEBUGGER
    dbg  = new SmsDebugger();
	v->attachDebugEventListener(dbg);
    this->attachDebugEventListener(dbg);
	cpu->attachDebugEventListener(dbg);
	mem->attachDebugEventListener(dbg);
	iom->attachDebugEventListener(dbg);
    dasm = new Z80Dasm(*env);
    dbg->setMemoryMapper(mem);
    dbg->setEnvironment(env);
    dbg->setVDP(v);
    dbg->setIOMapper(iom);
    dbg->setDasm(dasm);
    dbg->setCPU(cpu);
    old_cycles = 0;
#endif

    setupSDLVideo();

	/* Instanciation of input device is done here beacause JOYSTICK needs to be */
	/* initialised with SDL_init first.											*/
	switch (opt.inputType)
    {
        case DEFAULT_PAD:
            input= new PadInputDevice(iom, oc);
		break;

        case PADDLE:
	    	input= new PaddleInputDevice(iom, oc);
		break;

		case JOYSTICK:
			input= new JoystickInputDevice(iom, oc);
		break;

		default:
            input= new PadInputDevice(iom, oc);
		break;
    }

    cout << "Using as sms input device: " << input->getInputDeviceName() << endl;

    if (emu_opt.sound == true)
    {
        setupSDLAudio();
    }
}

/*--------------------------------------------------------------------*/
/* This method handles SDL keyboard events.			      */
/*--------------------------------------------------------------------*/
void OsmoseCore::handleSDLKeyboardEvent(SDL_Event e)
{
    ostringstream msg;
    int k = e.key.keysym.sym;
 
    if(e.type == SDL_KEYDOWN)
    {           
        if ( k == oc->PAUSE_KEY)
		{
            nmi = true;
			return;
        }

        if ( k == oc->TILESHOT_KEY)
		{
            captureTiles(v);
			return;
        }

        if ( k == oc->SCREENSHOT_KEY)
		{
            captureScreen();
			return;
		}

		/* Save State key is not configurable. */
        if ( k == SDLK_F12)
		{
            bool succes = false;
			succes = saveSaveState ();
			msg << "State saved in slot:" << saveStateSlotNbr;
			if (succes) tw->addText(msg.str().c_str(), 180);
			else tw->addText ("Save state failed!", 180);
			return;
		}

		/* Load State key is not configurable. */
        if ( k == SDLK_F11)
		{
            bool succes = false;
			succes = loadSaveState ();
    		msg << "State loaded from slot:" << saveStateSlotNbr;
			if (succes) tw->addText(msg.str().c_str(), 180);
			else tw->addText("Load state failed!", 180);
			return;
		}

		/* Decrement Save state slot key is not configurable. */
        if ( k == SDLK_KP_MINUS)
		{
			if (saveStateSlotNbr > minSaveStateSlot)
			{
				saveStateSlotNbr -=1;
			}
			msg << "using save state slot:" << saveStateSlotNbr;
			tw->addText(msg.str().c_str(), 180);
			return;
		}

		/* Increment Save state slot key is not configurable. */
        if ( k == SDLK_KP_PLUS)
		{
			if (saveStateSlotNbr < maxSaveStateSlot)
			{
				saveStateSlotNbr +=1;
			}
			msg << "using save state slot:" << saveStateSlotNbr;
			tw->addText(msg.str().c_str(), 180);
			return;
		}
#ifdef BUILT_IN_DEBUGGER
        if ( k == oc->DEBUGGER_KEY)
		{
            throwDebugEvent(DbgEvtUserTrigger, "OsmoseCore", "User trigger.");
	    	return;
        }
#endif
    }

	/* Key released event.*/
    if(e.type == SDL_KEYUP)
    {
		if (k == oc->SOUNDSHOT_KEY)
		{
            if (emu_opt.sound == true)
	    	{
        		if (sound_shot_toggle == true)
				{
	            	// We were recording sound, so stop it now.
                    tw->addText("stopping sound recording.", 120);
       	   	    	wavW->close();
		    		delete wavW; // To avoid memory leak.
		    		sound_shot_toggle = false;
				}
				else
				{
        	    	char snd_shot_filename[256];
	            	// We weren't recording sound, record it now, so create WaveWriter.
#ifdef __USE_UNIX98
                    sprintf(snd_shot_filename,"%ssnd/%s(%d).wav", oc->getUserHomeFolder().c_str(),game_name.c_str(), soundshotNbr);
#else
                    sprintf(snd_shot_filename,".\\snd\\%s(%d).wav", game_name.c_str(), soundshotNbr);
#endif
		   			wavW = new WaveWriter( snd_shot_filename );
		    		sound_shot_toggle = true;
		    		soundshotNbr++;
                    tw->addText("starting sound recording!", 120);
				}
	    	}
	    	return;
		}
    }

    // We had handled general emulation key, now call specific sms handler.
    input->handleDeviceChange(e);
}

/*--------------------------------------------------------------------*/
/* This method runs one frame.									      */
/* Note about frame variable:									      */
/* This variable is the total number of frame (displayed or not !)    */
/* emulated by Osmose. This value is use for speed synchronisation at */
/* 60 Fps. That's why the value is incremented even if the frame isn't*/
/* rendered.                                                          */
/*--------------------------------------------------------------------*/
void OsmoseCore::run_frame()
{
    bool drawline    = true;
    bool played      = false;
    float snd_update = 0;
    int scanline_number;
    unsigned int over_cycles = 0;

    scanline_number = (opt.ntsc == true) ? 262 : 313; // NTSC or PAL

	if (nmi == true)
	{
		cpu->nmi();
		nmi = false;
	}

	for (v->line=0; v->line < scanline_number; v->line++)
	{
		over_cycles = cpu->run(CPU_CYCLES_PER_LINE - over_cycles);
		if (emu_opt.sound == true)
		{
			snd_update+=(float)SND_TOGGLE;  // Needed to call update_sound_buffer 367.5/frame
			played = p->run(1);
			if (sound_shot_toggle == true && played==true)
			{
				wavW->writeData(p->getLastSample());
			}

			if (snd_update > 1.0f)
			{
				played = p->run(1);
				if (played == true)
				{
					snd_update = snd_update - (float)1.0f;
				}
				if (sound_shot_toggle == true && played==true)
				{
					wavW->writeData(p->getLastSample());
				}
			}
		}
		v->update(buffer, drawline);
	} // For i = 0 to scanline_number

	tw->update(buffer, drawline);

	if (snd_started == false && emu_opt.sound == true)
	{
		// Start playing only if sound buffer is full.
		// This avoid playing silence on emulator startup.
		if (!p->getFIFOSoundBuffer()->spaceAvailable())
		{
			snd_started = true;
			SDL_PauseAudio(0); // start playing !
		}
	}
	
	// To avoid infinite loop with cycleCounter overflow.
	cpu->setCycles(0);
	drawGLFrame();

	// Trackball, Paddle need to be updated in time.
	input->updateDevice();
}


/*--------------------------------------------------------------------*/
/* This method will save as bitmap, vdp graphics tiles.		      */
/* First, a 128x224 SDL_Surface is allocated.			      */
/* Then tiles are drawn there.					      */
/* A screenshot is taken					      */
/* The Surface is freed.					      */
/* Filename is tiles_rip_ + game_name.bmp.			      */
/*--------------------------------------------------------------------*/
void OsmoseCore::captureTiles(VDP *v)
{
    int status;
    char sName[256];
    unsigned short map_p = 0;
    SDL_Surface *tiles;

    // Allocate new software surface.
    tiles = SDL_CreateRGBSurface(SDL_SWSURFACE, 128,224,16,0xF800,0x7E0,0x1f,0x0);
    if (tiles == NULL)
    {
        cerr << "Couldn't get 128x224 surface: %s" << endl << SDL_GetError() << endl;;
        cerr << "Tiles are not saved." << endl << SDL_GetError() << endl;;
    }

    // Draw tiles there.
    for (int o=0; o<28;o++)
	for (int i=0; i<16;i++)
	{
	    int tile = map_p;
	    displayTiles(tiles, v, tile, i<<3, o<<3);
	    map_p++;
	}
    SDL_UpdateRect(screen, 0, 0, 0, 0);

    // Save it !
#ifdef __USE_UNIX98
    sprintf(sName,"%stiles/gfx_%s(%d).bmp", oc->getUserHomeFolder().c_str(), game_name.c_str(), tileshotNbr);
#else
    sprintf(sName,".\\tiles\\gfx_%s(%d).bmp", game_name.c_str(), tileshotNbr);
#endif

    tileshotNbr++;
    SDL_LockSurface(tiles);
    status = SDL_SaveBMP(tiles, sName);
    SDL_UnlockSurface(screen);
    if(status == 0)
    {
       tw->addText("gfx have been saved.", 120);
    }
    else
    {
       tw->addText("fail to save gfx!", 120);

    }
    SDL_FreeSurface(tiles);
}


/*--------------------------------------------------------------------*/
/* This method draws a tile n, at position x,y, assuming that the     */
/* Surface is 128 pixels wide.					      */
/*--------------------------------------------------------------------*/
void OsmoseCore::displayTiles(SDL_Surface *s, VDP *vd, int tile, int x, int y)
{
    unsigned short *ptr;
    unsigned char col_index, p0, p1, p2, p3;
    unsigned int ti, c;

    ti = tile<<5;
    ptr = (unsigned short *)s->pixels + ((y<<7)+x );
    for(int o=0; o<8;o++)
    {
	c = (o<<2) + ti;
	p0 = vd->VRAM[c++];
	p1 = vd->VRAM[c++];
	p2 = vd->VRAM[c++];
    	p3 = vd->VRAM[c++];

	for (int i=0; i<8;i++)
	{
	    col_index = (p0 >>7) | (p1 >> 7)<<1 | (p2 >> 7)<<2 | (p3 >> 7)<<3;
	    *(unsigned short *)ptr = vd->colors[col_index];
	    ptr++;
	    p0<<=1;
	    p1<<=1;
	    p2<<=1;
	    p3<<=1;
	}
    ptr += 120; // Complement to next line, based on 256 pixel width.
    }
}

/*--------------------------------------------------------------------*/
/* This method setup our default sound.				      */
/*--------------------------------------------------------------------*/
void OsmoseCore::setupAudioFormat()
{
    format.freq     = 22050;
    format.format   = AUDIO_S16LSB;
    format.channels = 1;
    format.samples  = SAMPLE_SIZE;
    format.callback = sndCallback;
    format.userdata = NULL;
}

/*--------------------------------------------------------------------*/
/* This method is called by SDL sound system, to fill the sound buffer*/
/* s is the place to put sound data, len is length of buffer in bytes.*/
/*--------------------------------------------------------------------*/
void sndCallback(void *ud, unsigned char *s, int len)
{
	p->getWave(s, len);
}

/*--------------------------------------------------------------------*/
/* This method setup SDL video system.				      */
/*--------------------------------------------------------------------*/
void OsmoseCore::setupSDLVideo()
{
    /* Initialize SDL */
    if ( SDL_Init(SDL_INIT_NOPARACHUTE | SDL_INIT_VIDEO | SDL_INIT_TIMER | SDL_INIT_AUDIO | SDL_INIT_JOYSTICK) < 0 ) {
            cerr << "Couldn't initialize SDL: "<< endl << SDL_GetError() << endl;
            exit(1);
    }

    // On exit, call SDL_Quit().
    atexit (SDL_Quit);
    SDL_GL_SetAttribute( SDL_GL_RED_SIZE, 5 );
    SDL_GL_SetAttribute( SDL_GL_GREEN_SIZE, 6 );
    SDL_GL_SetAttribute( SDL_GL_BLUE_SIZE, 5 );
    SDL_GL_SetAttribute( SDL_GL_DOUBLEBUFFER, 1 );
    SDL_GL_SetAttribute(SDL_GL_SWAP_CONTROL, 1);
    SDL_ShowCursor(SDL_DISABLE);

    if (emu_opt.fullscreen_flag == false)
    {
        if ( (screen=SDL_SetVideoMode(sdl_window_width, sdl_window_height,16, SDL_OPENGL  | SDL_GL_DOUBLEBUFFER | SDL_SWSURFACE | SDL_RESIZABLE)) == NULL ) {
                cerr << "Couldn't set video mode: %s" << endl << SDL_GetError() << endl;;
                exit(2);
        }
    }
    else
    {
    	// Get current video resolution, and use it for fullscreen.
    	const SDL_VideoInfo *vinfo = SDL_GetVideoInfo();
		sdl_window_width = vinfo->current_w;
		sdl_window_height = vinfo->current_h;
        if ( (screen=SDL_SetVideoMode(sdl_window_width, sdl_window_height,16, SDL_FULLSCREEN | SDL_OPENGL  | SDL_GL_DOUBLEBUFFER | SDL_SWSURFACE | SDL_RESIZABLE)) == NULL ) {
                cerr << "Couldn't set video mode: %s" << endl << SDL_GetError() << endl;;
                exit(2);
        }
        SDL_ShowCursor(SDL_DISABLE);
    }

    initOpenGL(sdl_window_width, sdl_window_height);

    /*
     	Allocate SDL surface for rendering screen. The SMS screen is usually 256*192 but some
     	OpenGL implementation (all maybe?) only handle square textures. So we allocate a 256x256
     	buffer. The end of the buffer will no be written, but will be transfered into GPU ram.
     */
    buffer = SDL_CreateRGBSurface(SDL_SWSURFACE | SDL_OPENGL, 256,256,16,0xF800,0x7E0,0x1f,0x0);
    if (buffer == NULL)
    {
        cerr << "Couldn't get 256x192x16 surface: %s" << endl << SDL_GetError() << endl;;
        exit (1);
    }
    cout << "Video surfaces successfully allocated." << endl;
}

/*--------------------------------------------------------------------*/
/* This method setup SDL audio system.				      */
/*--------------------------------------------------------------------*/
void OsmoseCore::setupSDLAudio()
{
    setupAudioFormat();
    int r = SDL_OpenAudio(&format, NULL);

    if (r >= 0)
    {
        cout << "Audio device successfully opened." << endl;
		SDL_PauseAudio(1);	// Set audio to pause.
    }
    else
    {
        cerr << "Couldn't open audio device:" << endl << SDL_GetError() ;
        cerr << "Activating -nosound option." << endl;
        cerr << "Disabling  -snd_shot option." << endl;
        emu_opt.sound    = false;
    }
}

/*--------------------------------------------------------------------*/
/* This method takes a screenshot of the game. The filename is        */
/* game_name +x .bmp , where x is the number of taken screenshot,     */
/* which is incremented every time captureScreen() is called.         */
/*--------------------------------------------------------------------*/
void OsmoseCore::captureScreen()
{
    int status;
    char sName[256];

    SDL_Surface *surface;
    SDL_Rect src_rect;

    src_rect.x = 0;
    src_rect.y = 0;
    src_rect.w = 256;
    src_rect.h = 192;

    // Allocate new software 256x192 surface (OpenGL renderer uses 256x256).
    surface = SDL_CreateRGBSurface(SDL_SWSURFACE, 256,192,16,0xF800,0x7E0,0x1f,0x0);
    SDL_BlitSurface(buffer, &src_rect, surface, NULL);

#ifdef __USE_UNIX98
    sprintf(sName,"%sscreen/%s(%d).bmp", oc->getUserHomeFolder().c_str(), game_name.c_str(), screenshotNbr);
#else
    sprintf(sName,".\\screen\\%s(%d).bmp", game_name.c_str(), screenshotNbr);
#endif
    screenshotNbr++;
    SDL_LockSurface(surface);
    status = SDL_SaveBMP(surface, sName);
    SDL_UnlockSurface(surface);
    if(status == 0)
    {
       tw->addText("screenshot saved.", 120);
    }
    else
    {
       tw->addText("fail to save screenshot!", 120);
    }

    SDL_FreeSurface(surface);
}

/*--------------------------------------------------------------------*/
/* This method generates save state file. Here is the format:         */
/* File Offset - Data type.                                           */
/* 0000-0003     unsigned char[4] marker "OESS"                       */
/* 0004-0007     unsigned char[4] 0 + version that create savestate.  */
/* 0008-0019     unsigned char[18] z80_8bits_registers.               */
/* 001A-0021     unsigned short[4] z80_16bits_registers.              */
/*--------------------------------------------------------------------*/
bool OsmoseCore::saveSaveState()
{
	ostringstream save_state_name;
#ifdef __USE_UNIX98
    save_state_name <<  oc->getUserHomeFolder() << "saves/" << mem->getROMName() << "_slot_" << saveStateSlotNbr <<".sta";
#else
    	save_state_name << ".\\saves\\" << mem->getROMName() << "_slot_" << saveStateSlotNbr <<".sta";
#endif


    ofstream output_file((char *)save_state_name.str().c_str(), ios::out | ios::binary );
    if (output_file.is_open() == false )
    {
        cout << "Cant create file :"<< save_state_name.str() << endl;
	    return false;
    }

    /* Write cpu data. */
    if (!cpu->saveState( output_file ) ) {cout << "CPU save fail." << endl; return false;}

    /* Write memory mapper data. */
    if (!mem->saveState( output_file ) ) {cout << "Mem Mapper save fail." << endl; return false;}

    /* Write VDP data. */
    if (!v->saveState( output_file ) ) {cout << "VDP save fail." << endl; return false;}

    /* Write SN76489 data. */
    if (!p->saveState( output_file ) ) {cout << "PSG save fail." << endl; return false;}
    output_file.close();
    
    return true;
}

bool OsmoseCore::loadSaveState()
{
	ostringstream load_state_name;
#ifdef __USE_UNIX98
    load_state_name << oc->getUserHomeFolder() <<"saves/" << mem->getROMName() << "_slot_" << saveStateSlotNbr <<".sta";
#else
    load_state_name << ".\\saves\\" << mem->getROMName() << "_slot_" << saveStateSlotNbr <<".sta";
#endif

    ifstream input_file((char *)load_state_name.str().c_str(), ios::in | ios::binary);
    if (input_file.is_open() == false )
    {
	    return false;
    }

    /* Load cpu data. */
    if (!cpu->loadState( input_file ) ) {cout << "CPU load fail." << endl; return false;}

    /* Load memory mapper data. */
	if (!mem->loadState( input_file ) ) {cout << "Mem Mapper load fail." << endl; return false;}

	/* Load VDP data. */
    if (!v->loadState( input_file ) ) {cout << "VDP load fail." << endl; return false;}

    /* Save SN76489 data. */
    if (!p->loadState( input_file ) ) {cout << "PSG load fail." << endl; return false;}
    input_file.close();
    return true;
}

/*--------------------------------------------------------------------*/
/* This method saves Battery Backed Memory if needed.                 */
/*--------------------------------------------------------------------*/
void OsmoseCore::save_bbr()
{
    char full_name[256];

#ifdef __USE_UNIX98
    sprintf(full_name,"%sbbr/%s.bbr", oc->getUserHomeFolder().c_str(), game_name.c_str());
#else
    sprintf(full_name,".\\bbr\\%s.bbr", game_name.c_str());
#endif
    mem->save_battery_backed_memory( string(full_name) );
}

/*--------------------------------------------------------------------*/
/* This method setup OpenGL parameters, and create a handle for the   */
/* texture where the SMS screen will be mapped to.		      */
/*--------------------------------------------------------------------*/
void OsmoseCore::initOpenGL(int Width, int Height)
{
	//cout << "OpenGL initialisation." << endl;
	glClearColor(0.0, 0.0, 0.0, 0.0);
	glEnable(GL_TEXTURE_2D);
	glGenTextures(1,textureName);
   	if (glGetError() != GL_NO_ERROR) std::cout << "glGenTextures returns an OpenGL error !" << std::endl;
	glBindTexture(GL_TEXTURE_2D, textureName[0]);
    if (glGetError() != GL_NO_ERROR) std::cout << "glBindTexture returns an OpenGL error !" << std::endl;
	if (opt.videoFilter == BILINEAR)
	{
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	}
	else
	{
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
	}

	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
    if (glGetError() != GL_NO_ERROR) std::cout << "glTexParameteri returns an OpenGL error !" << std::endl;
}


/*--------------------------------------------------------------------*/
/* This method setup the view port be: 0, 0 coordinate at upper left  */
/* Corner of the window, and of the size of the window.		      */
/*--------------------------------------------------------------------*/
void OsmoseCore::prepare2DViewport(int topleft_x, int topleft_y, int bottomrigth_x, int bottomrigth_y)
{
	glEnable(GL_TEXTURE_2D);

	glViewport(topleft_x, topleft_y, bottomrigth_x, bottomrigth_y);
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();

	glOrtho(topleft_x,bottomrigth_x,bottomrigth_y, topleft_y,-100,100);
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
}


/*--------------------------------------------------------------------*/
/* This method handle drawing through Open GL. The idea is to render  */
/* the screen into an OpenGL textured Quad, that match the window     */
/* size. The texture (screen output) needs to be sent to the GPU ram  */
/* each frame. The SMS screen is usually 256x192 but square textures  */
/* are expected. This difference is compensated by the 0.75 value in  */
/* texture coordinates.						      */
/*--------------------------------------------------------------------*/
void OsmoseCore::drawGLFrame()
{
    prepare2DViewport(0, 0, sdl_window_width, sdl_window_height);
    glLoadIdentity();
    glBindTexture(GL_TEXTURE_2D, textureName[0]);

    glTexImage2D (GL_TEXTURE_2D, 0, GL_RGB, 256, 256, 0, GL_RGB, GL_UNSIGNED_SHORT_5_6_5, buffer->pixels);
    glBegin(GL_QUADS);

    glTexCoord2f(0,0);
    glVertex2i(0, 0);

    glTexCoord2f(1,0);
    glVertex2i(sdl_window_width, 0);

    glTexCoord2f(1, 0.75);
    glVertex2i(sdl_window_width, sdl_window_height);

    glTexCoord2f(0, 0.75);
    glVertex2i(0, sdl_window_height);
    glEnd();

    SDL_GL_SwapBuffers();
}


/*--------------------------------------------------------------------*/
/* This method is called when the window is resized.                  */
/* This only occurs if Osmose runs in windowed mode :)		      */
/*--------------------------------------------------------------------*/
void OsmoseCore::Reshape(int w, int h)
{
	sdl_window_width = w;
	sdl_window_height = h;
	glDeleteTextures(1,textureName);
    SDL_ShowCursor(SDL_DISABLE);
    if ( (screen=SDL_SetVideoMode(sdl_window_width, sdl_window_height,16, SDL_OPENGL  | SDL_GL_DOUBLEBUFFER | SDL_SWSURFACE | SDL_RESIZABLE)) == NULL ) {
            cerr << "Couldn't set video mode: %s" << endl << SDL_GetError() << endl;;
            exit(2);
    }
    initOpenGL(w, h);
}
