/*************************************************************************
vStrip by [maven] (maven@maven.de)
parse_ifo.c: Parse the Cell-Position from the ProgramChain (PGC)
						 in an IFO-file. (tabsize 2)

based on code originally written by Thomas Mirlacher (dent@cosy.sbg.ac.at)
for the "free InFormatiOn project" (http://www.linuxvideo.org)

Many Thanks!
*************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifndef __UNIX__
#include <io.h>
#else
#include <unistd.h>
#endif
#include <fcntl.h>
#include "s_types.h"
#include "parse_ifo.h"
#include "file_io.h"

#define get2bytes(b) be2me_16(*((word *)(b)))
#define get4bytes(b) be2me_32(*((dword *)(b)))

static struct
{
	char descr[3];
	char lang_long[20];
} iifo_lang_tbl[] = {
/* The ISO 639 language codes.
	*/
	{"  ", "Not Specified"},
	{"aa", "Afar"},
	{"ab", "Abkhazian"},
	{"af", "Afrikaans"},
	{"am", "Amharic"},
	{"ar", "Arabic"},
	{"as", "Assamese"},
	{"ay", "Aymara"},
	{"az", "Azerbaijani"},
	{"ba", "Bashkir"},
	{"be", "Byelorussian"},
	{"bg", "Bulgarian"},
	{"bh", "Bihari"},
	{"bi", "Bislama"},
	{"bn", "Bengali; Bangla"},
	{"bo", "Tibetan"},
	{"br", "Breton"},
	{"ca", "Catalan"},
	{"co", "Corsican"},
	{"cs", "Czech(Ceske)"},
	{"cy", "Welsh"},
	{"da", "Dansk"},
	{"de", "Deutsch"},
	{"dz", "Bhutani"},
	{"el", "Greek"},
	{"en", "English"},
	{"eo", "Esperanto"},
	{"es", "Espanol"},
	{"et", "Estonian"},
	{"eu", "Basque"},
	{"fa", "Persian"},
	{"fi", "Suomi"},
	{"fj", "Fiji"},
	{"fo", "Faroese"},
	{"fr", "Francais"},
	{"fy", "Frisian"},
	{"ga", "Irish"},
	{"gd", "Scots Gaelic"},
	{"gl", "Galician"},
	{"gn", "Guarani"},
	{"gu", "Gujarati"},
	{"ha", "Hausa"},
	{"he", "Hebrew"},				// formerly iw
	{"hi", "Hindi"},
	{"hr", "Hrvatski"},				// Croatian
	{"hu", "Magyar"},
	{"hy", "Armenian"},
	{"ia", "Interlingua"},
	{"id", "Indonesian"},				// formerly in
	{"ie", "Interlingue"},
	{"ik", "Inupiak"},
	{"in", "Indonesian"},				// replaced by id
	{"is", "Islenska"},
	{"it", "Italiano"},
	{"iu", "Inuktitut"},
	{"iw", "Hebrew"},				// replaced by he
	{"ja", "Japanese"},
	{"ji", "Yiddish"},				// replaced by yi
	{"jw", "Javanese"},
	{"ka", "Georgian"},
	{"kk", "Kazakh"},
	{"kl", "Greenlandic"},
	{"km", "Cambodian"},
	{"kn", "Kannada"},
	{"ko", "Korean"},
	{"ks", "Kashmiri"},
	{"ku", "Kurdish"},
	{"ky", "Kirghiz"},
	{"la", "Latin"},
	{"ln", "Lingala"},
	{"lo", "Laothian"},
	{"lt", "Lithuanian"},
	{"lv", "Latvian, Lettish"},
	{"mg", "Malagasy"},
	{"mi", "Maori"},
	{"mk", "Macedonian"},
	{"ml", "Malayalam"},
	{"mn", "Mongolian"},
	{"mo", "Moldavian"},
	{"mr", "Marathi"},
	{"ms", "Malay"},
	{"mt", "Maltese"},
	{"my", "Burmese"},
	{"na", "Nauru"},
	{"ne", "Nepali"},
	{"nl", "Nederlands"},
	{"no", "Norsk"},
	{"oc", "Occitan"},
	{"om", "(Afan) Oromo"},
	{"or", "Oriya"},
	{"pa", "Punjabi"},
	{"pl", "Polish"},
	{"ps", "Pashto, Pushto"},
	{"pt", "Portugues"},
	{"qu", "Quechua"},
	{"rm", "Rhaeto-Romance"},
	{"rn", "Kirundi"},
	{"ro", "Romanian"},
	{"ru", "Russian"},
	{"rw", "Kinyarwanda"},
	{"sa", "Sanskrit"},
	{"sd", "Sindhi"},
	{"sg", "Sangho"},
	{"sh", "Serbo-Croatian"},
	{"si", "Sinhalese"},
	{"sk", "Slovak"},
	{"sl", "Slovenian"},
	{"sm", "Samoan"},
	{"sn", "Shona"},
	{"so", "Somali"},
	{"sq", "Albanian"},
	{"sr", "Serbian"},
	{"ss", "Siswati"},
	{"st", "Sesotho"},
	{"su", "Sundanese"},
	{"sv", "Svenska"},
	{"sw", "Swahili"},
	{"ta", "Tamil"},
	{"te", "Telugu"},
	{"tg", "Tajik"},
	{"th", "Thai"},
	{"ti", "Tigrinya"},
	{"tk", "Turkmen"},
	{"tl", "Tagalog"},
	{"tn", "Setswana"},
	{"to", "Tonga"},
	{"tr", "Turkish"},
	{"ts", "Tsonga"},
	{"tt", "Tatar"},
	{"tw", "Twi"},
	{"ug", "Uighur"},
	{"uk", "Ukrainian"},
	{"ur", "Urdu"},
	{"uz", "Uzbek"},
	{"vi", "Vietnamese"},
	{"vo", "Volapuk"},
	{"wo", "Wolof"},
	{"xh", "Xhosa"},
	{"yi", "Yiddish"},				// formerly ji
	{"yo", "Yoruba"},
	{"za", "Zhuang"},
	{"zh", "Chinese"},
	{"zu", "Zulu"},
	{"\0", "???"}
};

static char *iifoDecodeLang(uint16_t descr)
{
	int i;
	
	for (i = 0; iifo_lang_tbl[i].lang_long[0]; i++)
		if (!memcmp(&descr, &iifo_lang_tbl[i].descr, 2))
			return iifo_lang_tbl[i].lang_long;
		return iifo_lang_tbl[i].lang_long;
}

static char *iifoDecodeAudioMode(uint16_t descr)
{
	static char audio_coding_mode[][9] = {
		"AC3",
			"???",
			"MPEG1",
			"MPEG2ext",
			"LPCM",
			"???",
			"DTS"
	};
	
	return audio_coding_mode[descr & 0x07];
}

static char *iifoDecodeAudioStreamID(uint16_t descr, const int audio_idx)
{
	static dword stream_ofs[] = {0xbd80, 0x0000, 0xc000, 0xc800, 0xbda0, 0xbd88, 0xbd88};
	static char stream_id[10];
	
	if ((stream_ofs[descr & 7] & 0xff) != 0)
		sprintf(stream_id, "0x%02X 0x%02X", stream_ofs[descr & 7] >> 8, audio_idx + (stream_ofs[descr & 7] & 0xff));
	else
		sprintf(stream_id, "0x%02X", audio_idx + (stream_ofs[descr & 7] >> 8));
	return stream_id;
}

const pgc_program_map_t *iifoGetProgramMap(const pgc_t *pgc)
{
	const uint8_t *ptr = (uint8_t *)pgc;
	
	return (pgc_program_map_t *)(ptr + be2me_16(pgc->pgc_program_map_offset));
}

const ifo_pgci_caddr_t *iifoGetCellPlayInfo(const pgc_t *pgc)
{
	const uint8_t *ptr = (uint8_t *)pgc;
	
	return (ifo_pgci_caddr_t *)(ptr + be2me_16(pgc->cell_playback_tbl_offset));
}

const ifo_pgc_cpos_t *iifoGetCellPos(const pgc_t *pgc)
{
	const uint8_t *ptr = (uint8_t *)pgc;
	
	return (ifo_pgc_cpos_t *)(ptr + be2me_16(pgc->cell_position_tbl_offset));
}

uint8_t iifoGetNumCells(const pgc_t *pgc)
{
	return pgc->nr_of_cells;
}

uint8_t iifoGetNumPrograms(const pgc_t *pgc)
{
	return pgc->nr_of_programs;
}

const pgc_t *iifoGetPGCI(const ifo_t *ifo, const int title)
{
	pgci_sub_t *pgci_sub;
	ifo_hdr_t *hdr = NULL;
	uint8_t *ptr = NULL;
	
	hdr = (ifo_hdr_t *)ifo->title_pgci;
	
	if (hdr && title < be2me_16(hdr->num))
	{
		ptr = (uint8_t *)hdr;
		ptr += IFO_HDR_LEN;
		
		pgci_sub = (pgci_sub_t *)ptr + title;
		
		ptr = (uint8_t *)hdr + be2me_32(pgci_sub->start);
	}
	
	return (const pgc_t *)ptr;
}

#define OFF_TITLE_PGCI get4bytes (ifo->mat + 0xCC)

bool iifoIsVTS(ifo_t *ifo)
{
	return strncmp (ifo->mat, "DVDVIDEO-VTS", 12) == 0;
}

int iifoReadLB(tp_fio_file fp, dword pos, dword count, byte *data)
{
	if ((pos = fio_direct_lseek(fp, pos, SEEK_SET)) == -1)
		return -1;
	return fio_direct_read(fp, data, count);
}

bool iifoReadTBL(tp_fio_file fp, dword offset, byte **out_ptr)
{
	byte *data;
	ifo_hdr_t *hdr;
	dword len = 0;
	
	if (!offset)
		return FALSE;
	
	data =  (byte *)malloc(fio_SECTOR_SIZE);
	if (!data)
		return FALSE;
	
	if (iifoReadLB (fp, offset * fio_SECTOR_SIZE, fio_SECTOR_SIZE, data) <= 0)
		return FALSE;
	
	hdr = (ifo_hdr_t *)data;
	len = be2me_32(hdr->len) + 1;
	
	if (len > fio_SECTOR_SIZE)
	{
		if (!(data = (byte *)realloc((void *)data, len)))
			return FALSE;
		iifoReadLB (fp, offset * fio_SECTOR_SIZE, len, data);
	}
	*out_ptr = data;
	return TRUE;
}

// interface

VSTRIP_DLL_API ifo_t* VSTRIP_DLL_CC ifoOpen(const char *name, const dword fio_flags)
{
	ifo_t *ret_ifo = NULL;
	tp_fio_file fd;
	
	fd = fio_open(name, fio_flags, 0);
	if (fd)
	{
		ifo_t *ifo = calloc(sizeof *ifo, 1);
		
		ifo->mat = calloc(fio_SECTOR_SIZE, 1);
		if (iifoReadLB(fd, 0, fio_SECTOR_SIZE, ifo->mat) < 0)
		{
			free(ifo->mat);
			free(ifo);
		}
		else 
		{ // looking ok so far
			if (iifoIsVTS(ifo) && iifoReadTBL(fd, OFF_TITLE_PGCI, &ifo->title_pgci))
				ret_ifo = ifo;
			else
			{
				if (ifo->title_pgci)
					free(ifo->title_pgci);
				free (ifo->mat);
				free (ifo);
			}
		}
		fio_close(fd);
	}
	return ret_ifo;
}

VSTRIP_DLL_API bool VSTRIP_DLL_CC ifoClose (ifo_t *ifo)
{
	if (ifo)
	{
		if (ifo->mat)
			free (ifo->mat);
		if (ifo->title_pgci)
			free (ifo->title_pgci);
		free (ifo);
	}
	return TRUE;
}

VSTRIP_DLL_API int VSTRIP_DLL_CC ifoGetNumPGCI(const ifo_t *ifo)
{
	int num = 0;
	
	if (ifo && ifo->title_pgci)
		num = be2me_16(((ifo_hdr_t *)ifo->title_pgci)->num);
	return num;
}

// returns the number of cells and the length in time_out
VSTRIP_DLL_API int VSTRIP_DLL_CC ifoGetPGCIInfo(const ifo_t *ifo, const dword title, byte time_out[4])
{
	int num_cells = 0;
	const pgc_t *pgc;
	
	if (ifo && (pgc = iifoGetPGCI(ifo, title)))
	{
		num_cells = pgc->nr_of_cells;
		if (time_out)
		{
			int i;
			
			time_out[0] = pgc->playback_time.hour;
			time_out[1] = pgc->playback_time.minute;
			time_out[2] = pgc->playback_time.second;
			time_out[3] = pgc->playback_time.frame_u & DVD_TIME_AND;
			for (i = 0; i < 4; i++)
				time_out[i] = ((time_out[i] & 0xf0) >> 4) * 10 + (time_out[i] & 0x0f); // convert bcd to int
		}
	}
	return num_cells;
}

VSTRIP_DLL_API bool VSTRIP_DLL_CC ifoGetPGCICells(const ifo_t *ifo, const dword title, tp_vs_vobcellid cells)
{
	bool success = FALSE;
	const pgc_t *pgc;
	const ifo_pgc_cpos_t *cpos;
	const ifo_pgci_caddr_t *cpi;
	
	if (ifo && cells && (pgc = iifoGetPGCI(ifo, title)) && (cpos = iifoGetCellPos(pgc)) && (cpi = iifoGetCellPlayInfo(pgc)))
	{
		dword i, j, k = 0, anum = 0;
		const pgc_program_map_t *ctm;
		
		success = TRUE;
		for (i = 0; i < pgc->nr_of_cells; i++)
		{
			cells[i].start_lba = be2me_32(cpi[i].vobu_start);
			cells[i].end_lba = be2me_32(cpi[i].vobu_last_end);
			cells[i].vob_id = be2me_16(cpos[i].vob_id);
			cells[i].cell_id = cpos[i].cell_id;
			
			cells[i].time[0] = cpi[i].playback_time.hour;
			cells[i].time[1] = cpi[i].playback_time.minute;
			cells[i].time[2] = cpi[i].playback_time.second;
			cells[i].time[3] = cpi[i].playback_time.frame_u & DVD_TIME_AND;
			
			for (j = 0; j < 4; j++)
				cells[i].time[j] = ((cells[i].time[j] & 0xf0) >> 4) * 10 + (cells[i].time[j] & 0x0f); // convert bcd to int
			cells[i].time[3] |= (cpi[i].playback_time.frame_u & (~DVD_TIME_AND)); // set the flags again
			
			cells[i].chapter = 0;
			
			switch (cpi[i].chain_info & 0xc0)
			{
			case 0x00: // 00
				cells[i].angle = 0x11; // 1 of 1
				k = i;
				break;
			case 0x40: // 01
				cells[i].angle = 0xf1; // 1 of ?
				k = i;
				anum = 1;
				break;
			case 0x80: // 10
				cells[i].angle = 0xf0 | (++anum & 0xf); // anum of ?
				break;
			case 0xc0: // 11
				cells[i].angle = 0xf0 | (++anum & 0xf); // anum of ?
				anum = (anum << 4) & 0xf0;
				for (j = k; j <= i; j++) // now set all the ones we've found
					if ((cells[j].angle & 0xf0) == 0xf0)
						cells[j].angle = (cells[j].angle & 0x0f) | anum;
					break;
			}
		}
		
		ctm = iifoGetProgramMap(pgc);
		if (ctm)
		{ // now fill in chapter info
			for (i = 1; i < pgc->nr_of_programs; i++)
			{
				k = min(pgc->nr_of_cells, ctm[i] - 1);
				for (j = ctm[i - 1] - 1; j < k; j++)
					cells[j].chapter = i - 1;
			}
			for (j = ctm[pgc->nr_of_programs - 1] - 1; j < pgc->nr_of_cells; j++)
				cells[j].chapter = pgc->nr_of_programs - 1; // the last one
		}
		
	}
	return success;
}

VSTRIP_DLL_API int VSTRIP_DLL_CC ifoGetNumAudio(const ifo_t *ifo)
{
	int num = 0;
	
	if (ifo && ifo->mat)
	{
		audio_hdr_t *hdr = (audio_hdr_t *)(ifo->mat + IFO_OFFSET_AUDIO);
		bool used[IFO_NUM_AUDIO];
		int num_streams, i, j;
		
		num_streams = be2me_16(hdr->num);
		memset(used, 0, sizeof used);
		j = ifoGetNumPGCI(ifo);
		for (i = 0; i < j; i++)
		{
			const pgc_t *pgc = iifoGetPGCI(ifo, i);

			if (pgc)
			{
				int k;

				for (k = 0; k < num_streams; k++)
					if (pgc->audio_status[k].available)
						used[pgc->audio_status[k].stream_id] = TRUE;
			}
		}
		for (i = 0; i < IFO_NUM_AUDIO; i++)
			num += used[i];
	}
	return num;
}

VSTRIP_DLL_API const char* VSTRIP_DLL_CC ifoGetAudioDesc(const ifo_t *ifo, const int audio_idx)
{
	static char out[1024];
	
	if (ifo && ifo->mat)
	{
		audio_hdr_t *hdr = (audio_hdr_t *)(ifo->mat + IFO_OFFSET_AUDIO);
		ifo_audio_t *audio = NULL;
		byte used[IFO_NUM_AUDIO];
		int num_streams, i, j;
		int stream_idx, src_stream;
		
		num_streams = be2me_16(hdr->num);
		memset(used, 0, sizeof used);
		j = ifoGetNumPGCI(ifo);
		for (i = 0; i < j; i++)
		{
			const pgc_t *pgc = iifoGetPGCI(ifo, i);

			if (pgc)
			{
				int k;

				for (k = 0; k < num_streams; k++)
					if (pgc->audio_status[k].available)
						used[pgc->audio_status[k].stream_id] = k + 1;
			}
		}
		j = -1;
		for (i = 0; i < IFO_NUM_AUDIO; i++)
		{
			if (used[i])
			{
				j++;
				if (j == audio_idx)
				{
					stream_idx = i;
					src_stream = used[i] - 1;
					audio = &hdr->audio[src_stream];
					break;
				}
			}
		}

		if (audio)
		{
			char *end;
			
			sprintf(out, "%s (%s %dch, %s) [", iifoDecodeLang(audio->lang_code), iifoDecodeAudioMode(audio->coding_mode), audio->num_channels + 1, iifoDecodeAudioStreamID(audio->coding_mode, stream_idx));
			j = ifoGetNumPGCI(ifo);
			for (i = 0; i < j; i++)
			{
				const pgc_t *pgc = iifoGetPGCI(ifo, i);
				
				if (pgc && pgc->audio_status[src_stream].available)
				{
					char buf[5];
					
					sprintf(buf, "%i,", i);
					strcat(out, buf);
				}
			}
			end = strrchr(out, ',');
			if (end)
				*end = 0;
			strcat(out, "]");
			
			return out;
		}
	}
	return NULL;
}

VSTRIP_DLL_API int VSTRIP_DLL_CC ifoGetNumSubPic(const ifo_t *ifo)
{
	int num = 0;
	
	if (ifo && ifo->mat)
	{
		audio_hdr_t *hdr = (audio_hdr_t *)(ifo->mat + IFO_OFFSET_SUBPIC);
		ifo_video_t *vid = (ifo_video_t *)(ifo->mat + IFO_OFFSET_VIDEO);
		bool used[IFO_NUM_SPU];
		int num_streams, i, j;
		
		num_streams = be2me_16(hdr->num);
		memset(used, 0, sizeof used);
		j = ifoGetNumPGCI(ifo);
		for (i = 0; i < j; i++)
		{
			const pgc_t *pgc = iifoGetPGCI(ifo, i);

			if (pgc)
			{
				int k;

				for (k = 0; k < num_streams; k++)
					if (pgc->subp_status[k].available)
					{
						if (!vid->no_auto_letterbox)
						{
							used[pgc->subp_status[k].stream_id_letterbox] = TRUE;
							used[pgc->subp_status[k].stream_id_wide] = TRUE;
							if (!vid->no_auto_pan_scan)
								used[pgc->subp_status[k].stream_id_pan_scan] = TRUE;
						}
						else
							used[pgc->subp_status[k].stream_id_4_3] = TRUE;
					}
			}
		}
		for (i = 0; i < IFO_NUM_SPU; i++)
			num += used[i];
	}
	return num;
}

VSTRIP_DLL_API const char* VSTRIP_DLL_CC ifoGetSubPicDesc(const ifo_t *ifo, const int subp_idx)
{
	static char out[1024];
	
	if (ifo && ifo->mat)
	{
		spu_hdr_t *hdr = (spu_hdr_t *)(ifo->mat + IFO_OFFSET_SUBPIC);
		ifo_video_t *vid = (ifo_video_t *)(ifo->mat + IFO_OFFSET_VIDEO);
		ifo_spu_t *sub = NULL;
		dword used[IFO_NUM_SPU];
		int num_streams, i, j;
		int stream_idx, src_stream;
		
		num_streams = be2me_16(hdr->num);
		memset(used, 0, sizeof used);
		j = ifoGetNumPGCI(ifo);
		for (i = 0; i < j; i++)
		{
			const pgc_t *pgc = iifoGetPGCI(ifo, i);

			if (pgc)
			{
				int k;

				for (k = 0; k < num_streams; k++)
					if (pgc->subp_status[k].available)
					{ // set idx & flags
						if (!vid->no_auto_letterbox)
						{ // only look at these if video _is_ letterboxed
							used[pgc->subp_status[k].stream_id_letterbox] = k + 1 + 0x200;
							used[pgc->subp_status[k].stream_id_wide] = k + 1 + 0x800;
							if (!vid->no_auto_pan_scan)
								used[pgc->subp_status[k].stream_id_pan_scan] = k + 1 + 0x400;
						}
						else
							used[pgc->subp_status[k].stream_id_4_3] = k + 1 + 0x100;
					}
			}
		}
		j = -1;
		for (i = 0; i < IFO_NUM_SPU; i++)
		{
			if (used[i])
			{
				j++;
				if (j == subp_idx)
				{
					stream_idx = i;
					src_stream = (used[i] & 0xff) - 1;
					sub = &hdr->spu[src_stream];
					break;
				}
			}
		}
		
		if (sub)
		{
			char *end;
			
			sprintf(out, "%s %i (0xBD 0x%02X) [", iifoDecodeLang(sub->lang_code), src_stream, 0x20 + stream_idx);
			j = ifoGetNumPGCI(ifo);
			for (i = 0; i < j; i++)
			{
				const pgc_t *pgc = iifoGetPGCI(ifo, i);
				
				if (pgc)
				{
					int k;

					for (k = 0; k < num_streams; k++)
						if (pgc->subp_status[k].available)
						{
							bool flag = FALSE;

							if (!vid->no_auto_letterbox)
							{
								if (pgc->subp_status[k].stream_id_letterbox == stream_idx || pgc->subp_status[k].stream_id_wide == stream_idx)
									flag = TRUE;
								if (!vid->no_auto_pan_scan && pgc->subp_status[k].stream_id_pan_scan == stream_idx)
									flag = TRUE;
							}
							else if (pgc->subp_status[k].stream_id_4_3 == stream_idx)
								flag = TRUE;

							if (flag)
							{
								char buf[5];
								
								sprintf(buf, "%i,", i);
								strcat(out, buf);
								break;
							}
						}
				}
			}
			end = strrchr(out, ',');
			if (end)
				*end = 0;
			strcat(out, "]");
			if (!vid->no_auto_letterbox && (used[stream_idx] & 0xf00) != 0xe00)
			{ // have different stream_id's for different screen modes
				if ((used[stream_idx] & 0x200) != 0)
					strcat(out, " letbox");
				if ((used[stream_idx] & 0x400) != 0)
					strcat(out, " pan&scan");
				if ((used[stream_idx] & 0x800) != 0)
					strcat(out, " wide");
			}
			
			return out;
		}
	}
	return NULL;
}

VSTRIP_DLL_API const char* VSTRIP_DLL_CC ifoGetVideoDesc(const ifo_t *ifo)
{
	static char out[256];
	
	if (ifo && ifo->mat)
	{
		ifo_video_t *vid = (ifo_video_t *)(ifo->mat + IFO_OFFSET_VIDEO);
		const static char ar[4][5] = {"4:3", "???", "???", "16:9"};
		const static char vs[4][5] = {"NTSC", "PAL", "???", "???"};
		const static char cm[2][6] = {"MPEG1", "MPEG2"};
		const static char rs[2][4][8] = {{"720x480", "704x480", "352x480", "352x240"}, {"720x576", "704x576", "352x576", "352x288"}};
		const static char lb[2][7] = {"letbox", ""};
		const static char ps[2][8] = {"pan&scan", ""};

		sprintf(out, "%s %s %s %s %s %s", cm[vid->coding_mode], rs[vid->video_standard][vid->resolution], vs[vid->video_standard], ar[vid->apect_ratio], lb[vid->no_auto_letterbox], ps[vid->no_auto_pan_scan]);
		return out;
	}
	return NULL;
}

bool ifoParse(const char *ifo_name, const int ifo_pgc_idx, const t_ifo_flags flags, dword *num_cells, tp_vs_vobcellid *cell_ids)
{
	if (ifo_name && *ifo_name)
	{
		ifo_t *ifo = NULL;
		t_fio_flags ffl = 0;
		int i, j;
		
		if ((flags & ifo_PRINT_INFO) != 0)
			fprintf(stdout, "Parsing \"%s\"...\n", ifo_name);
		
		if ((flags & ifo_USE_ASPI) != 0)
			ffl |= fio_USE_ASPI;
		if ((flags & ifo_PREFER_ASPI) != 0)
			ffl |= fio_PREFER_ASPI;
		ifo = ifoOpen(ifo_name, ffl);
		if (ifo)
		{
			int num = ifoGetNumPGCI(ifo);
			
			if ((flags & ifo_PRINT_INFO) != 0)
			{
				fprintf(stdout, "Video:\n  %s\n", ifoGetVideoDesc(ifo));
				
				j = ifoGetNumAudio(ifo);
				if (j > 0)
					fputs("Audio:\n", stdout);
				for (i = 0; i < j; i++)
					fprintf(stdout, "  %u. %s\n", i, ifoGetAudioDesc(ifo, i));
				
				j = ifoGetNumSubPic(ifo);
				if (j > 0)
					fputs("SubPicture:\n", stdout);
				for (i = 0; i < j; i++)
					fprintf(stdout, "  %u. %s\n", i, ifoGetSubPicDesc(ifo, i));
				
				fputs("Program Chain(s):\n", stdout);
			}
			
			if (ifo_pgc_idx >= 0)
			{
				if (ifo_pgc_idx < num)
				{
					byte time[4];
					
					*num_cells = ifoGetPGCIInfo(ifo, ifo_pgc_idx, time);
					*cell_ids = calloc(*num_cells, sizeof **cell_ids);
					if ((flags & ifo_PRINT_INFO) != 0)
						fprintf(stdout, "  Using PGC #%u. Length: %02u:%02u:%02u:%02u in %i cell(s).\n", ifo_pgc_idx, time[0], time[1], time[2], time[3], *num_cells);
					ifoGetPGCICells(ifo, ifo_pgc_idx, *cell_ids);
					
					if ((flags & ifo_PRINT_CELL_POS) != 0 && (flags & ifo_PRINT_INFO) != 0)
						for (j = 0; j < (int)*num_cells; j++)
						{
							if ((*cell_ids)[j].angle != 0x11)
								fprintf(stdout, "    %u. VOB-ID: %02u/CELL-ID: %02u Chapter: %u (Angle %01u/%01u).\n", j, (*cell_ids)[j].vob_id, (*cell_ids)[j].cell_id, (*cell_ids)[j].chapter, ((*cell_ids)[j].angle) & 0xf, ((*cell_ids)[j].angle) >> 4);
							else
								fprintf(stdout, "    %u. VOB-ID: %02u/CELL-ID: %02u Chapter: %u.\n", j, (*cell_ids)[j].vob_id, (*cell_ids)[j].cell_id, (*cell_ids)[j].chapter);
						}
				}
				else
				{
					ifoClose(ifo);
					if ((flags & ifo_PRINT_INFO) != 0)
						fprintf(stderr, "* Program-Chain (PGC) Index out of range (only %i PGC(s) available)\n", num);
					return FALSE;
				}
			}
			else
			{ // print all PGCs and select none
				for (i = 0; i < num; i++)
				{
					byte time[4];
					int n_cells;
					
					n_cells = ifoGetPGCIInfo(ifo, i, time);
					
					if ((flags & ifo_PRINT_INFO) != 0)
						fprintf(stdout, "  %u. Length: %02u:%02u:%02u:%02u in %i cell(s).\n", i, time[0], time[1], time[2], time[3], n_cells);
					if (n_cells > 0 && (flags & ifo_PRINT_CELL_POS) != 0 && (flags & ifo_PRINT_INFO) != 0)
					{
						t_vs_vobcellid *cellids;
						
						cellids = calloc(n_cells, sizeof *cellids);
						ifoGetPGCICells(ifo, i, cellids);
						for (j = 0; j < n_cells; j++)
						{
							if (cellids[j].angle != 0x11)
								fprintf(stdout, "    %u. VOB-ID: %02u/CELL-ID: %02u Chapter: %u (Angle %01u/%01u).\n", j, cellids[j].vob_id, cellids[j].cell_id, cellids[j].chapter, cellids[j].angle & 0xf, cellids[j].angle >> 4);
							else
								fprintf(stdout, "    %u. VOB-ID: %02u/CELL-ID: %02u Chapter: %u.\n", j, cellids[j].vob_id, cellids[j].cell_id, cellids[j].chapter);
						}
						free(cellids);
					}
				}
			}
			ifoClose(ifo);
		}
		else
		{
			if ((flags & ifo_PRINT_INFO) != 0)
				fprintf(stderr, "* Unable to parse \"%s\" as IFO-File\n", ifo_name);
			return FALSE;
		}
	}
	return TRUE;
}