/*
 * LibrePCB - Professional EDA for everyone!
 * Copyright (C) 2013 LibrePCB Developers, see AUTHORS.md for contributors.
 * https://librepcb.org/
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef LIBREPCB_CORE_PROJECTMETADATA_H
#define LIBREPCB_CORE_PROJECTMETADATA_H

/*******************************************************************************
 *  Includes
 ******************************************************************************/
#include "../attribute/attribute.h"
#include "../serialization/serializableobject.h"
#include "../types/elementname.h"

#include <QtCore>

/*******************************************************************************
 *  Namespace / Forward Declarations
 ******************************************************************************/
namespace librepcb {

/*******************************************************************************
 *  Class ProjectMetadata
 ******************************************************************************/

/**
 * @brief The ProjectMetadata class
 */
class ProjectMetadata final : public QObject, public SerializableObject {
  Q_OBJECT

public:
  // Constructors / Destructor
  ProjectMetadata() = delete;
  ProjectMetadata(const ProjectMetadata& other) = delete;
  ProjectMetadata(const Uuid& uuid, const ElementName& name,
                  const QString& author, const QString& version,
                  const QDateTime& created, const QDateTime& lastModified);
  ProjectMetadata(const SExpression& node, const Version& fileFormat);
  ~ProjectMetadata() noexcept;

  // Getters
  const Uuid& getUuid() const noexcept { return mUuid; }

  /**
   * @brief Get the name of the project
   *
   * @return The name of the project
   */
  const ElementName& getName() const noexcept { return mName; }

  /**
   * @brief Get the author of the project
   *
   * @return The author of the project
   */
  const QString& getAuthor() const noexcept { return mAuthor; }

  /**
   * @brief Get the version of the project
   *
   * @return The version of the project (arbitrary string)
   */
  const QString& getVersion() const noexcept { return mVersion; }

  /**
   * @brief Get the date and time when the project was created
   *
   * @return The local date and time of creation
   */
  const QDateTime& getCreated() const noexcept { return mCreated; }

  /**
   * @brief Get the date and time when the project was last modified
   *
   * @return The local date and time of last modification
   *
   * @todo    Dynamically determine the datetime of the last modification from
   *          version control system, file attributes or something like that.
   */
  const QDateTime& getLastModified() const noexcept { return mLastModified; }

  /**
   * @brief Get the list of attributes
   *
   * @return All attributes in a specific order
   */
  const AttributeList& getAttributes() const noexcept { return mAttributes; }

  // Setters

  /**
   * @brief Set the name of the project
   *
   * @param newName           The new name
   */
  void setName(const ElementName& newName) noexcept;

  /**
   * @brief Set the author of the project
   *
   * @param newAuthor         The new author
   */
  void setAuthor(const QString& newAuthor) noexcept;

  /**
   * @brief Set the version of the project
   *
   * @param newVersion        The new version (can be an arbitrary string)
   */
  void setVersion(const QString& newVersion) noexcept;

  /**
   * @brief Set all project attributes
   *
   * @param newAttributes     The new list of attributes
   */
  void setAttributes(const AttributeList& newAttributes) noexcept;

  /**
   * @brief Update the last modified datetime
   */
  void updateLastModified() noexcept;

  // General Methods

  /// @copydoc ::librepcb::SerializableObject::serialize()
  void serialize(SExpression& root) const override;

  // Operator Overloadings
  ProjectMetadata& operator=(const ProjectMetadata& rhs) = delete;

signals:
  void attributesChanged();

private:  // Data
  Uuid mUuid;  ///< the UUID of the project
  ElementName mName;  ///< the name of the project
  QString mAuthor;  ///< the author of the project
  QString mVersion;  ///< the version of the project (arbitrary string)
  QDateTime mCreated;  ///< the datetime of the project creation
  QDateTime mLastModified;  ///< the datetime of the last project modification
  AttributeList mAttributes;  ///< all attributes in a specific order
};

/*******************************************************************************
 *  End of File
 ******************************************************************************/

}  // namespace librepcb

#endif
