/*
 * Decompiled with CFR 0.152.
 */
package org.openhab.core.audio.utils;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import javax.sound.sampled.AudioFileFormat;
import javax.sound.sampled.AudioInputStream;
import javax.sound.sampled.AudioSystem;
import org.eclipse.jdt.annotation.NonNullByDefault;
import org.openhab.core.audio.AudioFormat;
import org.openhab.core.audio.AudioStream;
import org.openhab.core.audio.ByteArrayAudioStream;

@NonNullByDefault
public class ToneSynthesizer {
    private final long sampleRate;
    private final int bitDepth;
    private final int bitRate;
    private final int channels;
    private final boolean bigEndian;

    public static Set<AudioFormat> getSupportedFormats() {
        return Set.of(new AudioFormat("WAVE", "PCM_SIGNED", false, null, null, null, null));
    }

    public static List<Tone> parseMelody(String melody) throws ParseException {
        ArrayList<Tone> melodySounds = new ArrayList<Tone>();
        String[] noteTextList = melody.split("\\s");
        int melodyTextIndex = 0;
        int i = 0;
        while (i < noteTextList.length) {
            String noteText = noteTextList[i];
            String[] noteTextParts = noteText.split(":");
            int soundMillis = 200;
            switch (noteTextParts.length) {
                case 2: {
                    try {
                        soundMillis = Integer.parseInt(noteTextParts[1]);
                    }
                    catch (NumberFormatException e) {
                        throw new ParseException("Unable to parse note duration " + noteText, melodyTextIndex);
                    }
                }
                case 1: {
                    String note = noteTextParts[0];
                    int octaves = (int)note.chars().filter(ch -> ch == 39).count();
                    note = note.replace("'", "");
                    Optional<Note> noteObj = Note.fromString(note);
                    if (noteObj.isPresent()) {
                        melodySounds.add(ToneSynthesizer.noteTone(noteObj.get(), soundMillis, octaves));
                        break;
                    }
                    if ("O".equals(note) || "0".equals(note)) {
                        melodySounds.add(ToneSynthesizer.silenceTone(soundMillis));
                        break;
                    }
                }
                default: {
                    throw new ParseException("Unable to parse note " + noteText, melodyTextIndex);
                }
            }
            melodyTextIndex += noteText.length() + 1;
            ++i;
        }
        return melodySounds;
    }

    public static Tone noteTone(Note note, long millis) {
        return ToneSynthesizer.noteTone(note, millis, 0);
    }

    public static Tone noteTone(Note note, long millis, int octaves) {
        return new Tone(note.getFrequency() * (double)(octaves + 1), millis);
    }

    public static Tone silenceTone(long millis) {
        return new Tone(0.0, millis);
    }

    public ToneSynthesizer(AudioFormat audioFormat) {
        assert (audioFormat.getFrequency() != null);
        this.sampleRate = audioFormat.getFrequency();
        assert (audioFormat.getBitDepth() != null);
        this.bitDepth = audioFormat.getBitDepth();
        assert (audioFormat.getBitRate() != null);
        this.bitRate = audioFormat.getBitRate();
        assert (audioFormat.getChannels() != null);
        this.channels = audioFormat.getChannels();
        Boolean bigEndian = audioFormat.isBigEndian();
        assert (bigEndian != null);
        this.bigEndian = bigEndian;
    }

    public AudioStream getStream(List<Tone> tones) throws IOException {
        int byteRate = (int)(this.sampleRate * (long)this.bitDepth * (long)this.channels / 8L);
        byte[] audioBuffer = new byte[]{};
        ArrayList<Tone> fixedTones = new ArrayList<Tone>(tones);
        fixedTones.add(ToneSynthesizer.silenceTone(100L));
        for (Tone sound : fixedTones) {
            double frequency = sound.frequency;
            long millis = sound.millis;
            int samplesPerChannel = (int)Math.ceil((double)this.sampleRate * ((double)millis / 1000.0));
            byte[] audioPart = this.getAudioBytes(frequency, samplesPerChannel);
            audioBuffer = ByteBuffer.allocate(audioBuffer.length + audioPart.length).put(audioBuffer).put(audioPart).array();
        }
        int minByteSize = (int)Math.ceil((double)byteRate * 0.5);
        if (audioBuffer.length < minByteSize) {
            byte[] padBytes = new byte[minByteSize - audioBuffer.length];
            audioBuffer = ByteBuffer.allocate(minByteSize).put(padBytes).put(audioBuffer).array();
        }
        if (!this.bigEndian) {
            return this.getAudioStreamWithRIFFHeader(audioBuffer);
        }
        return this.getAudioStream(audioBuffer);
    }

    private double getSample(double frequency, int sampleNum) {
        return 4095.0 * Math.sin(frequency * (Math.PI * 2) * (double)sampleNum / (double)this.sampleRate);
    }

    private ByteArrayAudioStream getAudioStreamWithRIFFHeader(byte[] audioBytes) throws IOException {
        javax.sound.sampled.AudioFormat jAudioFormat = new javax.sound.sampled.AudioFormat(this.sampleRate, this.bitDepth, this.channels, true, this.bigEndian);
        AudioInputStream audioInputStreamTemp = new AudioInputStream(new ByteArrayInputStream(audioBytes), jAudioFormat, (long)Math.ceil((double)audioBytes.length / (double)jAudioFormat.getFrameSize()));
        ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
        AudioSystem.write(audioInputStreamTemp, AudioFileFormat.Type.WAVE, outputStream);
        return this.getAudioStream(outputStream.toByteArray());
    }

    private ByteArrayAudioStream getAudioStream(byte[] audioBytes) {
        return new ByteArrayAudioStream(audioBytes, new AudioFormat("WAVE", "PCM_SIGNED", this.bigEndian, this.bitDepth, this.bitRate, this.sampleRate, this.channels));
    }

    private byte[] getAudioBytes(double frequency, int samplesPerChannel) {
        ByteBuffer audioBuffer = ByteBuffer.allocate(samplesPerChannel * (this.bitDepth / 8) * this.channels);
        audioBuffer.order(this.bigEndian ? ByteOrder.BIG_ENDIAN : ByteOrder.LITTLE_ENDIAN);
        int i = 0;
        while (i < samplesPerChannel) {
            short sample = (short)this.getSample(frequency, i);
            int c = 0;
            while (c < this.channels) {
                switch (this.bitDepth) {
                    case 8: {
                        audioBuffer.put((byte)(sample & 0xFF));
                        break;
                    }
                    case 16: {
                        audioBuffer.putShort(sample);
                        break;
                    }
                    case 24: {
                        this.putInt24Bits(audioBuffer, sample << 8);
                        break;
                    }
                    case 32: {
                        audioBuffer.putInt(sample << 16);
                    }
                }
                ++c;
            }
            ++i;
        }
        return audioBuffer.array();
    }

    private void putInt24Bits(ByteBuffer buffer, int value) {
        if (this.bigEndian) {
            buffer.put((byte)(value >> 16 & 0xFF));
            buffer.put((byte)(value >> 8 & 0xFF));
            buffer.put((byte)(value & 0xFF));
        } else {
            buffer.put((byte)(value & 0xFF));
            buffer.put((byte)(value >> 8 & 0xFF));
            buffer.put((byte)(value >> 16 & 0xFF));
        }
    }

    public static enum Note {
        B(List.of("B", "Si"), 493.88),
        Bb(List.of("A#", "Bb", "LA#", "SIb"), 466.16),
        A(List.of("A", "LA"), 440.0),
        Ab(List.of("G#", "Ab", "SOL#", "LAb"), 415.3),
        G(List.of("G", "SOL"), 392.0),
        Gb(List.of("F#", "Gb", "FA#", "SOLb"), 369.99),
        F(List.of("F", "FA"), 349.23),
        E(List.of("E", "MI"), 329.63),
        Eb(List.of("D#", "Eb", "RE#", "MIb"), 311.13),
        D(List.of("D", "RE"), 293.66),
        Cb(List.of("C#", "Db", "DO#", "REb"), 277.18),
        C(List.of("C", "DO"), 261.63);

        private final List<String> names;
        private final double frequency;

        private Note(List<String> names, double frequency) {
            this.names = names;
            this.frequency = frequency;
        }

        public double getFrequency() {
            return this.frequency;
        }

        public static Optional<Note> fromString(String note) {
            return Arrays.stream(Note.values()).filter(note1 -> note1.names.stream().filter(note::equalsIgnoreCase).count() == 1L).findAny();
        }
    }

    public static class Tone {
        private final double frequency;
        private final long millis;

        private Tone(double frequency, long millis) {
            this.frequency = frequency;
            this.millis = millis;
        }
    }
}

