/*
 * $Id: Session.java,v 1.10 2007/02/27 17:42:22 rbair Exp $
 *
 * Copyright 2004 Sun Microsystems, Inc., 4150 Network Circle,
 * Santa Clara, California 95054, U.S.A. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package org.jdesktop.http;

import java.io.ByteArrayOutputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.CookieHandler;
import java.net.HttpRetryException;
import java.net.HttpURLConnection;
import java.net.URI;
import java.net.URL;
import java.net.URLConnection;
import java.net.URLEncoder;
import java.net.UnknownHostException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.zip.GZIPInputStream;
import javax.net.ssl.*;
import org.jdesktop.beans.AbstractBean;

/**
 * Represents a user's "session" on the web. Think of it as a "tab" in a
 * tabbed web browser.
 * 
 * It may access multiple web sites during one "session", but remembers the cookies
 * for all of them.
 *
 * @author rbair
 */
public class Session extends AbstractBean {
    /**
     * Specifies a value to use for security, either Low, Medium, or High. This
     * is currently used for determining how to treat SSL connections.
     *
     * @see #setSslSecurityLevel
     */
    public enum SecurityLevel {Low, Medium, High};
    
    private SecurityLevel sslSecurity;
    private SecurityHandler handler;
    
    /** Creates a new Session. Automatically installs the {@link CookieManager}.*/
    public Session() {
        this(true);
    }
    
    /** 
     * Creates a new Session. If <code>installCookieManager</code> is true,
     * then the CookieManager is installed automatically. Otherwise, the
     * <code>CookieManager</code> will not be installed, allowing you to use some
     * other cookie manager.
     * 
     * @param installCookieManager
     */
    public Session(boolean installCookieManager) {
        setSslSecurityLevel(SecurityLevel.Medium);
        //register a default security handler
        setMediumSecurityHandler(new DefaultSecurityHandler());
        if (installCookieManager) {
            CookieManager.install();
        }
    }
    
    /**
     * Sets the security level to use with SSL.
     *
     * @param level one of High, Medium, or Low. Low will not prompt or fail for self signed certs.
     *              Medium will prompt for self signed certs. High will fall back on the default
     *              behavior, and simply fail for self signed certs.
     */
    public void setSslSecurityLevel(SecurityLevel level) {
        SecurityLevel old = getSslSecurityLevel();
        sslSecurity = level;
        firePropertyChange("sslSecurityLevel", old, getSslSecurityLevel());
    }
    
    /**
     * Gets the SecurityLevel used for SSL connections.
     *
     * @return the SecurityLevel
     * @see #setSslSecurityLevel
     */
    public SecurityLevel getSslSecurityLevel() {
        return sslSecurity;
    }
    
    void setMediumSecurityHandler(SecurityHandler h) {
        SecurityHandler old = getMediumSecurityHandler();
        this.handler = h;
        firePropertyChange("mediumSecurityHandler", old, getMediumSecurityHandler());
    }
    
    SecurityHandler getMediumSecurityHandler() {
        return handler;
    }
    
    private SSLSocketFactory createSocketFactory(String host) {
        try {
            TrustManager tm = null;
            Session.SecurityLevel level = getSslSecurityLevel();
            if (level == Session.SecurityLevel.Low) {
                tm = new LowSecurityX509TrustManager(null);
            } else if (level == Session.SecurityLevel.Medium) {
                tm = new MediumSecurityX509TrustManager(host, getMediumSecurityHandler(), null);
            } else {
                tm = new HighSecurityX509TrustManager(null);
            }
            SSLContext context = SSLContext.getInstance("SSL");
            context.init(
              null, 
              new TrustManager[] {tm}, 
              null);
            return context.getSocketFactory();
        } catch (Exception e) {
            throw new AssertionError(e);
        }
    }
    
    /**
     * Constructs and executes a {@link Request} using the Method.GET method.
     * This method blocks.
     *
     * @param url The url to hit. This url may contain a query string (ie: params).
     *            The url cannot be null.
     * @return the {@link Response} to the {@link Request}.
     * @throws Exception if an error occurs while creating or executing the
     *         <code>Request</code> on the client machine. That is, if normal
     *         http errors occur, they will not throw an exception (such as BAD_GATEWAY, etc).
     */
    public Response get(String url) throws Exception {
        return execute(Method.GET, url);
    }
    
    /**
     * Constructs and executes a {@link Request} using the Method.GET method.
     * This method blocks.
     *
     * @param url The url to hit. This url may contain a query string (ie: params).
     *            The url cannot be null.
     * @param params The params to include in the request. This may be null.
     * @return the {@link Response} to the {@link Request}.
     * @throws Exception if an error occurs while creating or executing the
     *         <code>Request</code> on the client machine. That is, if normal
     *         http errors occur, they will not throw an exception (such as BAD_GATEWAY, etc).
     */
    public Response get(String url, Parameter... params) throws Exception {
        return execute(Method.GET, url, params);
    }
    
    /**
     * Constructs and executes a {@link Request} using the Method.POST method.
     * This method blocks.
     *
     * @param url The url to hit. This url may contain a query string (ie: params).
     *            The url cannot be null.
     * @return the {@link Response} to the {@link Request}.
     * @throws Exception if an error occurs while creating or executing the
     *         <code>Request</code> on the client machine. That is, if normal
     *         http errors occur, they will not throw an exception (such as BAD_GATEWAY, etc).
     */
    public Response post(String url) throws Exception {
        return execute(Method.POST, url);
    }
    
    /**
     * Constructs and executes a {@link Request} using the Method.POST method.
     * This method blocks.
     *
     * @param url The url to hit. This url may contain a query string (ie: params).
     *            The url cannot be null.
     * @param params The params to include in the request. This may be null.
     * @return the {@link Response} to the {@link Request}.
     * @throws Exception if an error occurs while creating or executing the
     *         <code>Request</code> on the client machine. That is, if normal
     *         http errors occur, they will not throw an exception (such as BAD_GATEWAY, etc).
     */
    public Response post(String url, Parameter... params) throws Exception {
        return execute(Method.POST, url, params);
    }
    
    /**
     * Constructs and executes a {@link Request} using the Method.PUT method.
     * This method blocks.
     *
     * @param url The url to hit. This url may contain a query string (ie: params).
     *            The url cannot be null.
     * @return the {@link Response} to the {@link Request}.
     * @throws Exception if an error occurs while creating or executing the
     *         <code>Request</code> on the client machine. That is, if normal
     *         http errors occur, they will not throw an exception (such as BAD_GATEWAY, etc).
     */
    public Response put(String url) throws Exception {
        return execute(Method.PUT, url);
    }
    
    /**
     * Constructs and executes a {@link Request} using the Method.PUT method.
     * This method blocks.
     *
     * @param url The url to hit. This url may contain a query string (ie: params).
     *            The url cannot be null.
     * @param params The params to include in the request. This may be null.
     * @return the {@link Response} to the {@link Request}.
     * @throws Exception if an error occurs while creating or executing the
     *         <code>Request</code> on the client machine. That is, if normal
     *         http errors occur, they will not throw an exception (such as BAD_GATEWAY, etc).
     */
    public Response put(String url, Parameter... params) throws Exception {
        return execute(Method.PUT, url, params);
    }
    
    /**
     * Constructs and executes a {@link Request}, and returns the {@link Response}.
     * This method blocks. The given <code>method</code>, <code>url</code>
     * will be used to construct the <code>Request</code>.
     * All other <code>Request</code> properties are left in their default state.
     *
     * @param method The HTTP {@link Method} to use. This must not be null.
     * @param url The url to hit. This url may contain a query string (ie: params).
     *            The url cannot be null.
     * @return the {@link Response} to the {@link Request}.
     * @throws Exception if an error occurs while creating or executing the
     *         <code>Request</code> on the client machine. That is, if normal
     *         http errors occur, they will not throw an exception (such as BAD_GATEWAY, etc).
     */
    public Response execute(Method method, String url) throws Exception {
        return execute(method, url, new Parameter[0]);
    }
    
    /**
     * Constructs and executes a {@link Request}, and returns the {@link Response}.
     * This method blocks. The given <code>method</code>, <code>url</code>, and
     * <code>params</code> will be used to construct the <code>Request</code>.
     * All other <code>Request</code> properties are left in their default state.
     *
     * @param method The HTTP {@link Method} to use. This must not be null.
     * @param url The url to hit. This url may contain a query string (ie: params).
     *            The url cannot be null.
     * @param params The params to include in the request. This may be null.
     * @return the {@link Response} to the {@link Request}.
     * @throws Exception if an error occurs while creating or executing the
     *         <code>Request</code> on the client machine. That is, if normal
     *         http errors occur, they will not throw an exception (such as BAD_GATEWAY, etc).
     */
    public Response execute(Method method, String url, Parameter... params) throws Exception {
        if (method == null) {
            throw new NullPointerException("method cannot be null");
        }
        if (url == null) {
            throw new NullPointerException("url cannot be null");
        }
        
        //create and handle the request
        Request req = new Request();
        req.setParameters(params);
        req.setMethod(method);
        req.setUrl(url); //make sure the URL is set after the params, or else
        //if the url had any params, they will be hosed!
        return execute(req);
    }
    
    /**
     * Executes the given {@link Request}, and returns a {@link Response}.
     * This method blocks.
     *
     * @return the {@link Response} to the {@link Request}.
     * @throws Exception if an error occurs while creating or executing the
     *         <code>Request</code> on the client machine. That is, if normal
     *         http errors occur, they will not throw an exception (such as BAD_GATEWAY, etc).
     */
    public Response execute(Request req) throws Exception {
        try {
            // 0. Create the URL
                // a. If NOT a POST, then specify the parameters in the URL
            StringBuffer surl = new StringBuffer(req.getUrl());
            if (surl.length() == 0) {
                throw new IllegalStateException("Cannot excecute a request that has no URL specified");
            }
            
            if (req.getMethod() != Method.POST) {
                boolean first = true;
                for (Parameter p : req.getParameters()) {
                    if (first) {
                        surl.append("?");
                        first = false;
                    } else {
                        surl.append("&");
                    }
                    String name = URLEncoder.encode(p.getName(), "UTF-8");
                    String value = URLEncoder.encode(p.getValue(), "UTF-8");
                    surl.append(name + "=" + value);
                }
            }
            
            // 1. Create the HttpURLConnection
            URL url = new URL(surl.toString());
            URLConnection conn = url.openConnection();
            if (!(conn instanceof HttpURLConnection)) {
                throw new AssertionError("Must be an HTTP or HTTPS based URL");
            }
            HttpURLConnection http = (HttpURLConnection)conn;
            
            // 2. Configure the connection
                // a. Set the HTTP method
                // b. Set whether to follow redirects
                // c. Set the connection timeout
                // d. Set any request headers
                    // i. If not already set, specify that gzip is an acceptable encoding
            http.setRequestMethod(req.getMethod().name());
            http.setInstanceFollowRedirects(req.getFollowRedirects());
//            http.setChunkedStreamingMode(req.getChunkSize() > 0 ? req.getChunkSize() : -1);
//            http.setConnectTimeout(req.getConnectionTimeout());
//            http.setFixedLengthStreamingMode(contentLength);
            for (Header h : req.getHeaders()) {
                http.setRequestProperty(h.getName(), h.getValue());
            }
            if (req.getHeader("Accept-Encoding") == null) {
                //TODO should check to see what Accept-Encoding on the HttpURLConnection
                //already is, to avoid clobbering stuff...
                String acceptEncoding = http.getRequestProperty("Accept-Encoding");
                if (acceptEncoding != null) {
                    System.out.println(acceptEncoding);
                }
//                http.setRequestProperty("Accept-Encoding", "gzip");
            }
            
            // 3. If I supported a cache, this is where I'd configure it!
            
            // 4. Configure the request parameters
                // a. If NOT a POST, then the params have already been added to the
                //    URL.
                // b. If a POST, the params must be added to the content body. Prepare
                //    for that by creating a byte[] containing the params.
            byte[] postParams = null;
            if (req.getMethod() == Method.POST) {
                StringBuffer b = new StringBuffer();
                for (Parameter p : req.getParameters()) {
                    b.append(p.getName());
                    b.append("=");
                    b.append(p.getValue());
                    b.append("\n");
                }
                postParams = b.toString().getBytes();
            }

            // 5. Set the request body, if any.
                // a. If a POST, send the params first
                // b. If the body is specified, send it next.
            OutputStream out = null;
            InputStream body = req.getBody();
            if (postParams != null || body != null) {
                http.setDoOutput(true);
                out = http.getOutputStream();
            }
            
            if (postParams != null) {
                //spit out the post params
                out.write(postParams);
                out.flush();
            }
            if (body != null) {
                //spit out the body
                byte[] buffer = new byte[8096];
                int length = -1;
                while ((length = body.read(buffer)) != -1) {
                    out.write(buffer, 0, length);
                }
                body.close();
            }
            if (out != null) {
                //close the output stream
                out.close();
            }
            
            // 6. Get the response
                // a. Retry if necessary
                // b. Read headers
                // c. Read response
                    // i. If FileNotFoundException, then read the error stream, if any
                    // ii. Otherwise, read in the response body. Unzip if required.
                // d. Read status codes
                // e. Construct the response object.
            
            byte[] responseBody = null;
            try {
                if (http instanceof HttpsURLConnection) {
                    ((HttpsURLConnection)http).setSSLSocketFactory(createSocketFactory(url.getHost()));
                }
                
                http.connect(); //I don't think this is strictly necessary, but it makes me feel good
                InputStream responseStream = http.getInputStream();
                //if this is GZIP encoded, then wrap the input stream TODO!!!
//                if (gzipEncoded) {
//                    in = new GZIPInputStream(in);
//                }
                responseBody = readFully(responseStream);
            } catch (FileNotFoundException e) {
                //check for an error stream
                InputStream errorStream = http.getErrorStream();
                responseBody = readFully(errorStream);
            } catch (HttpRetryException e) {
                //TODO
                System.out.println("Got a retry exception");
                e.printStackTrace();
            } catch (UnknownHostException e) {
                http.disconnect();
                return new Response(StatusCode.NOT_FOUND, "Unknown host", null, null, req.getUrl());
            }
            
            Set<Header> headers = new HashSet<Header>();
            for (Map.Entry<String, List<String>> entry : http.getHeaderFields().entrySet()) {
                String headerKey = entry.getKey();
                String headerValue = http.getHeaderField(headerKey);
                if (headerKey == null) {
                    continue;
                }
                List<String> values = entry.getValue();
                Header.Element[] elements = new Header.Element[values.size()];
                for (int j=0; j<elements.length; j++) {
                    elements[j] = new Header.Element(new Parameter(values.get(j), values.get(j)));
                }

                headers.add(new Header(headerKey, headerValue, elements));
            }
            
            Response response = new Response(StatusCode.valueOf(http.getResponseCode()), http.getResponseMessage(),
                    responseBody, headers, req.getUrl());
            
            // 7. Disconnect (as it is unclear how to reuse the HttpURLConnection, for Session anyway)
            http.disconnect();

            // return!
            return response;
        } finally {
            //anything?
        }
    }
    
    private static byte[] readFully(InputStream in) throws IOException {
        ByteArrayOutputStream out = new ByteArrayOutputStream(8096);
        byte[] buffer = new byte[8096];
        int length = -1;

        while ((length = in.read(buffer)) != -1) {
            out.write(buffer, 0, length);
        }
        in.close();
        return out.toByteArray();
    }
}
