/*
 * $Id: Request.java,v 1.7 2007/04/01 00:13:15 rbair Exp $
 *
 * Copyright 2004 Sun Microsystems, Inc., 4150 Network Circle,
 * Santa Clara, California 95054, U.S.A. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package org.jdesktop.http;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.HashSet;
import java.util.Set;
import org.jdesktop.beans.AbstractBean;
import org.jdesktop.xpath.XPathUtils;
import org.w3c.dom.Document;

/**
 * <p>Represents an http request. A <code>Request</code> is constructed and then
 * passed to a {@link Session} for execution. The <code>Session</code> then returns
 * a {@link Response} after execution finishes.</p>
 * 
 * <p>It is not possible to reuse <code>Request</code>s with content bodies because
 * those bodies are specified as <code>InputStream</code>s. This is done for
 * efficient handling of large files that may be used as content bodies.</p>
 * 
 * <p>To help simplify reuse of <code>Request</codes>s, a copy constructor is provided
 * which will copy everything _except_ the content body from the source
 * <code>Request</code></p>.
 * 
 * @author rbair
 */
public class Request extends AbstractBean {
    private Set<Header> headers = new HashSet<Header>();
    private Set<Parameter> params = new HashSet<Parameter>();
    private boolean followRedirects = true;
    private Method method = Method.GET;
    private String url;
    private InputStream requestBody;
    
    /** 
     * <p>Creates a new instance of Request. The following default values are
     * used:
     * <ul>
     *      <li>headers: empty set</li>
     *      <li>parameters: empty set</li>
     *      <li>followRedirects: true</li>
     *      <li>method: GET</li>
     *      <li>url: null</li>
     *      <li>requestBody: null</li>
     * </ul></p>
     */
    public Request() {
    }
    
    /**
     * <p>Creates a new instance of Request, using <code>source</code> as the
     * basis for all of the initial property values (except for requestBody, which
     * is always null). This is a copy constructor.</p>
     * 
     * @param source The source Request to copy
     */
    public Request(Request source) {
        if (source != null) {
            headers.addAll(source.headers);
            params.addAll(source.params);
            followRedirects = source.followRedirects;
            method = source.method;
            url = source.url;
        }
    }
    
    /**
     * Returns the Header with the given name, or null if there is no such header.
     * 
     * @param name the name to look for. This must not be null.
     * @return the Header with the given name.
     */
    public Header getHeader(String name) {
        if (name == null) {
            throw new NullPointerException("name cannot be null");
        }
        
        for (Header h : headers) {
            if (name.equalsIgnoreCase(h.getName())) {
                return h;
            }
        }
        return null;
    }
    
    /**
     * Adds the given header to the set of headers.
     * 
     * @param header the Header to add. This must not be null.
     */
    public void setHeader(Header header) {
        if (header == null) {
            throw new NullPointerException("header cannot be null");
        }
        headers.add(header);
    }
    
    /**
     * Removes the given header from this Request.
     *
     * @param header the Header to remove. If null, nothing happens. If the header
     *        is not specified in this Request, nothing happens.
     */
    public void removeHeader(Header header) {
        if (header != null) {
            headers.remove(header);
        }
    }
    
    /**
     * Removes the given named header from this Request. The header is case-insensitive.
     *
     * @param header the name of the Header to remove. If null, nothing happens. If
     *        the header is not specified in this Request, nothing happens. Matches
     *        in a case-insensitive manner.
     */
    public void removeHeader(String header) {
        if (header != null) {
            for (Header h : headers) {
                if (h.getName().equalsIgnoreCase(header)) {
                    headers.remove(h);
                    break;
                }
            }
        }
    }
    
    /**
     * Gets an array of all the Headers for this Request. This array will never
     * be null. Ordering of items is not guaranteed.
     * 
     * @return the array of Headers for this request
     */
    public Header[] getHeaders() {
        return headers.toArray(new Header[0]);
    }
    
    /**
     * Sets the headers to use with this Request. This replaces whatever headers
     * may have been previously defined. If null, this array is treated as an empty
     * array.
     * 
     * @param headers the Headers to set for this Request. May be null.
     */
    public void setHeaders(Header... headers) {
        this.headers.clear();
        if (headers != null) {
            for (Header h : headers) {
                setHeader(h);
            }
        }
    }
    
    /**
     * Returns the Parameter with the given name, or null if there is no such Parameter.
     * 
     * @param name the name to look for. This must not be null.
     * @return the Parameter with the given name.
     */
    public Parameter getParameter(String name) {
        if (name == null) {
            throw new NullPointerException("name cannot be null");
        }
        
        for (Parameter p : params) {
            if (name.equals(p.getName())) {
                return p;
            }
        }
        return null;
    }
    
    /**
     * Adds the given parameter to the set of parameters.
     * 
     * @param parem the Parameter to add. This must not be null.
     */
    public void setParameter(Parameter param) {
        if (param == null) {
            throw new NullPointerException("param cannot be null");
        }
        params.add(param);
    }
    
    /**
     * Gets an array of all the Parameters for this Request. This array will never
     * be null. Ordering of items is not guaranteed.
     * 
     * @return the array of Parameters for this request
     */
    public Parameter[] getParameters() {
        return params.toArray(new Parameter[0]);
    }
    
    /**
     * Sets the parameters to use with this Request. This replaces whatever parameters
     * may have been previously defined. If null, this array is treated as an empty
     * array.
     * 
     * @param params the Parameters to set for this Request. May be null.
     */
    public void setParameters(Parameter... params) {
        this.params.clear();
        if (params != null) {
            for (Parameter p : params) {
                setParameter(p);
            }
        }
    }
    
    /**
     * Specifies whether to automatically follow redirects. An HTTP response may
     * indicate that the system should be redirected to a new page. In that case,
     * if followRedirects is true, this will happen automatically and the Response
     * will be the new page (as long as the new page does not also cause a
     * redirect). It is possible to encounter infinite redirects.
     * 
     * boolean b whether to automatically follow redirects
     */
    public void setFollowRedirects(boolean b) {
        boolean old = getFollowRedirects();
        this.followRedirects = b;
        firePropertyChange("followRedirects", old, getFollowRedirects());
    }
    
    /**
     * Gets whether to automatically follow redirct requests. @see #setFollowRedirects(boolean).
     * 
     * @return whether to automatically follow redirects.
     */
    public boolean getFollowRedirects() {
        return followRedirects;
    }
    
    /**
     * Sets the http {@link Method} to use for this Request. If null, a GET method
     * will be used.
     * 
     * @param method the {@link Method} to use. If null, <code>Method.GET</code> is
     *               used.
     */
    public void setMethod(Method method) {
        Method old = getMethod();
        this.method = method == null ? Method.GET : method;
        firePropertyChange("method", old, getMethod());
    }
    
    /**
     * Gets the http Method used.
     * 
     * @return the {@link Method} for this Request.
     */
    public Method getMethod() {
        return method;
    }
    
    /**
     * <p>The URL to request content from. This must be an absolute URL. An
     * IllegalArgumentException will be thrown if this url is malformed. This value
     * may be null, but must be specified prior to executing this Request, otherwise
     * an IllegalStateException will occur at execution time.</p>
     * 
     * <p>This URL <em>may</em> contain parameters (ie: in the query string). These
     * parameters will be left in place. Any parameters added via #setParameters(Parameter[])
     * will be appened to this query string if this is not a POST request, otherwise, they
     * will be included in the body of the post.</p>
     * 
     * @param url The url to request content from. May be null
     * @throws IllegalArgumentException if the url is malformed.
     */
    public void setUrl(String url) throws IllegalArgumentException {
        String old = getUrl();
        this.url = url;
        firePropertyChange("url", old, getUrl());
    }
    
    /**
     * Returns the URL to request content from.
     * 
     * @return the url
     */
    public String getUrl() {
        return url;
    }
    
    /**
     * Sets the request body to be the specified String.
     * 
     * @param body the String to use for the body. May be null.
     */
    public void setBody(String body) {
        setBody(body == null ? null : body.getBytes());
    }
    
    /**
     * Sets the request body to be the specified array of bytes.
     * 
     * @param body the byte array to use for the body. May be null.
     */
    public void setBody(byte[] body) {
        if (body == null || body.length == 0) {
            requestBody = null;
        } else {
            setBody(new ByteArrayInputStream(body));
        }
    }
    
    /**
     * Sets the request body to be the specified {@link SimpleDocument}.
     * 
     * @param body the DOM document to use for the body. May be null.
     */
    public void setBody(Document body) {
        setBody(body == null ? null : XPathUtils.toXML(body));
    }
    
    /**
     * Sets the request body to be the specified <code>InputStream</code>.
     * 
     * @param body the InputStream to use for the body. May be null.
     */
    public void setBody(InputStream body) {
        this.requestBody = body;
    }
    
    /**
     * Package private method which returns the request body. This is only called
     * by the Session. This is package private since it is an implementation detail:
     * it supposes that the internal representation of the body is an InputStream.
     */
    InputStream getBody() {
        return requestBody;
    }
}
