/*
 * $Id: AbstractForm.java,v 1.4 2006/12/19 23:48:47 rbair Exp $
 *
 * Copyright 2004 Sun Microsystems, Inc., 4150 Network Circle,
 * Santa Clara, California 95054, U.S.A. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package org.jdesktop.html.form;

import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import org.jdesktop.beans.AbstractBean;
import org.jdesktop.http.Response;
import org.jdesktop.http.Session;
import org.jdesktop.http.Parameter;

/**
 * An abstract implementation of the Form interface.
 * 
 * @author rbair
 */
public abstract class AbstractForm extends AbstractBean implements Form {
    private Set<Input> inputs = new HashSet<Input>();
    private String baseUrl;

    /**
     * Create a new AbstractForm with the given baseUrl and set of Inputs.
     * The baseUrl is necessary for evaluating action urls that are relative.
     * 
     * @param baseUrl the base url. May be null (but then relative action paths
     *        will fail).
     * @param inputs the set of inputs. May be null.
     */
    protected AbstractForm(String baseUrl, Set<Input> inputs) {
        this.baseUrl = baseUrl;
        this.inputs.addAll(inputs);
    }
    
    /**
     * Create a new AbstractForm with the given baseUrl and array of Inputs.
     * The baseUrl is necessary for evaluating action urls that are relative.
     * 
     * @param baseUrl the base url. May be null (but then relative action paths
     *        will fail).
     * @param inputs the array of inputs. May be null.
     */
    protected AbstractForm(String baseUrl, Input... inputs) {
        this.baseUrl = baseUrl;
        if (inputs != null) {
            this.inputs.addAll(Arrays.asList(inputs));
        }
    }
    
    /**
     * @inheritDoc
     */
    public Input[] getInputs() {
        return inputs.toArray(new Input[0]);
    }
    
    /**
     * @inheritDoc
     */
    public Input getInput(String name) {
        for (Input i : inputs) {
            if (name.equals(i.getName())) {
                return i;
            }
        }
        return null;
    }
    
    /**
     * @inheritDoc
     */
    public Parameter[] getInputParameters() {
        return Forms.getParameters(this);
    }
    
    /**
     * @inheritDoc
     */
    public Response submit(Session session) throws Exception {
        return Forms.submit(this, session);
    }
    
    /**
     * @inheritDoc
     */
    public String getBaseUrl() {
        return baseUrl;
    }
}
