/******************************************************************************
 *{@C
 *      Copyright:      2005-2010 Paul Obermeier (obermeier@tcl3d.org)
 *
 *                      See the file "Tcl3D_License.txt" for information on
 *                      usage and redistribution of this file, and for a
 *                      DISCLAIMER OF ALL WARRANTIES.
 *
 *      Module:         Tcl3D -> tcl3dOgl
 *      Filename:       heightmap.i
 *
 *      Author:         Paul Obermeier
 *
 *      Description:    SWIG file for wrapping low-level function
 *                      tcl3dDemoUtilPhoto2Heightmap. Usage of the functionality
 *                      of this function should be done with the high-level
 *                      tcl3dDemoUtilHeightmapFromPhoto function.
 *                      Low-level functions may be subject to change without
 *                      taking care of compatibility issues. 
 *
 *                      This function builds a heightmap out of a photo image.
 *                      The C code is taken from NeHe Lesson45 and slightly
 *                      modified. See nehe.gamedev.net for the original code.
 *
 *****************************************************************************/

%{
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <tcl.h>
#include <tk.h>

static float PtHeight (int nX, int nY, unsigned char *imgData, int w, int h)
{
    int nPos;
    float flR, flG, flB;

    /* Calculate The Position In The Texture, Careful Not To Overflow */
    nPos = ((nX % w) + ((nY % h) * w)) * 3;
    flR = (float) imgData[nPos];
    flG = (float) imgData[nPos+1];
    flB = (float) imgData[nPos+2];
    /* Calculate The Height Using The Luminance Algorithm */
    return (0.299f*flR + 0.587f*flG + 0.114f*flB); 
}

static int tcl3dDemoUtilPhoto2Heightmap (Tcl_Interp *interp, char *photoName,
                                         void *vtxVector, void *texVector,
                                         float flHeightScale, float flResolution)
{
    Tk_PhotoHandle photo;
    Tk_PhotoImageBlock photoCont;
    int x, y, w, h, chansInImg, chansToSkip;
    int nX, nZ, nTri, vtxCount;
    float flX, flZ;
    unsigned char *srcPtr;
    float *vtxVec = (float *) vtxVector;
    float *texVec = (float *) texVector;

    photo = Tk_FindPhoto(interp, photoName);
    if (!photo) {
        Tcl_AppendResult (interp, "cannot find photo image: ",
                          photoName, (char *)NULL);
        return TCL_ERROR;
    }
    Tk_PhotoGetImage(photo, &photoCont);
    w = photoCont.width;
    h = photoCont.height;
    chansInImg = photoCont.pixelSize;

    if (chansInImg < 3) {
        Tcl_AppendResult (interp, "Photo must have 3 or more channels",
                          (char *)NULL);
        return TCL_ERROR;
    }
    if (chansInImg > 3) {
        chansToSkip = 1;
    } else {
        chansToSkip = 0;
    }

    srcPtr = photoCont.pixelPtr;
 
    vtxCount = (int) ( w * h * 6 / (flResolution * flResolution));
    for (nZ=0; nZ<h; nZ+=(int)flResolution) {
        for (nX=0; nX<w; nX+=(int)flResolution) {
            for (nTri=0; nTri<6; nTri++) {
                /* Using This Quick Hack, Figure The X,Z Position Of The Point */
                flX = (float)nX + ((nTri==1 || nTri==2 || nTri==5) ? flResolution: 0.0);
                flZ = (float)nZ + ((nTri==2 || nTri==4 || nTri==5) ? flResolution: 0.0);

                /* Set The Data, Using PtHeight To Obtain The Y Value */
                *vtxVec++ = flX - (w/2);
                *vtxVec++ = PtHeight ((int)flX, (int)flZ, srcPtr, w, h) * flHeightScale;
                *vtxVec++ = flZ - (h/2);

                /* Stretch The Texture Across The Entire Mesh */
                *texVec++ = flX / w;
                *texVec++ = flZ / h;
            }
        }
    }
    return TCL_OK;
}

%}

int tcl3dDemoUtilPhoto2Heightmap (Tcl_Interp *interp, char *photoName,
                                  void *vtxVector, void *texVector,
                                  float flHeightScale, float flResolution);
