"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.projects = exports.startNamedPipeServer = void 0;
const fs = require("fs");
const net = require("net");
const collectExtractProps_1 = require("./requests/collectExtractProps");
const componentInfos_1 = require("./requests/componentInfos");
const getImportPathForFile_1 = require("./requests/getImportPathForFile");
const getPropertiesAtLocation_1 = require("./requests/getPropertiesAtLocation");
const getQuickInfoAtPosition_1 = require("./requests/getQuickInfoAtPosition");
const utils_1 = require("./utils");
let started = false;
function startNamedPipeServer(ts, serverKind, currentDirectory) {
    if (started) {
        return;
    }
    started = true;
    const pipeFile = process.platform === 'win32'
        ? `\\\\.\\pipe\\vue-tsp-${process.pid}`
        : `/tmp/vue-tsp-${process.pid}`;
    const server = net.createServer(connection => {
        connection.on('data', data => {
            const text = data.toString();
            const request = JSON.parse(text);
            const fileName = request.args[0];
            const project = getProject(ts.server.toNormalizedPath(fileName));
            if (request.type === 'projectInfoForFile') {
                connection.write(JSON.stringify(project
                    ? {
                        name: project.info.project.getProjectName(),
                        kind: project.info.project.projectKind,
                    }
                    : null));
            }
            else if (project) {
                const requestContext = {
                    typescript: ts,
                    languageService: project.info.languageService,
                    languageServiceHost: project.info.languageServiceHost,
                    language: project.language,
                    isTsPlugin: true,
                    getFileId: (fileName) => fileName,
                };
                if (request.type === 'collectExtractProps') {
                    const result = collectExtractProps_1.collectExtractProps.apply(requestContext, request.args);
                    connection.write(JSON.stringify(result ?? null));
                }
                else if (request.type === 'getImportPathForFile') {
                    const result = getImportPathForFile_1.getImportPathForFile.apply(requestContext, request.args);
                    connection.write(JSON.stringify(result ?? null));
                }
                else if (request.type === 'getPropertiesAtLocation') {
                    const result = getPropertiesAtLocation_1.getPropertiesAtLocation.apply(requestContext, request.args);
                    connection.write(JSON.stringify(result ?? null));
                }
                else if (request.type === 'getQuickInfoAtPosition') {
                    const result = getQuickInfoAtPosition_1.getQuickInfoAtPosition.apply(requestContext, request.args);
                    connection.write(JSON.stringify(result ?? null));
                }
                // Component Infos
                else if (request.type === 'getComponentProps') {
                    const result = componentInfos_1.getComponentProps.apply(requestContext, request.args);
                    connection.write(JSON.stringify(result ?? null));
                }
                else if (request.type === 'getComponentEvents') {
                    const result = componentInfos_1.getComponentEvents.apply(requestContext, request.args);
                    connection.write(JSON.stringify(result ?? null));
                }
                else if (request.type === 'getTemplateContextProps') {
                    const result = componentInfos_1.getTemplateContextProps.apply(requestContext, request.args);
                    connection.write(JSON.stringify(result ?? null));
                }
                else if (request.type === 'getComponentNames') {
                    const result = componentInfos_1.getComponentNames.apply(requestContext, request.args);
                    connection.write(JSON.stringify(result ?? null));
                }
                else if (request.type === 'getElementAttrs') {
                    const result = componentInfos_1.getElementAttrs.apply(requestContext, request.args);
                    connection.write(JSON.stringify(result ?? null));
                }
                else {
                    console.warn('[Vue Named Pipe Server] Unknown request type:', request.type);
                }
            }
            else {
                console.warn('[Vue Named Pipe Server] No project found for:', fileName);
            }
            connection.end();
        });
        connection.on('error', err => console.error('[Vue Named Pipe Server]', err.message));
    });
    cleanupPipeTable();
    const table = (0, utils_1.readPipeTable)();
    table.push({
        path: pipeFile,
        serverKind,
        currentDirectory,
    });
    (0, utils_1.updatePipeTable)(table);
    try {
        fs.unlinkSync(pipeFile);
    }
    catch { }
    server.listen(pipeFile);
}
exports.startNamedPipeServer = startNamedPipeServer;
function cleanupPipeTable() {
    for (const server of (0, utils_1.readPipeTable)()) {
        (0, utils_1.connect)(server.path).then(client => {
            if (client) {
                client.end();
            }
            else {
                let table = (0, utils_1.readPipeTable)();
                table = table.filter(item => item.path !== server.path);
                (0, utils_1.updatePipeTable)(table);
            }
        });
    }
}
exports.projects = new Map();
function getProject(filename) {
    for (const [project, data] of exports.projects) {
        if (project.containsFile(filename)) {
            return data;
        }
    }
}
//# sourceMappingURL=server.js.map