/*
 * Copyright (c) 2005 Versant Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * Versant Corporation - initial API and implementation
 */

package org.eclipse.jsr220orm.core.util;

import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.util.HashMap;
import java.util.Map;

/**
 * Utility methods to convert type codes from java.sql.Types to/from Strings. 
 */
public class JdbcUtils {
	
    /**
     * qualified name -> instance   
     */
    private static Map namedValueMap = new HashMap(); 
    /**
     * java.sql.Types int constant value to qualified name "Types.VARCHAR" etc.
     */
    private static Map jdbcIntQNameMap = new HashMap();
    /**
     * java.sql.Types int constant value to name "VARCHAR" etc.
     */
    private static Map jdbcIntNameMap = new HashMap();
    /**
     * unqualified java.sql.Types constant name to value
     */
    private static Map jdbcNameIntMap = new HashMap();
    
    /**
     * Find and register all of the constants from java.sql.Types. 
     */
    static {
        Field[] a = java.sql.Types.class.getFields();
        for (int i = a.length - 1; i >= 0; i--) {
            Field f = a[i];
            int m = f.getModifiers();
            if (Modifier.isFinal(m) && Modifier.isStatic(m)) {
                Integer value;
                try {
                    value = (Integer)f.get(null);
                } catch (Exception e) {
                    // ignore, not possible really
                    continue;
                }
                String tname = f.getName();
                String name = "Types." + tname;
                String fullName = "java.sql." + name;
                jdbcIntQNameMap.put(value, name);
                jdbcIntNameMap.put(value, tname);
                jdbcNameIntMap.put(tname, (Integer)value);
                jdbcNameIntMap.put(name, (Integer)value);
                jdbcNameIntMap.put(fullName, (Integer)value);
            }
        }
    }
    
    /**
     * Convert the int value of a constant from java.sql.Types into its
     * qualified constant name (e.g. Types.VARCHAR) or null if it does not 
     * exist.
     */
    public static String getQualifiedJdbcTypeName(Integer value) {
        return (String) jdbcIntQNameMap.get(value);
    }
    
    /**
     * Convert the int value of a constant from java.sql.Types into its
     * constant name (e.g. Types.VARCHAR) or null if it does not 
     * exist.
     */
    public static String getJdbcTypeName(Integer value) {
        return (String) jdbcIntNameMap.get(value);
    }
    
    public static String getJdbcTypeName(int value) {
        return getJdbcTypeName(new Integer(value));
    }
    
    /**
     * Convert the name of a constant from java.sql.Types (e.g. VARCHAR,
     * Types.VARCHAR or java.sql.Types.VARCHAR) into its int value.
     * 
     * @exception IllegalArgumentException if name does not exist
     */
    public static int getJdbcTypeValue(String name) {
        Integer integer = (Integer)jdbcNameIntMap.get(name);
        if (integer == null) {
        	throw new IllegalArgumentException("Invalid JDBC type name: '" + 
        			name + "'");
        }
        return integer.intValue();
    }
 
    /**
     * Get the map of String to int used by {@link #getJdbcTypeValue(String)}.
     */
    public static Map getJdbcNameIntMap() {
    	return jdbcNameIntMap;
    }
    
}
