/*
 * Copyright (c) 2005 Versant Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * Versant Corporation - initial API and implementation
 */

package org.eclipse.jsr220orm.core.nature;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IProjectDescription;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.QualifiedName;
import org.eclipse.jsr220orm.core.internal.product.OrmProductRegistry;
import org.eclipse.jsr220orm.core.internal.product.OrmProject;

public class OrmNatureUtils {
    
    private static final QualifiedName PERSISTENCE_PROPS = new QualifiedName("OrmPersistence", "persistence_props");
    private static List natureListeners = new ArrayList(5);

    /**
     * Helper method to load the ORM properties for this project from the
     * .persistence file.
     * 
     * @throws CoreException
     * @throws IOException
     */
    static Properties getProperties(IProject project, boolean forceRefresh)
            throws CoreException, IOException {
        if(project == null || !project.isOpen()){
            return new Properties();
        }
        if(!project.isSynchronized(1)){
            project.refreshLocal(1, new NullProgressMonitor());
        }
        Properties properties = (Properties)  project.getSessionProperty(PERSISTENCE_PROPS);
        if (properties == null) {
            properties = new Properties();
            project.setSessionProperty(PERSISTENCE_PROPS, properties);
        }else{
            if(!forceRefresh){
                return properties;
            }
        }
        IFile persistProps = project.getFile(PersistenceProperties.PERSISTENCE_FILE_NAME);
        if (persistProps.exists()) {
            InputStream in = persistProps.getContents(true);
            try {
                properties.load(in);
            } finally {
                try {
                    in.close();
                } catch (Exception x) {
                    // Do nothing
                }
            }
        } else {
            // when a project is imported, we get a first delta for the addition
            // of the .project, but the .persistence is not accessible
            // so default to using java.io.File
            File file = persistProps.getLocation().toFile();
            if (file.exists()) {
                InputStream in = new FileInputStream(file);
                try {
                    properties.load(in);
                } finally {
                    try {
                        in.close();
                    } catch (Exception x) {
                        // Do nothing
                    }
                }
            }
        }
        return properties;
    }

    /**
     * Helper method to save the ORM properties for this project to the
     * .persistence file.
     * 
     * @throws IOException
     * @throws CoreException
     */
    static void setProperties(IProject project, Properties properties)
            throws IOException, CoreException {
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        properties.store(out,
                "Properties of the Persistence nature for eclipse project: "
                        + project.getName());
        InputStream inputStream = new ByteArrayInputStream(out.toByteArray());
        IFile persistProps = project.getFile(PersistenceProperties.PERSISTENCE_FILE_NAME);
        // update the resource content
        if (persistProps.exists()) {
            if (persistProps.isReadOnly()) {
                // provide opportunity to checkout read-only .persistence file
                ResourcesPlugin.getWorkspace().validateEdit(
                        new IFile[] { persistProps }, null);
            }
            persistProps.setContents(inputStream, IResource.FORCE, null);
        } else {
            persistProps.create(inputStream, IResource.FORCE, null);
        }
    }

    /**
     * Helper method to add or remove ORM nature and set the active product.
     */
    public static OrmNature setActiveProduct(IProject project, String prodKey)
            throws Exception {
        Properties properties = getProperties(project, true);
        if (prodKey != null) {
            OrmProductRegistry ormProductRegistry = OrmProductRegistry.INSTANCE;
            if (!ormProductRegistry.containtsProductKey(prodKey)) {
                throw new Exception(
                        "No Persistence plugin found for product key: "
                                + prodKey);
            }
            properties.setProperty(PersistenceProperties.ACTIVE_PERSISTENT_PRODUCT, prodKey);
            setProperties(project, properties);
            if (project.hasNature(OrmNature.ID)) {
                OrmNature nature = (OrmNature) project.getNature(OrmNature.ID);
                nature.loadActiveProject();
                return nature;
            }else{
                addOrmNature(project);
                return (OrmNature) project.getNature(OrmNature.ID);
            }
        } else {
            properties.remove(PersistenceProperties.ACTIVE_PERSISTENT_PRODUCT);
            setProperties(project, properties);
            if (project.hasNature(OrmNature.ID)) {
                removeOrmNature(project);
            }
            return null;
        }
    }

    private static void addOrmNature(IProject project) throws CoreException {
        IProjectDescription description = project.getDescription();
        String[] natures = description.getNatureIds();
        String[] newNatures = new String[natures.length + 1];
        System.arraycopy(natures, 0, newNatures, 0, natures.length);
        newNatures[natures.length] = OrmNature.ID;
        description.setNatureIds(newNatures);
        IStatus status = project.getWorkspace().validateNatureSet(natures);
        if (status.getCode() == IStatus.OK) {
            project.setDescription(description, null);
        } else {
            throw new CoreException(status);
        }
    }

    private static void removeOrmNature(IProject project) throws CoreException {
        IProjectDescription description = project.getDescription();
        String[] natures = description.getNatureIds();
        String id = OrmNature.ID;
        boolean hasNature = project.hasNature(id);
        if (!hasNature) {
            return;
        }
        OrmNature nature = (OrmNature) project.getNature(id);
        String[] newNatures = new String[natures.length - 1];
        for (int i = 0, x = 0; i < natures.length && x < newNatures.length; i++) {
            if (!natures[i].equals(id))
                newNatures[x++] = natures[i];
        }
        description.setNatureIds(newNatures);
        IStatus status = project.getWorkspace().validateNatureSet(natures);
        if (status.getCode() == IStatus.OK) {
            project.setDescription(description, null);
        } else {
            throw new CoreException(status);
        }
        if(nature != null){
            nature.closeProject();
            nature.fireNatureRemoved();
        }
    }

    public static void addOrmNatureListener(OrmNatureListener listener) {
        natureListeners.add(listener);
    }

    public static void removeOrmNatureListener(OrmNatureListener listener) {
        natureListeners.remove(listener);
    }

    public static void fireNatureRemoved(OrmNature nature) {
        OrmNatureListener[] listeners = new OrmNatureListener[natureListeners.size()];
        natureListeners.toArray(listeners);
        OrmNatureEvent ormNatureEvent = new OrmNatureEvent(nature, null);
        for(int x = listeners.length-1; x >= 0; x--){
            listeners[x].ormNatureRemoved(ormNatureEvent);
        }
    }
  
    public static void fireProjectActivated(OrmNature nature, OrmProject activeProject){
        OrmNatureListener[] listeners = new OrmNatureListener[natureListeners.size()];
        natureListeners.toArray(listeners);
        OrmNatureEvent ormNatureEvent = new OrmNatureEvent(nature, activeProject);
        for(int x = listeners.length-1; x >= 0; x--){
            listeners[x].projectActivated(ormNatureEvent);
        }
    }

    public static void fireProjectDeactivated(OrmNature nature, OrmProject activeProject) {
        OrmNatureListener[] listeners = new OrmNatureListener[natureListeners.size()];
        natureListeners.toArray(listeners);
        OrmNatureEvent ormNatureEvent = new OrmNatureEvent(nature, activeProject);
        for(int x = listeners.length-1; x >= 0; x--){
            listeners[x].projectDeactivated(ormNatureEvent);
        }
    }
}
