/*
 * Copyright (c) 2005 Versant Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * Versant Corporation - initial API and implementation
 */

package org.eclipse.jsr220orm.ui.internal.views;

import org.eclipse.emf.common.notify.Adapter;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.notify.Notifier;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jsr220orm.core.IEntityModelManager;
import org.eclipse.jsr220orm.core.nature.OrmNatureEvent;
import org.eclipse.jsr220orm.core.nature.OrmNatureListener;
import org.eclipse.jsr220orm.core.nature.OrmNatureUtils;
import org.eclipse.jsr220orm.metadata.EntityMetaData;
import org.eclipse.jsr220orm.metadata.EntityModel;
import org.eclipse.jsr220orm.metadata.MetadataElement;
import org.eclipse.jsr220orm.ui.internal.selection.PersistenceSelection;
import org.eclipse.jsr220orm.ui.internal.selection.PersistenceSelectionProvider;
import org.eclipse.ui.IPageListener;
import org.eclipse.ui.IPartListener;
import org.eclipse.ui.IPerspectiveDescriptor;
import org.eclipse.ui.IPerspectiveListener;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchPartSite;
import org.eclipse.ui.part.ViewPart;

public abstract class AbstractMetadataView extends ViewPart implements
        ISelectionChangedListener, IPartListener, IPageListener,
        IPerspectiveListener, OrmNatureListener {

    private ISelection selection;
    protected PersistenceSelectionProvider selectionProvider;
    private MetadataElement parentElement;
    private EntityModel entityModel;
    private boolean busy;
    private boolean needUpdate;
    
    public AbstractMetadataView() {
        OrmNatureUtils.addOrmNatureListener(this);
    }

    public void dispose() {
        super.dispose();
        OrmNatureUtils.removeOrmNatureListener(this);
    }

    public void selectionChanged(SelectionChangedEvent event) {
        ISelectionProvider eventSelectionProvider = event.getSelectionProvider();
        if(eventSelectionProvider != selectionProvider){
            if(eventSelectionProvider != null){
                eventSelectionProvider.removeSelectionChangedListener(this);
            }
            return;
        }
        selection = event.getSelection();
        doUpdate(false);
    }

    public void doUpdate(boolean dataChanged) {
        if(dataChanged){
            selection = null;
        }
        MetadataElement element = getMetadataElement();
        IWorkbenchPartSite site = getSite();
        if (site != null && site.getPage().isPartVisible(this)) {
            doUpdateImpl(element);
            needUpdate = false;
        }else{
            needUpdate = true;
        }
    }
    protected abstract void doUpdate(MetadataElement element);

    private void doUpdateImpl(MetadataElement element) {
    	// listen to our selection's parent so we can tell if our selection
    	// is removed from the model
    	// This doesn't solve the problem completely so left out for now
        if(element != null){
            if(entityModel == null){
                entityModel = (EntityModel) element.getAncestor(EntityModel.class);
                if(entityModel != null){
                    entityModel.eAdapters().add(parentListener);
                }
            }
        }
    	MetadataElement np = element == null 
    		? null : element.getParentElement();
    	if (np != parentElement) {
    		if (parentElement != null) {
    			parentElement.eAdapters().remove(parentListener);
    		}
    		if ((parentElement = np) != null) {
    			parentElement.eAdapters().add(parentListener);
    		}
    	}
        try{
            if (!busy) {
                busy = true;
                doUpdate(element);
            }
        }finally{
            busy = false;
        }
    }
    
    private final Adapter parentListener = new Adapter() {

		public void notifyChanged(Notification no) {
			switch (no.getEventType()) {
			case Notification.REMOVE:
			case Notification.ADD:
                asyncRefresh();
				break;
			};
		}

		public Notifier getTarget() {
			return null;
		}

		public void setTarget(Notifier newTarget) {
		}

		public boolean isAdapterForType(Object type) {
			return type instanceof Class 
				&& EntityMetaData.class.isAssignableFrom((Class)type)
				&& EntityModel.class.isAssignableFrom((Class)type);
		}
    };

    protected MetadataElement getMetadataElement() {
        if (selection == null && selectionProvider != null) {
            selection = selectionProvider.getSelection();
        }
        if (selection instanceof PersistenceSelection) {
            MetadataElement element = ((PersistenceSelection) selection)
                    .getSelectedElement();
            return element;
        }
        return null;
    }

    protected void setSite(IWorkbenchPartSite site) {
        super.setSite(site);
        IWorkbenchPage page = site.getPage();
        page.addPartListener(this);
        page.getWorkbenchWindow().addPageListener(this);
        page.getWorkbenchWindow().addPerspectiveListener(this);
        installWorkbenchPage(page.getActiveEditor());
    }

    protected void installWorkbenchPage(IWorkbenchPart part) {
        if(part == null){
            return;
        }
        boolean thisPart = part == this;
        if (thisPart) {
            if(selectionProvider == null){
                part = getSite().getPage().getActiveEditor();
                if(part == null){
                    return;
                }
//            }else{
//                return;
            }
        }
        PersistenceSelectionProvider newProvider = null;
            newProvider = (PersistenceSelectionProvider) part
                    .getAdapter(PersistenceSelectionProvider.class);
        if (selectionProvider != newProvider && newProvider != null) {
            if (selectionProvider != null) {
                selectionProvider.removeSelectionChangedListener(this);
            }
            selectionProvider = newProvider;
            if (selectionProvider != null) {
                selectionProvider.addSelectionChangedListener(this);
            }
        }
        if (!thisPart || needUpdate) {
            doUpdate(selectionProvider != null);
        }
    }

    public void partActivated(IWorkbenchPart part) {
        installWorkbenchPage(part);
    }

    public void partBroughtToTop(IWorkbenchPart part) {
    }

    public void partClosed(IWorkbenchPart part) {
    }

    public void partDeactivated(IWorkbenchPart part) {
    }

    public void partOpened(IWorkbenchPart part) {
        installWorkbenchPage(part);
    }

    public void pageActivated(IWorkbenchPage page) {
        installWorkbenchPage(page.getActivePart());
    }

    public void pageClosed(IWorkbenchPage page) {
    }

    public void pageOpened(IWorkbenchPage page) {
    }

    public void perspectiveActivated(IWorkbenchPage page,
            IPerspectiveDescriptor perspective) {
        if (page == getSite().getPage()) {
            installWorkbenchPage(page.getActivePart());
        }
    }

    public void perspectiveChanged(IWorkbenchPage page,
            IPerspectiveDescriptor perspective, String changeId) {
        if (page == getSite().getPage()) {
            installWorkbenchPage(page.getActivePart());
        }
    }

    public void ormNatureRemoved(OrmNatureEvent e) {
        refreshModel();
    }

    public void projectActivated(OrmNatureEvent e) {
        refreshModel();
    }

    public void projectDeactivated(OrmNatureEvent e) {
        refreshModel();
    }

    private void refreshModel() {
        if(entityModel != null){
            entityModel.eAdapters().remove(parentListener);
        }
        asyncRefresh();
    }
    
    private void asyncRefresh() {
        getSite().getShell().getDisplay().asyncExec(new Runnable() {
            public void run() {
                doUpdate(true);
            }
        });
    }
}