/*
 * Copyright (c) 2005 Versant Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * Versant Corporation - initial API and implementation
 */

package org.eclipse.jsr220orm.ui.internal.properties;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.jsr220orm.core.internal.product.OrmProduct;
import org.eclipse.jsr220orm.core.internal.product.OrmProductRegistry;
import org.eclipse.jsr220orm.core.nature.OrmNature;
import org.eclipse.jsr220orm.core.nature.OrmNatureUtils;
import org.eclipse.jsr220orm.core.nature.PersistenceProperties;
import org.eclipse.jsr220orm.core.util.RdbUtils;
import org.eclipse.jsr220orm.ui.OrmUiPlugin;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CCombo;
import org.eclipse.swt.custom.CLabel;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Link;
import org.eclipse.wst.rdb.internal.core.connection.ConnectionInfo;
import org.eclipse.wst.rdb.internal.core.definition.DatabaseDefinition;
import org.eclipse.wst.rdb.internal.core.definition.DatabaseDefinitionRegistry;
import org.eclipse.wst.rdb.internal.models.sql.schema.Database;
import org.eclipse.wst.rdb.internal.models.sql.schema.Schema;
import org.eclipse.wst.rdb.server.internal.ui.explorer.actions.popup.AddConnectionAction;

/**
 * This page activates the Eclipse ORM plugin and the selected vendor plugins. A
 * Table with available vendors is displayed for selection
 */
public class SelectProductComposite extends Composite {

	private static final String NONE =  "<NONE>";
    private IProject project;
    private CLabel lPersistenceProduct = null;
    private CCombo tPersistenceProduct = null;
    private Composite cOrmOptions = null;
    private Link lConnection = null;
    private CCombo tConnection = null;
    private CLabel lSchema = null;
    private CCombo tSchema = null;
    private CLabel lDataBase = null;
    private CCombo tDataBase = null;
    private CLabel lVersion = null;
    private CCombo tVersion = null;
    private boolean productChoice;
    
    public SelectProductComposite(Composite parent, boolean productChoice) {
		super(parent, SWT.NONE);
        initialize(productChoice);
    }

    private void initialize(boolean productChoice) {
        this.productChoice = productChoice;
        GridLayout gridLayout = new GridLayout();
        gridLayout.numColumns = 2;
        this.setLayout(gridLayout);
        GridData gridData = new GridData();
        gridData.widthHint = 200;
        lPersistenceProduct = new CLabel(this, SWT.NONE);          
        lPersistenceProduct.setText("Persistence Product");
        tPersistenceProduct = new CCombo(this, SWT.READ_ONLY | SWT.BORDER);
        tPersistenceProduct.setLayoutData(gridData);
        tPersistenceProduct.setEnabled(productChoice);
        createCOrmOptions();
        setSize(new org.eclipse.swt.graphics.Point(507,268));
	}

	public IProject getProject() {
		return project;
	}

	public void setProject(IProject project) throws CoreException, IOException {
		this.project = project;
        boolean hasProject = project != null;
        tPersistenceProduct.setEnabled(hasProject && productChoice);
        lConnection.setEnabled(hasProject);
        tConnection.setEnabled(hasProject);
        tSchema.setEnabled(hasProject);
        tDataBase.setEnabled(hasProject);
        tVersion.setEnabled(hasProject);
        if(!hasProject){
            return;
        }
        PersistenceProperties properties = new PersistenceProperties(project);
        DatabaseDefinitionRegistry ddr = RdbUtils.getDatabaseDefinitionRegistry();
        Iterator products = ddr.getProducts();
        tDataBase.removeAll();
        while(products.hasNext()){
            String prodName = (String) products.next();
            tDataBase.add(prodName);
        }
        tDataBase.setText(properties.getDatabaseName());
        loadVersions();
        tVersion.setText(properties.getDatabaseVersion());
        tDataBase.addModifyListener(new ModifyListener() {
            public void modifyText(ModifyEvent e) {
                loadVersions();
            }
        });
		tPersistenceProduct.removeAll();
		String activeProductId = properties.getPersistenceProduct();
        tPersistenceProduct.setText(NONE);
        if (activeProductId != null) {
            OrmProduct ormProduct = OrmProductRegistry.INSTANCE.getOrmProductById(activeProductId);
            if (ormProduct != null) {
                tPersistenceProduct.setText(ormProduct.getName());
            }
        }
        tPersistenceProduct.add(NONE);
        List ormProducts = new ArrayList(OrmProductRegistry.INSTANCE.getOrmProducts());
		for (Iterator it = ormProducts.iterator(); it.hasNext();) {
			OrmProduct product = (OrmProduct) it.next();
            String name = product.getName();
            tPersistenceProduct.add(name);
            tPersistenceProduct.setData(name, product);
		}
        loadSchemas(properties.getConnectionName());
        String schemaName = properties.getSchemaName();
        if (schemaName != null) {
            tSchema.setText(schemaName);
        }else{
            tSchema.setText(NONE);
        }
        tConnection.addModifyListener(new ModifyListener() {
            public void modifyText(ModifyEvent e) {
                loadSchemas();
                ConnectionInfo activeConnection = (ConnectionInfo) tConnection.getData(tConnection.getText());
                if(activeConnection != null){
                    DatabaseDefinition dd = activeConnection.getDatabaseDefinition();
                    tDataBase.setText(dd.getProduct());
                    tVersion.setText(dd.getVersion());
                }
            }
        });
	}

    private void loadVersions() {
        DatabaseDefinitionRegistry ddr = RdbUtils.getDatabaseDefinitionRegistry();
        tVersion.removeAll();
        Iterator versions = ddr.getVersions(tDataBase.getText());
        while(versions.hasNext()){
            String version = (String) versions.next();
            tVersion.add(version);
            tVersion.setText(version);
        }
    }

    private void loadSchemas(String connectionName) {
        tConnection.removeAll();
        if (connectionName != null) {
            tConnection.setText(connectionName);
        }else{
            tConnection.setText(NONE);
        }
        tConnection.add(NONE);
        ConnectionInfo[] connections = RdbUtils.getRdbConnections();
        if (connections != null) {
            for (int x = 0; x < connections.length; x++) {
                ConnectionInfo connection = connections[x];
                String name = connection.getName();
                tConnection.add(name);
                tConnection.setData(name, connection);
            }
        }
        loadSchemas();
    }

	private void loadSchemas() {
        ConnectionInfo activeConnection = (ConnectionInfo) tConnection.getData(tConnection.getText());
        tSchema.removeAll();
        tSchema.setText(NONE);
        if(activeConnection == null){
            tSchema.setEnabled(false);
        }else{
            tSchema.setEnabled(true);
            tSchema.add(NONE);
            try {
                if (project.hasNature(OrmNature.ID)) {
                    OrmNature nature = (OrmNature) project.getNature(OrmNature.ID);
                    if (nature != null) {
                        Schema activeSchema = nature.getActiveSchema();
                        if (activeSchema != null) {
                            Database database = activeSchema.getDatabase();
                            if(database != null && database.equals(activeConnection.getCachedDatabase())){
                                tSchema.setText(activeSchema.getName());
                            }
                        }
                    }
                }
            } catch (CoreException e) {
                OrmUiPlugin.logException(e, "Error loading schema.", getShell());
            }
            List schemas = OrmNature.getSchemas(null, activeConnection, null, null);
            if (schemas != null) {
                for (Iterator it = schemas.iterator(); it.hasNext();) {
                    Schema schema = (Schema) it.next();
                    String name = schema.getName();
                    tSchema.add(name);
                    tSchema.setData(name, schema);
                }
            }
        }
    }

    /**
     * This method initializes cOrmOptions	
     *
     */    
    private void createCOrmOptions() {
    	GridData gridData6 = new GridData();
    	gridData6.widthHint = 200;
    	GridData gridData5 = new GridData();
    	gridData5.widthHint = 50;
    	GridData gridData4 = new GridData();
    	gridData4.widthHint = 150;
    	GridLayout gridLayout1 = new GridLayout();
    	gridLayout1.numColumns = 2;
    	GridData gridData2 = new GridData();
    	gridData2.widthHint = 200;
    	GridData gridData1 = new GridData();
    	gridData1.horizontalSpan = 2;
    	gridData1.horizontalAlignment = org.eclipse.swt.layout.GridData.FILL;
    	gridData1.verticalSpan = 2;
    	gridData1.grabExcessHorizontalSpace = true;
    	cOrmOptions = new Composite(this, SWT.NONE);		   
    	cOrmOptions.setLayoutData(gridData1);
    	cOrmOptions.setLayout(gridLayout1);
    	lDataBase = new CLabel(cOrmOptions, SWT.NONE);
    	lDataBase.setText("Database");
    	tDataBase = new CCombo(cOrmOptions, SWT.BORDER | SWT.READ_ONLY);
    	tDataBase.setLayoutData(gridData6);
    	lVersion = new CLabel(cOrmOptions, SWT.NONE);
    	lVersion.setText("Version");
    	tVersion = new CCombo(cOrmOptions, SWT.BORDER | SWT.READ_ONLY);
    	tVersion.setLayoutData(gridData5);
    	lConnection = new Link(cOrmOptions, SWT.NONE);
    	lConnection.setText("<a>Connection</a>");
    	lConnection.addMouseListener(new org.eclipse.swt.events.MouseAdapter() { 
    		public void mouseDown(org.eclipse.swt.events.MouseEvent e) {    
                AddConnectionAction action = new AddConnectionAction();
                action.run(null);
                loadSchemas(null);
    		}
    	});
    	tConnection = new CCombo(cOrmOptions, SWT.BORDER | SWT.READ_ONLY);
    	tConnection.setEditable(false);
    	tConnection.setLayoutData(gridData2);
    	lSchema = new CLabel(cOrmOptions, SWT.NONE);
    	lSchema.setText("Schema");
    	tSchema = new CCombo(cOrmOptions, SWT.BORDER | SWT.READ_ONLY);
    	tSchema.setLayoutData(gridData4);
    }

    public boolean hasProducts() {
        return tPersistenceProduct.getItemCount() > 1;
    }

    public void clearSelection() {
        tPersistenceProduct.setText(NONE);
        tConnection.setText(NONE);
    }

    public String getSelectedProductId() {
        OrmProduct product = getSelectedProduct();
        if(product != null){
            return product.getId();
        }
        return null;
    }
    
    public OrmProduct getSelectedProduct() {
        return (OrmProduct) tPersistenceProduct.getData(tPersistenceProduct.getText());
    }

    public ConnectionInfo getSelectedConnection() {
        return (ConnectionInfo) tConnection.getData(tConnection.getText());
    }
    
    public String getSelectedSchema() {
        Schema schema = (Schema) tSchema.getData(tSchema.getText());
        if(schema != null){
            return schema.getName();
        }
        return null;
    }

    public boolean setProductOnProject() {
        try {
            OrmProduct selectedProduct = getSelectedProduct();
            PersistenceProperties properties = new PersistenceProperties(project);
            if(selectedProduct != null){
                properties.getProperties().putAll(selectedProduct.getProperties());
                properties.setPersistenceProduct(selectedProduct.getId());
            }
            ConnectionInfo selectedConnection = getSelectedConnection();
            if (selectedConnection != null) {
                properties.setConnectionName(selectedConnection.getName());
                properties.setConnectionUserName(selectedConnection
                        .getUserName());
                properties.setConnectionPassword(selectedConnection
                        .getPassword());
            }
            properties.setSchemaName(getSelectedSchema());
            properties.setDatabaseName(tDataBase.getText());
            properties.setDatabaseVersion(tVersion.getText());
            properties.save();
            if(selectedProduct != null){
                OrmNatureUtils.setActiveProduct(project, selectedProduct.getId());
            }
            return true;
        } catch (Exception e) {
            OrmUiPlugin.logException(e, "Could not set Persistence Product", getShell());
            return false;
        }
    }

    public void flushProps() {
        try {
            OrmProduct selectedProduct = getSelectedProduct();
            PersistenceProperties properties = new PersistenceProperties(project);
            if(selectedProduct != null){
                properties.getProperties().putAll(selectedProduct.getProperties());
                properties.setPersistenceProduct(selectedProduct.getId());
            }
            ConnectionInfo selectedConnection = getSelectedConnection();
            if (selectedConnection != null) {
                properties.setConnectionName(selectedConnection.getName());
                properties.setConnectionUserName(selectedConnection.getUserName());
                properties.setConnectionPassword(selectedConnection.getPassword());
            }
            properties.setSchemaName(getSelectedSchema());
            properties.setDatabaseName(tDataBase.getText());
            properties.setDatabaseVersion(tVersion.getText());
        } catch (Exception e) {
            //Do nothing
        }
    }
}  //  @jve:decl-index=0:visual-constraint="11,32"