/*
 * Copyright (c) 2005 Versant Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * Versant Corporation - initial API and implementation
 */

/*
 * Created on Jul 15, 2004
 */
package org.eclipse.jsr220orm.ui.internal.diagram.er.layout;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.draw2d.AbsoluteBendpoint;
import org.eclipse.draw2d.IFigure;
import org.eclipse.draw2d.PolygonDecoration;
import org.eclipse.draw2d.PolylineConnection;
import org.eclipse.draw2d.geometry.Dimension;
import org.eclipse.draw2d.geometry.Insets;
import org.eclipse.draw2d.geometry.Point;
import org.eclipse.draw2d.geometry.Rectangle;
import org.eclipse.draw2d.graph.CompoundDirectedGraph;
import org.eclipse.draw2d.graph.CompoundDirectedGraphLayout;
import org.eclipse.draw2d.graph.Edge;
import org.eclipse.draw2d.graph.Node;
import org.eclipse.draw2d.graph.NodeList;
import org.eclipse.draw2d.graph.Subgraph;
import org.eclipse.gef.ConnectionEditPart;
import org.eclipse.gef.editparts.AbstractConnectionEditPart;
import org.eclipse.jsr220orm.ui.internal.diagram.er.figure.TableFigure;
import org.eclipse.jsr220orm.ui.internal.diagram.er.part.ColumnEditPart;
import org.eclipse.jsr220orm.ui.internal.diagram.er.part.ColumnListEditPart;
import org.eclipse.jsr220orm.ui.internal.diagram.er.part.TableEditPart;

/**
 * Visitor with support for populating nodes and edges of DirectedGraph from
 * model objects
 * 
 * @author Phil Zoio
 */
public class DirectedGraphLayoutVisitor {

    private Map partToNodesMap;
    private CompoundDirectedGraph graph;
    private CompoundDirectedGraphLayout directedGraphLayout = new CompoundDirectedGraphLayout();

    /**
     * Public method for reading graph nodes
     */
    public void layoutDiagram(ColumnListEditPart diagram) {
        try {
            partToNodesMap = new HashMap();
            graph = new CompoundDirectedGraph();
            graph.setDefaultPadding(new Insets(0));
            addNodes(diagram);
            if (graph.nodes.size() > 0) {
                addEdges(diagram);
                directedGraphLayout.visit(graph);
                applyResults(diagram);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }

    }

    // ******************* SchemaDiagramPart contribution methods **********/

    protected void addNodes(ColumnListEditPart diagram) {
        IFigure fig = diagram.getFigure();
        for (int i = 0; i < diagram.getChildren().size(); i++) {
            TableEditPart tp = (TableEditPart) diagram.getChildren().get(i);
            addNodes(tp);
        }
    }

    /**
     * Adds nodes to the graph object for use by the GraphLayoutManager
     */
    protected void addNodes(TableEditPart tablePart) {
        Subgraph n = new Subgraph(tablePart);
        Dimension preferredSize = tablePart.getFigure().getPreferredSize();
        n.width = preferredSize.width;
        n.height = preferredSize.height;
        n.setPadding(new Insets(20));
        partToNodesMap.put(tablePart, n);
        graph.nodes.add(n);
        List cols = tablePart.getChildren();
        for (int i = 0; i < cols.size(); i++) {
            Object object = cols.get(i);
            if (object instanceof ColumnEditPart) {
                ColumnEditPart col = (ColumnEditPart) object;
                addNodes(n, col);
            }
        }
    }

    /**
     * Adds nodes to the graph object for use by the GraphLayoutManager
     */
    protected void addNodes(Subgraph table, ColumnEditPart column) {
        Node n = new Node(column);
        n.setParent(table);
        Rectangle bounds = column.getFigure().getBounds();
        n.width = 0;
        n.height = 0;
        n.setPadding(new Insets(0));
        table.addMember(n);
        partToNodesMap.put(column, n);
        Edge e = new Edge(null, table, n);
        graph.edges.add(e);
        graph.nodes.add(n);
    }

    protected void addEdges(ColumnListEditPart diagram) {
        for (int i = 0; i < diagram.getChildren().size(); i++) {
            TableEditPart tablePart = (TableEditPart) diagram.getChildren()
                    .get(i);
            addEdges(tablePart);
        }
    }

    // ******************* TablePart contribution methods **********/

    protected void addEdges(TableEditPart tablePart) {
        List cols = tablePart.getChildren();
        for (int i = 0; i < cols.size(); i++) {
            Object object = cols.get(i);
            if (object instanceof ColumnEditPart) {
                ColumnEditPart col = (ColumnEditPart) object;
                List outgoing = col.getSourceConnections();
                for (int x = 0; x < outgoing.size(); x++) {
                    ConnectionEditPart joinPair = (ConnectionEditPart) col
                            .getSourceConnections().get(x);
                    addEdges(joinPair);
                }
            }
        }
    }

    // ******************* RelationshipPart contribution methods **********/

    protected void addEdges(ConnectionEditPart joinPair) {
        Node source = (Node) partToNodesMap.get(joinPair.getSource());
        Node target = (Node) partToNodesMap.get(joinPair.getTarget());
        if(source == null || target == null){
            return;
        }
        Edge e = new Edge(joinPair, source, target);
        e.weight = 2;
        graph.edges.add(e);
        partToNodesMap.put(joinPair, e);
    }

    // ******************* SchemaDiagramPart apply methods **********/

    protected void applyResults(ColumnListEditPart diagram) {
        applyChildrenResults(diagram);
    }

    protected void applyChildrenResults(ColumnListEditPart diagram) {
        for (int i = 0; i < diagram.getChildren().size(); i++) {
            TableEditPart tablePart = (TableEditPart) diagram.getChildren()
                    .get(i);
            applyResults(tablePart);
        }
    }

    protected void applyOwnResults(ColumnListEditPart diagram) {
    }

    // ******************* TablePart apply methods **********/

    public void applyResults(TableEditPart tablePart) {

        Subgraph n = (Subgraph) partToNodesMap.get(tablePart);
        TableFigure tableFigure = (TableFigure) tablePart.getFigure();
//        Rectangle parentBounds = tableFigure.getParent().getBounds();
//        tableFigure.setLocation(new Point(n.x + parentBounds.x, 
//                n.y + parentBounds.y));
        Insets insets = tableFigure.getParent().getInsets();
        tableFigure.setLocation(new Point(n.x+insets.left, n.y+insets.top));

        Dimension preferredSize = tableFigure.getPreferredSize();
        tableFigure.setSize(preferredSize);
//        tableFigure.setSize(n.width, n.height);

        List cols = tablePart.getChildren();
        for (int i = 0; i < cols.size(); i++) {
            Object object = cols.get(i);
            if (object instanceof ColumnEditPart) {
                ColumnEditPart col = (ColumnEditPart) object;
                List outgoing = col.getSourceConnections();
                for (int x = 0; x < outgoing.size(); x++) {
                    AbstractConnectionEditPart joinPair = (AbstractConnectionEditPart) outgoing
                            .get(x);
                    applyResults(joinPair);
                }
            }
        }
    }

    // ******************* RelationshipPart apply methods **********/

    protected void applyResults(AbstractConnectionEditPart relationshipPart) {

        Edge e = (Edge) partToNodesMap.get(relationshipPart);
        if(e == null){
            return;
        }
        NodeList nodes = e.vNodes;

        PolylineConnection conn = (PolylineConnection) relationshipPart
                .getConnectionFigure();
        conn.setTargetDecoration(new PolygonDecoration());
        if (nodes != null) {
            List bends = new ArrayList();
            for (int i = 0; i < nodes.size(); i++) {
                Node vn = nodes.getNode(i);
                int x = vn.x;
                int y = vn.y;
                if (e.isFeedback) {
                    bends.add(new AbsoluteBendpoint(x, y + vn.height));
                    bends.add(new AbsoluteBendpoint(x, y));

                } else {
                    bends.add(new AbsoluteBendpoint(x, y));
                    bends.add(new AbsoluteBendpoint(x, y + vn.height));
                }
            }
            conn.setRoutingConstraint(bends);
        } else {
            conn.setRoutingConstraint(Collections.EMPTY_LIST);
        }

    }

}