/*
 * Copyright (c) 2005 Versant Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * Versant Corporation - initial API and implementation
 */

package org.eclipse.jsr220orm.generic.io;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import javax.persistence.JoinColumn;
import javax.persistence.JoinTable;
import javax.persistence.ManyToMany;
import javax.persistence.OneToMany;
import javax.persistence.OrderBy;
import javax.persistence.Table;
import javax.persistence.Transient;

import org.eclipse.jsr220orm.core.internal.options.IntOption;
import org.eclipse.jsr220orm.generic.GenericEntityModelManager;
import org.eclipse.jsr220orm.generic.Utils;
import org.eclipse.jsr220orm.generic.reflect.RAnnotatedElement;
import org.eclipse.jsr220orm.generic.reflect.RClass;
import org.eclipse.jsr220orm.metadata.AttributeMetaData;
import org.eclipse.jsr220orm.metadata.CollectionAttribute;
import org.eclipse.jsr220orm.metadata.EntityMetaData;
import org.eclipse.jsr220orm.metadata.EntityModel;
import org.eclipse.jsr220orm.metadata.Join;
import org.eclipse.jsr220orm.metadata.JoinPair;
import org.eclipse.jsr220orm.metadata.MetadataPackage;
import org.eclipse.jsr220orm.metadata.OrmColumn;
import org.eclipse.jsr220orm.metadata.OrmTable;
import org.eclipse.jsr220orm.metadata.ReferenceAttribute;
import org.eclipse.jsr220orm.metadata.TypeMetaData;

/**
 * Handles mapping for collection attributes (OneToMany, ManyToMany). 
 */
public class CollectionAttributeIO extends AttributeIO {

	protected CollectionAttribute amd;
	protected TypeMetaData oldElementType;
	protected String defaultJoinTableName;
	
	public static final IntOption MAPPING_MANY_TO_MANY = new IntOption(1,
			"Many to many", "Join table",
			Utils.getImage("ManyToMany16"));
	
	public static final IntOption MAPPING_ONE_TO_MANY = new IntOption(2,
			"One to many", "Inverse foreign key or join table",
			Utils.getImage("OneToMany16"));
	
	public CollectionAttributeIO(EntityIO entityIO) {
		super(entityIO);
	}
	
	public AttributeMetaData getAttributeMetaData() {
		return amd;
	}
	
    public void getPossibleMappings(List ans) {
        ans.add(MAPPING_MANY_TO_MANY);
        ans.add(MAPPING_ONE_TO_MANY);
        ans.add(MAPPING_NOT_PERSISTENT);
    }
    
	public IntOption getMapping() {
		if (amd == null || amd.isNonPersistent()) {
			return MAPPING_NOT_PERSISTENT;
		}
		if (amd.isOneToMany()) {
			return MAPPING_ONE_TO_MANY;
		} else {
			return MAPPING_MANY_TO_MANY;			
		}
	}
	
	public void setMapping(IntOption mapping) {
		if (MAPPING_NOT_PERSISTENT.equals(mapping)) {
			setPersistent(false);			
		} else {
			setPersistent(true);
			amd.setOneToMany(MAPPING_ONE_TO_MANY.equals(mapping));
		}
	}

	public boolean updateModelFromMetaData(RClass cls,
			RAnnotatedElement attribute, boolean metaDataChanged) {
		
		amd = (CollectionAttribute)initAttributeMetaData(amd, attribute, 
				MetadataPackage.eINSTANCE.getCollectionAttribute());
		if (amd.isNonPersistent()) {
			return true;
		}

		if (!hasTableAndPrimaryKey(entityIO.getEntityMetaData())) {
			return false;
		}

		EntityMetaData emd = entityIO.getEntityMetaData();
		if (!emd.isEntity()) {
			updateModelCleanup();
			entityIO.addProblem(getTypeName(emd) + " classes " +
					"may only have basic attributes", 
					attribute.getLocation());
			return true;
		}
		
		GenericEntityModelManager mm = entityIO.getModelManager();
		EntityModel model = mm.getEntityModel();
		
		ManyToMany manyToMany = attribute.getAnnotation(ManyToMany.class);
		OneToMany oneToMany = attribute.getAnnotation(OneToMany.class);
		
		AnnotationEx main;
		amd.setOneToMany(oneToMany != null);
		if (oneToMany != null) {
			main = (AnnotationEx)oneToMany;
			setFetchType(amd, oneToMany.fetch());
			amd.setCascadeType(getCascadeBits(oneToMany.cascade()));
			if (manyToMany != null) {
				entityIO.addProblem("ManyToMany and OneToMany may not be " +
						"used together", 
						((AnnotationEx)oneToMany).getLocation(null));
			}
		} else {
			if (manyToMany == null) {
				manyToMany = attribute.getAnnotation(ManyToMany.class, true);
			}
			main = (AnnotationEx)manyToMany;
			setFetchType(amd, manyToMany.fetch());
			amd.setCascadeType(getCascadeBits(manyToMany.cascade()));
		}

		String targetEntity;
		String[] ta = attribute.getActualTypeArguments();
		if (ta == null || ta.length != 1) {
			targetEntity = main.getClassValue("targetEntity");
			if (targetEntity == null) {
				entityIO.addProblem("Non-generic collections must specify " +
						"targetEntity", main.getLocation(null));		
			}
		} else {
			targetEntity = ta[0];
		}
		EntityMetaData target = null;
		if (targetEntity != null) {
			TypeMetaData tmd = model.findTypeByClassName(targetEntity);
			if (tmd == null) {
				String msg;
				if (mm.isPossibleEntity(targetEntity)) {
					msg = "Target " + targetEntity + " is not persistent";
				} else {
					msg = "Entity not found: '" + targetEntity + "'";
				}
				entityIO.addProblem(msg, main.getLocation("targetEntity"));		
			} else if (!(tmd instanceof EntityMetaData)) {
				entityIO.addProblem("Collections of non-entity types are " +
						"not supported", main.getLocation("targetEntity"));
			} else {
				target = (EntityMetaData)tmd;
				if (target.getEntityType() != EntityMetaData.TYPE_ENTITY) {
					entityIO.addProblem(
							target.getShortName() + " is " +
							getTypeName(target) + " so it may not be the " +
							"target of a persistent relationship", 
							attribute.getLocation());
					entityIO.addDependencyOn(target);
					target = null;
				} else if (!hasTableAndPrimaryKey(target)) {
					return false;
				}				
			}
		}
		amd.setElementType(oldElementType = target);
		entityIO.addDependencyOn(target);
		
		OrderBy orderBy = attribute.getAnnotation(OrderBy.class);
		if (orderBy != null) {
			amd.setOrderBy(orderBy.value());
		} else {
			amd.setOrderBy(null);
		}
		
		if (target == null) { // cleanup model and exit
			updateModelCleanup();
			return true;
		}

		boolean usingMappedBy = updateModelMappedBy(main, target);
		if (usingMappedBy) {
			OrmTable joinTable = amd.getJoinTable();
			if (joinTable != null) {
				joinTable.delete();
				amd.setJoinTable(null);
			}
		} else {
			updateModelJoinTable(attribute, metaDataChanged);
		}
		
		return true;
	}

	/**
	 * Get rid of anything for our attribute currently in the model. 
	 */
	protected void updateModelCleanup() {
		OrmTable joinTable = amd.getJoinTable();
		if (joinTable != null) {
			joinTable.delete();
			amd.setJoinTable(null);
		}
		amd.setMappedBy(null);
	}
	
	/**
	 * Update our model for any mappedBy inverse information.
	 */
	protected boolean updateModelMappedBy(AnnotationEx main, 
			EntityMetaData target) {
		String mappedBy = (String)main.get("mappedBy");
		AttributeMetaData mb = null;
		boolean usingMappedBy;
		if (usingMappedBy = (mappedBy != null && mappedBy.length() > 0)) {
			mb = getMappedByAttribute(main, mappedBy, target);
			Class expect = amd.isOneToMany() 
				? ReferenceAttribute.class 
				: CollectionAttribute.class;
			if (expect.isInstance(mb)) {
				if (amd.isOneToMany()) {
					if (mb.getJavaType() != entityIO.getEntityMetaData()) {
						entityIO.addProblem(
								"Reference '" + mappedBy + "' has incorrect type: " +
								mb.getJavaType().getClassName(),
								main.getLocation("mappedBy"));	
						mb = null;						
					}
				} else {
					CollectionAttribute ca = (CollectionAttribute)mb;
					TypeMetaData caet = ca.getElementType();
					if (caet != entityIO.getEntityMetaData()) {
						if (caet != null) {
							entityIO.addProblem(
									"Collection '" + mappedBy + "' contains " +
									"incorrect type: " + caet.getClassName(),
									main.getLocation("mappedBy"));	
						}
						mb = null;												
					}
				}
			} else if (mb != null) {
				entityIO.addProblem(
						"Attribute '" + mappedBy + "' is not valid for mappedBy",
						main.getLocation("mappedBy"));	
				mb = null;
			}
		}
		amd.setMappedBy(mb);
		return usingMappedBy;
	}	
	
	/**
	 * Update our joinTable in the model. 
	 */
	protected void updateModelJoinTable(RAnnotatedElement attribute, 
			boolean metaDataChanged) {

		GenericEntityModelManager mm = entityIO.getModelManager();
		
		EntityMetaData target = (EntityMetaData)amd.getElementType();
		EntityMetaData emd = entityIO.getEntityMetaData();
		OrmTable destTable = target.getTable();
		OrmTable srcTable = emd.getTable();
		List targetPk = destTable.getPrimaryKeyList();
		List srcPk = srcTable.getPrimaryKeyList();
		
		JoinTable joinTable = attribute.getAnnotation(JoinTable.class, true);
		Table tableAnn = joinTable.table();
		
		OrmTable table = entityIO.ensureTable(amd.getJoinTable());
        table.setParentElement(amd);
		table.setComment(getComment(amd));
		String tableName = tableAnn.name();
		defaultJoinTableName = mm.getDefaultJoinTableName(amd, 
				srcTable, destTable);
		if (tableName.length() == 0) {
			tableName = defaultJoinTableName;
		}
		table.setName(tableName);
		table.setCatalog(tableAnn.catalog());
		table.setSchema(tableAnn.schema());
		
		JoinColumn[] jca = joinTable.joinColumns();
		Join srcJoin = entityIO.getJoinIO().updateModelJoin(amd.getSrcJoin(), null, 
				jca.length == 0 ? null : jca, 
				((AnnotationEx)joinTable).getLocation("joinColumns"),
				metaDataChanged, table, srcTable, 
				mm.getJoinTableOwnerCNS(amd),
				getComment(amd), false, mm.getColumnPositionPrimaryKey());
		if (amd.getSrcJoin() != srcJoin) {
			amd.setSrcJoin(srcJoin);
		}
		
		jca = joinTable.inverseJoinColumns();
		Join destJoin = entityIO.getJoinIO().updateModelJoin(amd.getDestJoin(), null, 
				jca.length == 0 ? null : jca,
				((AnnotationEx)joinTable).getLocation("inverseJoinColumns"),
				metaDataChanged, table, destTable, 
				mm.getJoinTableInverseCNS(amd), 
				getComment(amd),
				amd.isOptional(), mm.getColumnPositionForeignKey());
		if (amd.getDestJoin() != destJoin) {
			amd.setDestJoin(destJoin);
		}
		
		if (srcJoin != null) {
			for (Iterator i = srcJoin.getSrcColumns().iterator(); i.hasNext(); ) {
				OrmColumn c = (OrmColumn)i.next();
				if (!c.isPrimaryKey()) {
					table.getPrimaryKeyList().add(c);
				}
			}
		}
		
		if (destJoin != null) {
			boolean oneToMany = amd.isOneToMany();
			for (Iterator i = destJoin.getSrcColumns().iterator(); i.hasNext(); ) {
				OrmColumn c = (OrmColumn)i.next();
				if (c.isPrimaryKey()) {
					if (oneToMany) {
						table.getPrimaryKeyList().remove(c);
					}
				} else {
					if (!oneToMany) {
						table.getPrimaryKeyList().add(c);						
					}
				}
			}				
		}
		
		table.sortColumns();
		
		if (amd.getJoinTable() != table) {
			amd.setJoinTable(table);
		}
	}
	
	public void updateMetaDataFromModel(RClass cls, RAnnotatedElement attribute) {
		GenericEntityModelManager mm = getModelManager();
		
		if (amd.isNonPersistent()) {
			Utils.removeAnnotation(attribute, OneToMany.class);			
			Utils.removeAnnotation(attribute, ManyToMany.class);			
			Utils.removeAnnotation(attribute, JoinTable.class);			
			ensureTransient(attribute);
			return;
		} else {
			Utils.removeAnnotation(attribute, Transient.class);			
		}
		
		AnnotationEx main = (AnnotationEx)attribute.getAnnotation(
				amd.isOneToMany() ? OneToMany.class : ManyToMany.class, true);
		if (main.getValueCount() == 0) {
			// if we will be creating a OneToOne or ManyToMany annotation then
			// get rid of any existing opposite one
			AnnotationEx toDelete = (AnnotationEx)attribute.getAnnotation(
					amd.isOneToMany() ? ManyToMany.class : OneToMany.class);
			if (toDelete != null) {
				toDelete.delete();
			}
		}
		
		TypeMetaData elementType = amd.getElementType();
		if (elementType == null) {
			// model is incomplete so exit now to avoid damaging meta data
			return;
		}
		
		String[] ta = attribute.getActualTypeArguments();
		if (ta == null || ta.length != 1) {
			main.set("targetEntity", amd.getElementType().getClassName());
		} else {
			main.set("targetEntity", null);
		}
		
		AttributeMetaData mb = amd.getMappedBy();
		main.set("mappedBy", mb == null ? null : mb.getName());
		if (mb == null) {
			main.setMarker(amd.isOneToMany() || mm.isUseMarkerAnnotations());
		} else {
			Utils.removeAnnotation(attribute, JoinTable.class);						
		}
		
		Utils.setIfNotNull(main, "fetch", getFetchType(amd));
		main.set("cascade", getCascadeTypes(amd.getCascadeType()));
		
		String ob = amd.getOrderBy();
		if (ob == null) {
			Utils.removeAnnotation(attribute, OrderBy.class);
		} else {
			AnnotationEx orderBy = (AnnotationEx)attribute.getAnnotation(
					OrderBy.class, true);
			orderBy.set("value", ob);
		}
		
		OrmTable table = amd.getJoinTable();
		if (table == null || oldElementType != elementType) {
			// we were non-persistent previously or our element type has 
			// changed - do nothing until our joinTable has been created
			// during the re-read
			return;
		}

		EntityMetaData target = (EntityMetaData)amd.getElementType();
		EntityMetaData emd = entityIO.getEntityMetaData();
		OrmTable destTable = target.getTable();
		OrmTable srcTable = emd.getTable();
		
		JoinTable joinTable = attribute.getAnnotation(JoinTable.class, true);
		AnnotationEx tableAnn = (AnnotationEx)joinTable.table();
		tableAnn.setDefault("name",	defaultJoinTableName);
		tableAnn.set("name", table.getName(), true);
		tableAnn.set("catalog", table.getCatalog(), true);
		tableAnn.set("schema", table.getSchema(), true);
		
		updateMetaDataJoin((AnnotationEx)joinTable, amd.getSrcJoin(), 
				"joinColumns", mm.getJoinTableOwnerCNS(amd));
		updateMetaDataJoin((AnnotationEx)joinTable, amd.getDestJoin(), 
				"inverseJoinColumns", mm.getJoinTableInverseCNS(amd));
	}

	protected void updateMetaDataJoin(AnnotationEx joinTableEx, Join join,
			String name, ColumnNameStrategy cns) {
		if (join == null) { // probably errors in meta data
			return;
		}
		List pairList = join.getPairList();
		int n = pairList.size();
		if (joinTableEx.setArraySize(name, n)) {
			JoinColumn[] a = (JoinColumn[])joinTableEx.get(name);
			for (int i = 0; i < n; i++) {
				entityIO.getJoinIO().updateMetaDataJoinColumn((AnnotationEx)a[i], join, 
						(JoinPair)pairList.get(i), cns, false);
			}
			// get rid of the annotations if they are all default
			if (n > 1) {
				int i = 0;
				for (; i < n; i++) {
					AnnotationEx jc = (AnnotationEx)a[i];
					int c = jc.getValueCount();
					if (jc.hasValue("referencedColumnName")) {
						--c;
					}
					if (c > 0) {
						break;
					}
				}	
				if (i == n) {
					joinTableEx.set(name, null);					
				}
			} else if (n == 1) {
				AnnotationEx jc = (AnnotationEx)a[0];
				if (jc.getValueCount() == 0) {
					joinTableEx.set(name, null);
				}				
			} else {
				joinTableEx.set(name, null);
			}
		}
	}
	
	public List getValidMappedByAttributes() {
		TypeMetaData et = amd.getElementType();
		if (!(et instanceof EntityMetaData)) {
			return Collections.EMPTY_LIST;
		}
		// the emd on amd will be null if it is non-persistent so dont use that
		EntityMetaData emd = entityIO.getEntityMetaData();
		EntityMetaData target = (EntityMetaData)et;
		ArrayList ans = new ArrayList();
		boolean oneToMany = amd.isOneToMany();
		Class expect = oneToMany 
			? ReferenceAttribute.class 
		    : CollectionAttribute.class;
		for (Iterator i = target.getAttributeList().iterator(); i.hasNext(); ) {
			AttributeMetaData a = (AttributeMetaData)i.next();
			if (a.getMappedBy() == null && expect.isInstance(a)) {
				if (oneToMany) {
					if (a.getJavaType() == emd) {
						ans.add(a);
					}
				} else {
					if (((CollectionAttribute)a).getElementType() == emd) {
						ans.add(a);
					}
				}
			}
		}
		return ans;
	}	
	
}
