/*
 * Copyright (c) 2005 Versant Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * Versant Corporation - initial API and implementation
 */

package org.eclipse.jsr220orm.generic;

import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Array;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.xmlbeans.XmlException;
import org.apache.xmlbeans.XmlOptions;
import org.eclipse.core.resources.ICommand;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IProjectDescription;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.jdt.core.IClasspathEntry;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.core.dom.ASTNode;
import org.eclipse.jdt.core.dom.CompilationUnit;
import org.eclipse.jdt.core.dom.ITypeBinding;
import org.eclipse.jdt.core.dom.StructuralPropertyDescriptor;
import org.eclipse.jsr220Orm.generic.xml.VendorDefinitionDocument;
import org.eclipse.jsr220Orm.xml.EntityManagerDocument;
import org.eclipse.jsr220orm.core.nature.PersistenceProperties;
import org.eclipse.jsr220orm.generic.io.AnnotationEx;
import org.eclipse.jsr220orm.generic.reflect.RAnnotatedElement;
import org.eclipse.jsr220orm.metadata.Join;
import org.eclipse.jsr220orm.metadata.OrmColumn;
import org.eclipse.jsr220orm.metadata.provider.MetadataEditPlugin;
import org.osgi.framework.Bundle;

/**
 * Misc static utility methods. 
 */
public class Utils {

    /**
     * XmlOptions to use when loading and saving deployment descriptions that 
     * already exist. 
     */
    protected static XmlOptions xmlOptions = new XmlOptions();
   
    /**
     * XmlOptions to use when saving new deployment descriptors (e.g. indent
     * size, pretty printing etc.)
     */
    protected static XmlOptions newFileXmlOptions = new XmlOptions();
  
    /**
     * Set default values for XmlOptions
     */
    static{
        xmlOptions.setLoadLineNumbers();
        xmlOptions.setUseDefaultNamespace();
        newFileXmlOptions.setSavePrettyPrint();
        newFileXmlOptions.setSavePrettyPrintIndent(4);
        newFileXmlOptions.setUseDefaultNamespace();
    }
    
    protected static final String PERSISTENCE_XML_NAMESPACE = 
        "http://jsr220orm.eclipse.org/xml";
    protected static final String VENDOR_DEFINITION_XML_NAMESPACE = 
        "http://jsr220orm.eclipse.org/generic/xml";
	
    /**
	 * Converts primitive Class objects (e.g. Integer.TYPE) into corresponding
	 * wrappers (e.g. Integer.class).
	 */
	public static final Map<Class, Class> PRIM_WRAPPER_MAP = new HashMap<Class, Class>();
	
	/**
	 * Converters wrapper Class objects (e.g. Integer.class) into corresponding
	 * default 'zero' instances (e..g new Integer(0)).
	 */
	public static final Map<Class, Object> WRAPPER_DEFAULT_MAP = new HashMap<Class, Object>();
	
	/**
	 * Converts primitive names (e.g. int) into their class objects 
	 * (Integer.TYPE).
	 */
	public static final Map<String, Class> PRIM_CLASS_MAP = new HashMap<String, Class>();
	
	static {
		Object[] a = new Object[]{
			Integer.TYPE, Integer.class, 0,
			Long.TYPE, Long.class, 0L,
			Short.TYPE, Short.class, (short)0,
			Byte.TYPE, Byte.class, (byte)0,
			Boolean.TYPE, Boolean.class, Boolean.FALSE,
			Character.TYPE, Character.class, (char)0,
			Float.TYPE, Float.class, 0.0f,
			Double.TYPE, Double.class, 0.0		
		};
		for (int i = 0; i < a.length; i += 3) {
			Class p = (Class)a[i];
			Class w = (Class)a[i + 1];
			PRIM_WRAPPER_MAP.put(p, w);
			WRAPPER_DEFAULT_MAP.put(w, a[i + 2]);
			PRIM_CLASS_MAP.put(p.getName(), p);
		}
	}		

	/**
	 * Close ins if it is not null, discarding all errors. 
	 */
    public static void close(InputStream ins) {
    	try {
    		if (ins != null) {
    			ins.close();
    		}
		} catch (IOException e) {
			// ignore
		}
    }
    
    /**
     * Convert a get method name into a property name.
     */
    public static String convertGetMethodToProperty(String methodName) {
    	char c = methodName.charAt(3);
    	if (Character.isLowerCase(c)) {
    		return methodName.substring(3);
    	}
    	return Character.toLowerCase(c) + methodName.substring(4);
    }
    
    /**
     * Convert a property name into a get method name.
     */
    public static String convertPropertyToGetMethod(String properyName) {
    	char c = properyName.charAt(0);
    	if (Character.isLowerCase(c)) {
    		return "get" + Character.toUpperCase(c) + properyName.substring(1);
    	} else {
    		return "get" + properyName;
    	}
    }
    
    /**
     * Replace one node in an AST tree with another. 
     */
	public static void replace(ASTNode old, ASTNode replacement) {
		StructuralPropertyDescriptor p = old.getLocationInParent();
		if (p == null) {
			// node is unparented
			return;
		}
		if (p.isChildProperty()) {
			old.getParent().setStructuralProperty(p, replacement);
			return;
		}
		if (p.isChildListProperty()) {
			List l = (List)old.getParent().getStructuralProperty(p);
			int i = l.indexOf(old);
			l.set(i, replacement);
		}
	} 
	
	/**
	 * Set value on the annotation if it is not null. If it is null then do
	 * nothing. 
	 */
	public static void setIfNotNull(AnnotationEx ann, String name, 
			Object value) {
		if (value != null) {
			ann.set(name, value);
		}
	}	
	
	/**
	 * Create marker location information for a node in an AST tree.
	 */
	public static Map createMarkerLocation(ASTNode node) {
		Map ans = new HashMap();
		CompilationUnit root = (CompilationUnit)node.getRoot();
		int start = node.getStartPosition();
		ans.put(IMarker.LINE_NUMBER, root.lineNumber(start));
		ans.put(IMarker.CHAR_START, start);
		ans.put(IMarker.CHAR_END, start + node.getLength());
		return ans;
	}	
	
	/**
	 * Extract the name of the attribute. For fields this is just its name.
	 * For methods (i.e. properties) this is the name of the method with
	 * get removed and the first character changed to lower case.
	 */
	public static String getAttributeName(RAnnotatedElement attribute) {
		if (attribute.isField()) {
			return attribute.getName();
		}
		return getAttributeNameForMethod(attribute.getName());
	}
	
	/**
	 * Convert a method name into an attribute name.
	 */
	public static String getAttributeNameForMethod(String s) {
		if (s.startsWith("get") && s.length() > 3) {
			char c = s.charAt(3);
			if (Character.isUpperCase(c)) {
				return Character.toLowerCase(c) + s.substring(4);
			} else {
				return s.substring(3);
			}
		}
		return s;		
	}
	
	/**
	 * Remove the annotation if it exists.
	 */
	public static void removeAnnotation(RAnnotatedElement attribute, 
			Class annotationType) {
		AnnotationEx a = (AnnotationEx)attribute.getAnnotation(annotationType);
		if (a != null) {
			a.delete();
		}
	}
	
	/**
	 * Load the class. This handles primtive names (e.g. int) and arrays
	 * (e.g. int[], java.lang.String[]) correctly. 
	 */
	public static Class loadClass(String className, boolean initialize, 
			ClassLoader loader) throws ClassNotFoundException {
		boolean array = className.endsWith("[]");
		if (array) {
			className = className.substring(0, className.length() - 2);
		}
		Class cls = PRIM_CLASS_MAP.get(className);
		if (cls == null) {
			cls = Class.forName(className, initialize, loader);
		}
		if (array) {
			return Array.newInstance(cls, 0).getClass();
		} else {
			return cls;
		}
	}

	/**
	 * Get the name of the type or null if not possible (resolveBindings
	 * failed). 
	 */
	public static String getTypeName(ITypeBinding binding) {
		if (binding == null) {
			return null;
		}
		if (binding.isParameterizedType()) {
			return binding.getBinaryName();
		}
		return binding.getQualifiedName();
	}	

    /**
     * Create our XmlBeans graph from the contents of persistence.xml 
     * as set in the .persistence for this project.
     */
    public static EntityManagerDocument loadPersistenceXml(IProject project) throws Exception {
        PersistenceProperties props = new PersistenceProperties(project);
        String name = props.getPersistenceFileName();
        return loadPersistenceXml(project, name);
    }
    
    /**
     * Create our XmlBeans graph from the contents of persistence.xml. If
     * this resource does not exist in the project then create a new
     * persistence.xml.
     */
    public static EntityManagerDocument loadPersistenceXml(IProject project, String fileName) throws Exception {
        IFile xmlFile = project.getFile(fileName);
        if(!xmlFile.isSynchronized(1)){
            xmlFile.refreshLocal(1, new NullProgressMonitor());
        }
        if (xmlFile.exists()) {
            InputStream ins = xmlFile.getContents();
            try {
                return EntityManagerDocument.Factory.parse(ins, 
                        getXmlOptions(PERSISTENCE_XML_NAMESPACE));
            } finally {
                close(ins);
            }
        } else {
            EntityManagerDocument newInstance = EntityManagerDocument.Factory.newInstance();
            newInstance.addNewEntityManager();
            return newInstance;
        }
    }

    /**
     * Overwrite persistence.xml from a XmlBeans graph. 
     */
    public static void savePersistenceXml(IProject project, EntityManagerDocument doc) throws Exception {
        PersistenceProperties props = new PersistenceProperties(project);
        String name = props.getPersistenceFileName();
        IFile xmlFile = project.getFile(name);
        XmlOptions ops;
        boolean exists = xmlFile.exists();
        if (exists) {
            ops = getNewFileXmlOptions(PERSISTENCE_XML_NAMESPACE);
        } else {
            ops = getXmlOptions(PERSISTENCE_XML_NAMESPACE);
        }
        InputStream ins = doc.newInputStream(ops);
        try {
            if (exists) {
                xmlFile.setContents(ins, IResource.FORCE, null);
            } else {
                xmlFile.create(ins, IResource.FORCE, null);
            }
        } finally {
            Utils.close(ins);
        }
    }   
    
    /**
     * Create new from XML resource on classpath. 
     */
    public static VendorDefinitionDocument loadVendorDefinition(String bundleNamespace, 
            String vendorDefResouce) 
            throws XmlException, IOException {
        InputStream ins = openBundleResource(bundleNamespace, vendorDefResouce);
        try {
            return VendorDefinitionDocument.Factory.parse(ins, 
                getXmlOptions(VENDOR_DEFINITION_XML_NAMESPACE));
        } finally {
            close(ins);
        }
    }

	public static InputStream openBundleResource(String bundleNamespace, 
			String resourceName) throws IOException {
		Bundle bundle = Platform.getBundle(bundleNamespace);
        URL resourceUrl = bundle.getResource(resourceName);
        if (resourceUrl == null) {
            resourceUrl = bundle.getResource("/" + resourceName);
            if (resourceUrl == null) {
                throw new IllegalArgumentException(
                        "Unable to find resource '" + 
                        resourceName + "' on " + bundleNamespace + " classpath");
            }
        }
        return resourceUrl.openStream();
	}

    /**
     * XmlOptions to use when loading and saving deployment descriptions that 
     * already exist. This returns a copy of the options configured so that 
     * defaultNamespace does not have to be declared in the input XML.
     */
    public static XmlOptions getXmlOptions(String defaultNamespace) {
        XmlOptions ops = new XmlOptions(xmlOptions);
        HashMap map = new HashMap();
        map.put("", defaultNamespace);
        ops.setLoadSubstituteNamespaces(map);
        return ops;
    }
    
    /**
     * XmlOptions to use when saving new deployment descriptors (e.g. indent
     * size, pretty printing etc.). This returns a copy of the options
     * configured so that defaultNamespace is not declared in the output
     * XML.
     */
    public static XmlOptions getNewFileXmlOptions(String defaultNamespace) {
        XmlOptions ops = new XmlOptions(newFileXmlOptions);
        HashMap map = new HashMap();
        map.put("", defaultNamespace);
        ops.setSaveImplicitNamespaces(map);
        return ops;
    }

    /**
     * Add a container variable to the project's build path.
     */
    public static void addCpVarTo(IJavaProject jProject, String classPathVariableName) throws JavaModelException{
        //create the container variable
        IClasspathEntry varEntry = JavaCore.newContainerEntry(new Path(classPathVariableName));

        IClasspathEntry[] oldclasspath = jProject.getRawClasspath();
        IClasspathEntry[] newclasspath =
            new IClasspathEntry[oldclasspath.length + 1];

        for (int i = 0; i < oldclasspath.length; i++) {
            IClasspathEntry iClasspathEntry = oldclasspath[i];
            if (iClasspathEntry.equals(varEntry)) {
                return;
            }
            newclasspath[i] = iClasspathEntry;
        }

        newclasspath[newclasspath.length - 1] = varEntry;
        jProject.setRawClasspath(newclasspath, null);
    }

    /**
     * Remove a container variable to the project's build path.
     */
    public static void removeCpVarFrom(IJavaProject jProject, String classPathVariableName) throws JavaModelException {
        //create the container variable
        IClasspathEntry varEntry = JavaCore.newContainerEntry(new Path(classPathVariableName));

        IClasspathEntry[] oldclasspath = jProject.getRawClasspath();
        int entryCount = 0;
        for (int i = 0; i < oldclasspath.length; i++) {
            if (varEntry.equals(oldclasspath[i])) {
                entryCount++;
            }
        }
        if(entryCount == 0){
            return;
        }
        IClasspathEntry[] newclasspath =
            new IClasspathEntry[oldclasspath.length - entryCount];
        entryCount = 0;
        for (int i = 0; i < newclasspath.length; ) {
            IClasspathEntry iClasspathEntry = oldclasspath[i+entryCount];
            if (iClasspathEntry.equals(varEntry)) {
                entryCount++;
                continue;
            }
            newclasspath[i] = iClasspathEntry;
            i++;
        }
        jProject.setRawClasspath(newclasspath, null);
    }
    	
    public static Object getImage(String name) {
        return MetadataEditPlugin.INSTANCE.getImage("full/obj16/" + name);
    }	
    
    /**
     * Delete the join and its src columns. Returns true if join was not null
     * (i.e. we did stuff) and false otherwise (i.e. NOP).
     */
    public static boolean deleteJoinAndSrcCols(Join join) {
    	if (join != null) {
			List srcCols = join.getSrcColumns();
			join.delete();
			for (int i = srcCols.size() - 1; i >= 0; i--) {
				((OrmColumn)srcCols.get(i)).delete();
			}    	
			return true;
    	}
    	return false;
    }

    public static void addBuilder(IProject project, String builderId) throws CoreException {
        if(builderId == null || builderId.trim().length() == 0){
            return;
        }
        IProjectDescription desp = project.getDescription();
        ICommand[] commands = desp.getBuildSpec();
        boolean found = false;

        for (int i = 0; i < commands.length; ++i) {
            String builderName = commands[i].getBuilderName();
            if (builderName.equals(builderId)) {
                found = true;
                break;
            }
        }

        if (!found) {
            ICommand command = desp.newCommand();
            command.setBuilderName(builderId);
            ICommand[] newCommands = new ICommand[commands.length + 1];

            // Add it after the other builders.
            System.arraycopy(commands, 0, newCommands, 0, commands.length);
            newCommands[commands.length] = command;
            desp.setBuildSpec(newCommands);
            project.setDescription(desp, null);
        }
    }

    public static void removeBuilder(IProject project, String builderId) throws CoreException {
        if(builderId == null || builderId.trim().length() == 0){
            return;
        }
        IProjectDescription desp = project.getDescription();
        ICommand[] commands = desp.getBuildSpec();

        boolean found = false;
        int index = -1;

        for (int i = 0; i < commands.length; ++i) {
            if (commands[i].getBuilderName().equals(builderId)) {
                index = i;
                found = true;
                break;
            }
        }

        if (found) {
            List list = new ArrayList(commands.length);
            for (int i = 0; i < commands.length; i++) {
                list.add(commands[i]);
            }
            list.remove(index);
            commands = new ICommand[commands.length - 1];
            list.toArray(commands);
            desp.setBuildSpec(commands);
            project.setDescription(desp, null);
        }
    }

}

