/*
 * Copyright (c) 2005 Versant Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 * Versant Corporation - initial API and implementation
 */

package org.eclipse.jsr220orm.generic;

import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IAdapterFactory;
import org.eclipse.emf.common.notify.Notification;
import org.eclipse.jdt.core.ElementChangedEvent;
import org.eclipse.jdt.core.IBuffer;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.core.dom.AST;
import org.eclipse.jdt.core.dom.ASTParser;
import org.eclipse.jdt.core.dom.CompilationUnit;
import org.eclipse.jdt.internal.ui.javaeditor.DocumentAdapter;
import org.eclipse.jdt.internal.ui.javaeditor.EditorUtility;
import org.eclipse.jdt.internal.ui.javaeditor.JavaEditor;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.IUndoManager;
import org.eclipse.jface.text.TextViewer;
import org.eclipse.jface.text.source.ISourceViewer;
import org.eclipse.jsr220Orm.generic.xml.VendorDefinitionDocument.VendorDefinition.CollectionType;
import org.eclipse.jsr220Orm.generic.xml.VendorDefinitionDocument.VendorDefinition.SimpleType;
import org.eclipse.jsr220orm.core.IEntityModelManager;
import org.eclipse.jsr220orm.core.OrmPlugin;
import org.eclipse.jsr220orm.core.internal.options.IntOption;
import org.eclipse.jsr220orm.core.nature.OrmNature;
import org.eclipse.jsr220orm.core.nature.PersistenceProperties;
import org.eclipse.jsr220orm.core.options.IIntOption;
import org.eclipse.jsr220orm.core.options.IOptionsDescriptor;
import org.eclipse.jsr220orm.core.util.JdbcUtils;
import org.eclipse.jsr220orm.core.util.RdbUtils;
import org.eclipse.jsr220orm.generic.VendorDef.JdbcTypeInfo;
import org.eclipse.jsr220orm.generic.io.AnnotationRegistry;
import org.eclipse.jsr220orm.generic.io.AttributeIO;
import org.eclipse.jsr220orm.generic.io.ColumnNameStrategy;
import org.eclipse.jsr220orm.generic.io.EntityIO;
import org.eclipse.jsr220orm.generic.io.ast.AstRClass;
import org.eclipse.jsr220orm.generic.io.ast.AstRClassFactory;
import org.eclipse.jsr220orm.generic.reflect.RAnnotatedElement;
import org.eclipse.jsr220orm.generic.reflect.RClass;
import org.eclipse.jsr220orm.generic.reflect.RMethod;
import org.eclipse.jsr220orm.metadata.AttributeMetaData;
import org.eclipse.jsr220orm.metadata.BasicAttribute;
import org.eclipse.jsr220orm.metadata.CollectionAttribute;
import org.eclipse.jsr220orm.metadata.CollectionTypeMetaData;
import org.eclipse.jsr220orm.metadata.EntityMetaData;
import org.eclipse.jsr220orm.metadata.EntityModel;
import org.eclipse.jsr220orm.metadata.MetadataFactory;
import org.eclipse.jsr220orm.metadata.OrmColumn;
import org.eclipse.jsr220orm.metadata.OrmTable;
import org.eclipse.jsr220orm.metadata.ReferenceAttribute;
import org.eclipse.jsr220orm.metadata.SimpleTypeMetaData;
import org.eclipse.jsr220orm.metadata.TypeMetaData;
import org.eclipse.text.edits.TextEdit;
import org.eclipse.ui.IEditorPart;
import org.eclipse.wst.rdb.internal.core.connection.ConnectionInfo;
import org.eclipse.wst.rdb.internal.core.definition.DatabaseDefinition;
import org.eclipse.wst.rdb.internal.models.dbdefinition.PredefinedDataTypeDefinition;

/**
 * IEntityModelManager for generic EJB 3 persistence.
 */
public class GenericEntityModelManager implements  IEntityModelManager, 
		IAdapterFactory {

	protected MetadataFactory factory;
    protected EntityModel model;
	protected IJavaProject project;
	protected OrmNature nature;
	protected PersistenceXmlManager persistenceXmlManager;	
	protected ASTParser astParser;
	protected AnnotationRegistry annotationRegistry;
	protected AstRClassFactory astRClassFactory;

    protected ConnectionInfo connectionInfo;
    protected int[] jdbcTypes;
    protected String vendorDefinitionResource = "vendor-generic.xml";
    protected VendorDef vendorDef;
    protected DatabaseDefinition dbdef;
    
    protected int ignoreEvents;
    protected Set<EntityIO> forMetaDataUpdate = new HashSet();
    protected String classPathVariableName;
    
    protected int startModelEventBatchCount;
    protected Set<EntityIO> batchSet = new LinkedHashSet();
    
    protected int columnPositionPrimaryKey = -5000;
    protected int columnPositionVersion = -4000;
    protected int columnPositionDiscriminator = -3000;
    protected int columnPositionForeignKey = 0;
    
    protected boolean useMarkerAnnotations;
    protected String builderId;
    
	protected static final int[] VALID_ACCESS_TYPES = new int[]{
		EntityMetaData.ACCESS_TYPE_FIELD,
		EntityMetaData.ACCESS_TYPE_PROPERTY
	};
	
	protected static final int[] VALID_ID_GENERATOR_TYPES = new int[]{
		EntityMetaData.ID_GENERATOR_TYPE_AUTO,
		EntityMetaData.ID_GENERATOR_TYPE_IDENTITY,
		EntityMetaData.ID_GENERATOR_TYPE_NONE,
		EntityMetaData.ID_GENERATOR_TYPE_SEQUENCE,
		EntityMetaData.ID_GENERATOR_TYPE_TABLE
	};	
	
	protected static final int[] VALID_FETCH_TYPES = new int[]{
		AttributeMetaData.FETCH_TYPE_EAGER,
		AttributeMetaData.FETCH_TYPE_LAZY
	};

    protected static final int[] VALID_CASCADE_TYPES = new int[]{
        AttributeMetaData.CASCADE_TYPE_PERSIST,
	    AttributeMetaData.CASCADE_TYPE_MERGE,
	    AttributeMetaData.CASCADE_TYPE_REMOVE,
	    AttributeMetaData.CASCADE_TYPE_REFRESH
	};
    
    protected static final int[] VALID_INHERITANCE_BASE = new int[]{
        EntityMetaData.INHERITANCE_NONE, 
        EntityMetaData.INHERITANCE_SINGLE_TABLE,
        EntityMetaData.INHERITANCE_JOINED
    };    
	
    protected static final int[] VALID_INHERITANCE_BASE_WITH_SUBS = new int[]{
    	EntityMetaData.INHERITANCE_SINGLE_TABLE,
    	EntityMetaData.INHERITANCE_JOINED
    };    
    
    protected static final int[] EMPTY_INT_ARRAY = new int[0];    
    
	public GenericEntityModelManager() {
	}

    public void init(IProject project, OrmNature nature, 
    		DatabaseDefinition definition) throws Exception {
		this.project = JavaCore.create(project);
		this.nature = nature;
        this.dbdef = definition;
        PersistenceProperties persistenceProperties = nature.getPersistenceProperties();
        builderId = persistenceProperties.getProperty("ejb3.verndor.builder");
        try {
            Utils.addBuilder(project, builderId);
        } catch (CoreException e) {
            GenericPlugin.getDefault().getLog().log(e.getStatus());
        }
        classPathVariableName = persistenceProperties.getProperty("classpath.variable.name");
        if (classPathVariableName != null) {
            try {
                Utils.addCpVarTo(this.project, classPathVariableName);
            } catch (JavaModelException e) {
                GenericPlugin.getDefault().getLog().log(e.getStatus());
            }
        }
        String vendorDefRes = persistenceProperties.getProperty("ejb3.vendor.definition");
        if (vendorDefRes != null) {
            vendorDefinitionResource = vendorDefRes;
        }
        factory = createMetadataFactory();
		astParser = ASTParser.newParser(AST.JLS3);
		annotationRegistry = createAnnotationRegistry();
		annotationRegistry.load(GenericPlugin.getPluginId());
		astRClassFactory = createAstRClassFactory();
		jdbcTypes = getAllJdbcTypeInts();
    	vendorDef = createVendorDef();
    	vendorDef.init(vendorDefinitionResource, this);
    	model = createEntityModel();
    	persistenceXmlManager = createPersistenceXmlManager();
    	persistenceXmlManager.loadPersistenceXml();
    	if (persistenceXmlManager.updateModelFromPersistenceXml()) {
    		persistenceXmlManager.savePersistenceXml();
    	}
	}
    
    /**
     * Cleanup anything not required after init.
     */
    public void initCleanup() {
    	vendorDef.initCleanup();
    }
    
    protected VendorDef createVendorDef() {
    	return new VendorDef();
    }
    
    protected PersistenceXmlManager createPersistenceXmlManager() {
    	return new PersistenceXmlManager(this);
    }
    
    protected AnnotationRegistry createAnnotationRegistry() {
    	return new AnnotationRegistry();
    }
    
    protected AstRClassFactory createAstRClassFactory() {
    	return new AstRClassFactory(project, annotationRegistry);
    }
    
    protected MetadataFactory createMetadataFactory() {
    	return MetadataFactory.eINSTANCE;
    }
    
    protected EntityModel createEntityModel() throws Exception {
    	EntityModel model = factory.createEntityModel();
    	List typeList = model.getTypeList();
    	
    	SimpleType[] simpleTypes = vendorDef.getSimpleTypes();
    	for (SimpleType st : simpleTypes) {
			SimpleTypeMetaData tmd = factory.createSimpleTypeMetaData();
			Class cls = Utils.loadClass(st.getName(), true, 
					getClass().getClassLoader());
			tmd.setCls(cls);
			tmd.setClassName(st.getName());
			OrmColumn c = factory.createOrmColumn();
			int jdbcType = JdbcUtils.getJdbcTypeValue(st.getJdbcType());
			c.setJdbcType(jdbcType);
			c.setLength(st.getLength());
			c.setScale(st.getScale());
			c.setNullable(st.getNullable());
			c.setInsertable(true);
			c.setUpdatable(true);
			VendorDef.JdbcTypeInfo info = vendorDef.getJdbcTypeInfo(jdbcType);
			if (info != null) {
				c.setDatabaseType(info.dataTypeName);
			}
			tmd.setColumn(c);
			typeList.add(tmd);
		}
    	
    	CollectionType[] collectionTypes = vendorDef.getCollectionTypes();
    	for (CollectionType ct : collectionTypes) {
			CollectionTypeMetaData cmd = factory.createCollectionTypeMetaData();
			Class cls = Utils.loadClass(ct.getName(), true, 
					getClass().getClassLoader());
			cmd.setCls(cls);
			cmd.setClassName(cls.getName());
			typeList.add(cmd);			
		}
    	
    	return model;
    }
    
	public OrmNature getNature() {
		return nature;
	}

	public IJavaProject getProject() {
		return project;
	}
	
	public MetadataFactory getFactory() {
		return factory;
	}

	public EntityModel getEntityModel() {
        return model;
    }

	public ASTParser getASTParser() {
		return astParser;
	}

	public AnnotationRegistry getAnnotationRegistry() {
		return annotationRegistry;
	}

	public AstRClassFactory getAstRClassFactory() {
		return astRClassFactory;
	}
	
	public DatabaseDefinition getDatabaseDefinition() {
		return dbdef;
	}

	public void dispose() {
        if (classPathVariableName != null) {
            try {
                Utils.removeCpVarFrom(project, classPathVariableName);
            } catch (JavaModelException e) {
                GenericPlugin.getDefault().getLog().log(e.getStatus());
            }
        }
        try {
            Utils.removeBuilder(project.getProject(), builderId);
        } catch (CoreException e) {
            GenericPlugin.getDefault().getLog().log(e.getStatus());
        }
        if(persistenceXmlManager != null){
            persistenceXmlManager.dispose();
        }
        if (model != null) {
			for (Iterator i = model.getTypeList().iterator(); i.hasNext(); ) {
				TypeMetaData tmd = (TypeMetaData)i.next();
				if (tmd instanceof EntityMetaData) {
					EntityIO entityIO = (EntityIO)tmd.adapt(EntityIO.class);
					if (entityIO != null) {
						entityIO.dispose();
					}
				}
			}
		}
		model = null;
	}

	public int[] getValidAccessTypes(EntityMetaData entityMetaData) {
		return VALID_ACCESS_TYPES;
	}
    
    public int[] getValidInheritances(EntityMetaData emd) {
    	if (emd.isBaseEntity()) {
    		if (emd.hasSubEntities()) {
    			return VALID_INHERITANCE_BASE_WITH_SUBS;
    		} else {
    			return VALID_INHERITANCE_BASE;
    		}
    	} else {
    		return EMPTY_INT_ARRAY;
    	}
    }
    
	public int[] getValidIdGeneratorTypes(EntityMetaData entityMetaData) {
		return VALID_ID_GENERATOR_TYPES;
	}

    public int[] getValidFetchTypes(AttributeMetaData attributeMetaData) {
        return VALID_FETCH_TYPES;
    }
    
    public int[] getValidCascadeTypes(ReferenceAttribute attributeMetaData){
        return VALID_CASCADE_TYPES;
    }
    
    public int[] getValidCascadeTypes(CollectionAttribute attributeMetaData){
        return VALID_CASCADE_TYPES;
    }
	
	/**
	 * Lookup an entity by class name. Returns null if no such entity in model.
	 */
	public EntityMetaData getEntityMetaData(String className) {
		TypeMetaData tmd = model.findTypeByClassName(className);
		return tmd instanceof EntityMetaData ? (EntityMetaData)tmd : null;
	}

    public void makePersistent(Collection names, IIntOption option) {
        try {
        	int type;
        	if (EntityIO.MAPPING_ENTITY.equals(option)) {
        		type = EntityMetaData.TYPE_ENTITY;
        	} else if (EntityIO.MAPPING_EMBEDDABLE.equals(option)) {
        		type = EntityMetaData.TYPE_EMBEDDABLE;
        	} else if (EntityIO.MAPPING_EMBEDDABLE_SUPERCLASS.equals(option)) {
        		type = EntityMetaData.TYPE_EMBEDDABLE_SUPERCLASS;
        	} else {
        		throw new IllegalArgumentException("Invalid option: " + 
        				option);
        	}
			persistenceXmlManager.makePersistent(names, type);
		} catch (Exception e) {
			OrmPlugin.log(e);
		}
    }
 
    public VendorDef getVendorDef() {
		return vendorDef;
	}

	public List getPossibleAttributeMappings(AttributeMetaData amd) {
        AttributeIO aio = (AttributeIO)amd.adapt(AttributeIO.class);
        List ans = new ArrayList();
        if (aio != null) {
        	aio.getPossibleMappings(ans);
        } else {
        	ans.add(AttributeIO.MAPPING_NOT_PERSISTENT);
        }
        return ans;
    }

    public IIntOption getAttributeMapping(AttributeMetaData amd) {
    	AttributeIO aio = (AttributeIO)amd.adapt(AttributeIO.class);
    	if (aio == null) {
    		return AttributeIO.MAPPING_NOT_PERSISTENT;
    	}
    	return aio.getMapping();
    }
    
    public void setAttributeMapping(AttributeMetaData amd, IIntOption op) {
    	System.out.println("%%% setAttributeMapping " + amd + " = " + op);
        AttributeIO aio = (AttributeIO)amd.adapt(AttributeIO.class);
		if (aio != null && !op.equals(aio.getMapping())) {
			try {
				++ignoreEvents;
				aio.setMapping((IntOption) op);
				updateMetaDataFromModel(Collections.singleton(aio.getEntityIO()));
			} finally {
				--ignoreEvents;
			}
		}
    }

    public int[] getValidJdbcTypes(OrmColumn column) {
        return jdbcTypes;
    }
    
    public void setDatabaseDefinition(DatabaseDefinition definition) {
        this.dbdef = definition;
    }
    
    /**
     * Find and register all of the constants from java.sql.Types. 
     */
    protected int[] getAllJdbcTypeInts() {
        ArrayList<Integer> types = new ArrayList<Integer>();
        Field[] a = java.sql.Types.class.getFields();
        for (int i = a.length - 1; i >= 0; i--) {
            Field f = a[i];
            int m = f.getModifiers();
            if (Modifier.isFinal(m) && Modifier.isStatic(m)) {
                Integer value;
                try {
                    value = (Integer) f.get(null);
                    types.add(value);
                } catch (Exception e) {
                    // ignore, not possible really
                    continue;
                }
            }
        }
        Collections.sort(types);
        int[] intTypes = new int[types.size()];
        for(int x = 0; x < types.size(); x++){
            intTypes[x] = types.get(x);
        }
        return intTypes;
    }

    /**
     * Get DDL for the column (i.e. a columnDefinition for it). This will
     * try to use RDB but fallback to just creating a generic String itself if
     * this fails. Note that this ignores the columnDefinition attribute
     * of the column.
     */
    public String getColumnDefinition(OrmColumn c) {
        int jdbcType = c.getJdbcType();
        JdbcTypeInfo info = vendorDef.getJdbcTypeInfo(jdbcType);
        PredefinedDataTypeDefinition typeDef = null;
        String typeName = null;
        if (info != null) {
            typeDef = info.dataTypeDef;
            typeName = info.dataTypeName;
        }
        return RdbUtils.getColumnDefinition(dbdef, typeDef, typeName, c);
    }
    
    /**
     * Get PredefinedDataTypeDefinition (RDB) for the column 
     */
    public PredefinedDataTypeDefinition getColumnDataTypeDefinition(OrmColumn c) {
        int jdbcType = c.getJdbcType();
        JdbcTypeInfo info = vendorDef.getJdbcTypeInfo(jdbcType);
        if (info != null) {
            return info.dataTypeDef;
        }
        return null;
    }

    /**
     * Update the databaseType attribute of col based on its name and
     * columnDefinition.
     * 
     * @see #getDatabaseType(String, String)
     */
    public void updateDatabaseType(OrmColumn col) {
    	col.setDatabaseType(getDatabaseType(col.getName(), 
    			col.getColumnDefinition()));
    }
    
    /**
     * Get the basic database type information from the name of the column and
     * its columnDefinition suitable for display on diagrams and so on (e.g. 
     * "VARCHAR(255)"). 
     */
    public String getDatabaseType(String name, String columnDefinition) {
    	int i = columnDefinition.indexOf(name);
    	if (i < 0) {
    		return columnDefinition;
    	}
    	i += name.length();
    	int j = columnDefinition.lastIndexOf("NOT NULL");
    	if (j < 0) {
    		j = columnDefinition.lastIndexOf("NULL");
    	}
    	String s;
    	if (j > 0) {
    		s = columnDefinition.substring(i, j);
    	} else {
    		s = columnDefinition.substring(i);
    	}
    	return s.trim();
    }
    
    /**
     * Return all entities in root and recursively those that depend on them.
     */
    protected Set<EntityIO> findDependentEntities(Collection<EntityIO> root) {
		Set<EntityIO> ans = new HashSet();
		ans.addAll(root);
    	for (;;) {
    		Set<EntityIO> extra = new HashSet();
    		for (EntityIO entityIO : ans) {
    			extra.addAll(entityIO.getDependOnUs());
			}
    		int before = ans.size();
    		ans.addAll(extra);
    		if (ans.size() == before) {
    			return ans;
    		}
    	}
    }

	/**
	 * Update the model for the entityIO in response to a source file change
	 * event.
	 */
	public void updateModelFromMetaData(EntityIO entityIO, 
			ElementChangedEvent event) {
		ArrayList a = new ArrayList(1);
		a.add(entityIO);
		updateModelFromMetaData(a, event, entityIO);
	}
    
	/**
	 * Update the model for the entityIO's in todoCol and all of the
	 * entities dependent on them.
	 */
	public void updateModelFromMetaData(Collection<EntityIO> todoCol) {
		updateModelFromMetaData(todoCol, null, null);
	}
	
	/**
	 * Update the model for the entityIO and all of the
	 * entities dependent on it.
	 */
	public void updateModelFromMetaData(EntityIO entityIO) {
		updateModelFromMetaData(Collections.singleton(entityIO), null, null);
	}
	
	protected void updateModelFromMetaData(Collection<EntityIO> todoCol, 
			ElementChangedEvent event, EntityIO eventTarget) {
		long start = System.currentTimeMillis();
		forMetaDataUpdate.clear();
		List<EntityIO> todo = new ArrayList(findDependentEntities(todoCol));
		Collections.sort(todo);
		System.out.println("%%% updateModelFromMetaData " + todo);
		for (Iterator<EntityIO> i = todo.iterator(); i.hasNext();) {
			EntityIO entityIO = i.next();
			try {
				entityIO.updateModelFromMetaDataPre();
			} catch (Exception e) {
				OrmPlugin.log(entityIO.toString(), e);
			}
		}
		// Keep calling each EntityIO's updateModelFromMetaData method until
		// all have completed processing and have been removed from todo set.
		// If all EntityIO's in the todo set return false indicating no 
		// progress then the loop is stopped and the model will be incomplete.
		try {
			++ignoreEvents;
			if (event != null) {
				eventTarget.renameAttributes(event);
			}
			astRClassFactory.begin();
			int pass = 1;
			for (; ; ++pass) {
				List<EntityIO> newTodo = new ArrayList();
				boolean nothingChanged = true;
				int n = todo.size();
				for (int i = 0; i < n; i++) {
					EntityIO entityIO = todo.get(i);
					try {
						RClass cls = astRClassFactory.findClass(
								entityIO.getType().getFullyQualifiedName(), false);
						int oldStatus = entityIO.getModelUpdateStatus();
						boolean changed = entityIO.updateModelFromMetaData(cls, 
								eventTarget == entityIO);
						int newStatus = entityIO.getModelUpdateStatus();
						if (newStatus != EntityIO.STATUS_COMPLETE) {
							newTodo.add(entityIO);					
						}
						if (changed || newStatus != oldStatus) {
							nothingChanged = false;
						}
					} catch (Exception e) {
						OrmPlugin.log(entityIO.toString(), e);
					}
				}
				todo = newTodo;
				if (nothingChanged || todo.isEmpty()) {
					break;
				}
			}
			for (Iterator<EntityIO> i = todoCol.iterator(); i.hasNext();) {
				EntityIO entityIO = i.next();
				try {
					RClass cls = astRClassFactory.findClass(
							entityIO.getType().getFullyQualifiedName(), false);
					entityIO.updateModelFromMetaDataPost(cls, 
							todo.contains(entityIO));
				} catch (Exception e) {
					OrmPlugin.log(entityIO.toString(), e);
				}
			}
			// write out meta data for all EntityIO's that need to do so
			for (EntityIO entityIO : forMetaDataUpdate) {
				updateMetaDataFromModelImp(entityIO);
			}
			long ms = System.currentTimeMillis() - start;
			System.out.println("%%% updateModelFromMetaData DONE " + 
					ms + " ms " + pass + " pass(es) " + todo);
		} finally {
			--ignoreEvents;
			astRClassFactory.finish();
		}
	}
	
	/**
	 * EntityIO's that require their meta data to be written out from the
	 * model call this method while updating the model. This is used to
	 * handle source code changes for mappedBy (bidirectional) relationships. 
	 * When the attribute owning the mapping is renamed the meta data for the 
	 * attribute on the mappedBy side needs to be written to reflect the
	 * new name.
	 */
	public void registerForMetaDataUpdate(EntityIO entityIO) {
		forMetaDataUpdate.add(entityIO);
	}

	/**
	 * Update the annotations and/or XML from the entity and then re-read this
	 * information back into the model to make sure everything is up to date.
	 * This also provides immediate feedback if model changes are not persisted
	 * to the meta data correctly as the changes will be undone by the re-read. 
	 */
	public void updateMetaDataFromModel(Collection<EntityIO> todoCol) {
        if(!project.isOpen()){
            return;
        }
		try {
			++ignoreEvents;
			long start = System.currentTimeMillis();
			List<EntityIO> todo = new ArrayList(todoCol);
			Collections.sort(todo);
	        System.out.println("%%% updateMetaDataFromModel " + todo);
			for (EntityIO entityIO : todo) {
				updateMetaDataFromModelImp(entityIO);
			}
			long ms = System.currentTimeMillis() - start;
	        System.out.println("%%% updateMetaDataFromModel DONE " + ms + " ms ");
			updateModelFromMetaData(todo);
		} finally {
			--ignoreEvents;
		}
	}	

	protected void updateMetaDataFromModelImp(EntityIO entityIO) {
		AstRClassFactory f = getAstRClassFactory();
		f.begin(entityIO.getAstState());
		try {
			AstRClass cls = (AstRClass)f.findClass(
					entityIO.getType().getFullyQualifiedName(), true);
			ICompilationUnit typeCu = cls.getTypeCu();
			CompilationUnit root = cls.getRoot();
			root.recordModifications();
			IBuffer buffer = typeCu.getBuffer();
			IDocument doc = ((DocumentAdapter)buffer).getDocument();

			entityIO.updateMetaDataFromModel(cls);
			
			TextEdit edits = root.rewrite(doc, 
					typeCu.getJavaProject().getOptions(true));
			
			IUndoManager undoManager = getUndoManager(entityIO.getType());
			try {
				if (undoManager != null) {
					undoManager.beginCompoundChange();
				}
				edits.apply(doc, TextEdit.UPDATE_REGIONS);
			} finally {
				if (undoManager != null) {
					undoManager.endCompoundChange();
				}
			}
			
			// reconcile now while we are ignoring events instead of letting
			// the automatic process reconcile and send an event later
			typeCu.reconcile(AST.JLS3, false, typeCu.getOwner(), null);
		} catch (Exception e) {
			OrmPlugin.log(e);
		} finally {
			f.finish();
			entityIO.setAstState(null);
		}
	}

	/**
	 * Get the IUndoManager for the editor for the source for type or
	 * null if not possible.
	 */
	protected IUndoManager getUndoManager(IType type) {
		IEditorPart ep = EditorUtility.isOpenInEditor(type);
		if (ep instanceof JavaEditor) {
			ISourceViewer v = ((JavaEditor)ep).getViewer();
			if (v instanceof TextViewer) {
				return ((TextViewer)v).getUndoManager();
			}
		}
		return null;
	}
	
	/**
	 * Our EntityIO's call this when they receive notification of changes
	 * to the model.
	 */
	public void notifyChanged(Notification no, EntityIO entityIO) {
		if (ignoreEvents > 0 
				|| no.getEventType() == Notification.REMOVING_ADAPTER) {
			return;
		}
		if (startModelEventBatchCount == 0) {
			updateMetaDataFromModel(Collections.singleton(entityIO));
		} else {
			batchSet.add(entityIO);
		}
	}

	public boolean isIgnoreEvents() {
		return ignoreEvents > 0;
	}
	
    public AttributeMetaData getAttributeMetaData(EntityMetaData emd, 
    		String name, boolean isMethod) {
    	if (isMethod) {
    		if (!name.startsWith("get")) {
    			return null;
    		}
    		name = Utils.getAttributeNameForMethod(name);
    	}
        AttributeMetaData amd = emd.findAttributeMetaData(name);
        if (amd == null) {
        	EntityIO entityIO = EntityIO.get(emd);
        	if (entityIO != null) {
        		amd = entityIO.getNonPeristentAttribute(name);
                if (amd != null) {
                    amd.registerAdapterFactory(this);
                }
        	}
        }
		return amd;
    }

    public Object getAdapter(Object adaptableObject, Class adapterType) {
        if(IEntityModelManager.class.equals(adapterType)){
            return this;
        }
        if(IOptionsDescriptor.class.equals(adapterType)){
            return getEntityModel().adapt(IOptionsDescriptor.class);
        }
        return null;
    }

    public Class[] getAdapterList() {
        return new Class[]{IEntityModelManager.class, IOptionsDescriptor.class};
    }

    public List getPossibleElementTypes(CollectionAttribute collectionAttribute) {
        //Todo: check for generic and return only that type (??and sub types??)
        List allTypes = getEntityModel().getTypeList();
        ArrayList possibleTypes = new ArrayList(allTypes.size());
        boolean supportBasicTypes = false; //Todo: get this value from vendor def
        boolean supportCollections = false; //Todo: get this value from vendor def
        boolean supportMaps = false; //Todo: get this value from vendor def
        for(Iterator types = allTypes.iterator(); types.hasNext();){
            TypeMetaData type = (TypeMetaData) types.next();
            if (type instanceof EntityMetaData) {
                possibleTypes.add(type);
            }
            //Todo: check booleans to add more types if possible
        }
        return possibleTypes;
    }
    
    /**
     * Get the default discriminator value for an entity. 
     */
    public String getDefaultDiscriminatorValue(EntityMetaData emd) {
    	return emd.getSchemaName();
    }

    public List getValidMappedByAttribute(AttributeMetaData amd) {
    	AttributeIO aio = AttributeIO.get(amd);
    	if (aio == null) {
    		return Collections.EMPTY_LIST;
    	}
    	return aio.getValidMappedByAttributes();
    }

	public int getColumnPositionDiscriminator() {
		return columnPositionDiscriminator;
	}

	public int getColumnPositionPrimaryKey() {
		return columnPositionPrimaryKey;
	}

	public int getColumnPositionForeignKey() {
		return columnPositionForeignKey;
	}

	public int getColumnPositionVersion() {
		return columnPositionVersion;
	}

    public boolean isUseDiscriminatorColumnEnabled(EntityMetaData emd) {
    	EntityIO entityIO = EntityIO.get(emd);
        return entityIO == null || entityIO.isUseDiscriminatorColumnEnabled();
    }

    public void setUseDiscriminatorColumn(EntityMetaData emd, boolean on) {
       	EntityIO entityIO = EntityIO.get(emd);
        if (entityIO != null) {
        	entityIO.setUseDiscriminatorColumn(on);
        }
    }

	public void executeModelChanges(Runnable runnable) {
		try {
			++startModelEventBatchCount;
			batchSet.clear();
			runnable.run();
		} finally {
			--startModelEventBatchCount;
		}
		if (startModelEventBatchCount == 0) {
			updateMetaDataFromModel(batchSet);
			batchSet.clear();
		}
	}

    /**
     * Return the default table name for an entity name.
     */
    public String getDefaultTableName(String entityName) {
        return entityName.toUpperCase();
    }
    
    /**
     * Return the default table name for a join table for a collection.
     */
	public String getDefaultJoinTableName(CollectionAttribute amd, 
			OrmTable srcTable, OrmTable destTable) {
		return srcTable.getName() + "_" + destTable.getName();
	}
	
	/**
	 * Get the strategy for naming owning column names in the join table
	 * for a collection.
	 */
	public ColumnNameStrategy getJoinTableOwnerCNS(
			final CollectionAttribute amd) {
		return USE_DEST_NAME;
	}
	    
	/**
	 * Get the strategy for naming owning column names in the join table
	 * for a collection.
	 */
	public ColumnNameStrategy getJoinTableInverseCNS(
			final CollectionAttribute amd) {
		return USE_DEST_NAME;
	}

	/**
	 * Get the strategy for naming column names for the table for emd that
	 * join to the primary key columns for its superclass.
	 */
	public ColumnNameStrategy getJoinedInheritanceCNS(
			final EntityMetaData emd) {
		return USE_DEST_NAME;
	}
	
	/**
	 * Get the strategy for naming column names for the table for emd that
	 * join to the primary key columns for its superclass.
	 */
	public ColumnNameStrategy getReferenceCNS(final ReferenceAttribute amd) {
		return new ColumnNameStrategy() {
			public String getColumnName(OrmColumn dest) {
				return amd.getName().toUpperCase() + "_" + dest.getName();
			}
		};
	}
	
	/**
	 * Use the name of the dest column.
	 */
	protected static final ColumnNameStrategy USE_DEST_NAME = 
			new ColumnNameStrategy() {
		public String getColumnName(OrmColumn dest) {
			return dest.getName();
		}
	};
	
	/**
	 * Get the default name of a column for a basic attribute. 
	 */
	public String getDefaultColumnName(BasicAttribute amd) {
		return amd.getName().toUpperCase();
	}
	
	/**
	 * Does fullyQualifiedName represent the name of a type that could
	 * be an entity (i.e. that could be made persistent)? Example: 
	 * A type that is class in the project. Returns false if 
	 * fullyQualifiedName is null.
	 */
	public boolean isPossibleEntity(String fullyQualifiedName) {
		if (fullyQualifiedName == null) {
			return false;
		}
		try {
			IType t = project.getJavaProject().findType(fullyQualifiedName);
			return t != null && !t.isBinary() && t.isClass();
		} catch (JavaModelException e) {
			OrmPlugin.log(e);
			return false;
		}
	}
	
	/**
	 * Is name a valid method attribute (i.e. property) name?
	 */
	public boolean isValidPropertyName(String name) {
		return name.length() > 3 && name.startsWith("get");
	}
	
	/**
	 * Is attribute persistent? Returns null if it is valid or an error 
	 * message.
	 */
	public String checkValidAttribute(RAnnotatedElement attribute) {
		int m = attribute.getModifiers();
		if (Modifier.isStatic(m)) {
			return "May not be static";
		}
		if (Modifier.isFinal(m)) {
			return "May not be final";
		}
		if (Modifier.isTransient(m)) {
			return "May not be transient";
		}
		if (attribute instanceof RMethod) {
			RMethod method = (RMethod)attribute;
			if (!isValidPropertyName(method.getName())) {
				return "Name must follow getXXX pattern";
			}
			if (!method.hasNoParameters()) {
				return "Cannot have parameters";
			}			
		}
		return null;
	}
	
	/**
	 * Should marker (i.e. empty) annotations be kept even when they could be
	 * safely removed? Example: Basic on an int field. 
	 */
	public boolean isUseMarkerAnnotations() {
		return useMarkerAnnotations;
	}
	
	/**
	 * Should the primary key columns for the entity have identity = true? 
	 */
	public boolean isUsingIdentityColumns(EntityMetaData emd) {
		return emd.getIdGeneratorType() 
			== EntityMetaData.ID_GENERATOR_TYPE_IDENTITY;
	}
	
	public boolean isIdGeneratorEnabled(EntityMetaData emd) {
		return !emd.getPrimaryKeyList().isEmpty();
	}

	public List getPossibleEntityMappings(EntityMetaData emd) {
		List ans = new ArrayList();
		EntityIO entityIO = EntityIO.get(emd);
		if (entityIO == null) {
			ans.add(EntityIO.MAPPING_ENTITY);
			ans.add(EntityIO.MAPPING_EMBEDDABLE);
			ans.add(EntityIO.MAPPING_EMBEDDABLE_SUPERCLASS);
		} else {
			entityIO.getPossibleEntityMappings(ans);
		}
		return ans;
	}

	public IIntOption getEntityMapping(EntityMetaData emd) {
		EntityIO entityIO = EntityIO.get(emd);
		if (entityIO == null) {
			return EntityIO.MAPPING_NOT_PERSISTENT;
		} else {
			return entityIO.getEntityMapping();
		}
	}

	public void setEntityMapping(EntityMetaData emd, IIntOption option) {
		System.out.println("$ setEntityMapping " + emd.getShortName() + " " +
				option);
		EntityIO entityIO = EntityIO.get(emd);
		if (entityIO != null) {
			if (EntityIO.MAPPING_NOT_PERSISTENT.equals(option)) {
				// special handling for the not persistent case as just
				// changing the entityType triggers useless round of model 
				// updating as removing it from persistence.xml rebuilds
				// the whole model anyway
				try {
					++ignoreEvents;
					entityIO.setEntityMapping(option);
					updateMetaDataFromModelImp(entityIO);
					persistenceXmlManager.makeNotPersistent(
							Collections.singleton(emd));
				} catch (Exception e) {
					OrmPlugin.log(e);
				} finally {
					--ignoreEvents;
				}
			} else {
				entityIO.setEntityMapping(option);
			}
		}
	}

    public Set getExtraSqlObjects() {
        return null;
    }
 }
