"use strict";
/*********************************************************************
 * Copyright (c) 2019 Kichwa Coders and others
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *********************************************************************/
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
const cp = require("child_process");
const path = require("path");
const utils_1 = require("./utils");
const chai_1 = require("chai");
const os = require("os");
describe('launch remote', function () {
    let dc;
    const emptyProgram = path.join(utils_1.testProgramsDir, 'empty');
    const emptySrc = path.join(utils_1.testProgramsDir, 'empty.c');
    beforeEach(function () {
        return __awaiter(this, void 0, void 0, function* () {
            dc = yield (0, utils_1.standardBeforeEach)('debugTargetAdapter.js');
        });
    });
    afterEach(function () {
        return __awaiter(this, void 0, void 0, function* () {
            yield dc.stop();
        });
    });
    it('can launch remote and hit a breakpoint', function () {
        return __awaiter(this, void 0, void 0, function* () {
            yield dc.hitBreakpoint((0, utils_1.fillDefaults)(this.test, {
                program: emptyProgram,
                target: {
                    type: 'remote',
                },
            }), {
                path: emptySrc,
                line: 3,
            });
        });
    });
    it('can print a message to the debug console sent from a socket server', function () {
        return __awaiter(this, void 0, void 0, function* () {
            const socketServer = cp.spawn('node', [`${path.join(utils_1.testProgramsDir, 'socketServer.js')}`], {
                cwd: utils_1.testProgramsDir,
            });
            // Ensure that the socket port is defined prior to the test.
            let socketPort = '';
            socketServer.stdout.on('data', (data) => {
                socketPort = data.toString();
                socketPort = socketPort.substring(0, socketPort.indexOf('\n'));
            });
            // Sleep for 1 second before running test to ensure socketPort is defined.
            yield new Promise((f) => setTimeout(f, 1000));
            (0, chai_1.expect)(socketPort).not.eq('');
            yield dc.getDebugConsoleOutput((0, utils_1.fillDefaults)(this.test, {
                program: emptyProgram,
                openGdbConsole: false,
                initCommands: ['break _fini'],
                target: {
                    uart: {
                        socketPort: socketPort,
                        eolCharacter: 'LF',
                    },
                },
            }), 'Socket', `Hello World!${os.EOL}`);
            // Kill the spawned process.
            socketServer.kill();
        });
    });
    it('can print a message to the debug console sent from across a serial line', function () {
        return __awaiter(this, void 0, void 0, function* () {
            // Skip this test on Windows - socat utility only available on Linux.
            if (os.platform() === 'win32')
                this.skip();
            // Start a virtual serial line. Use /tmp/ttyV0 and /tmp/ttyV1 to refer to the two ends.
            const virtualSerialLine = cp.spawn('socat', [
                '-d',
                '-d',
                'pty,rawer,echo=0,link=/tmp/ttyV0',
                'pty,rawer,echo=0,link=/tmp/ttyV1',
            ]);
            yield dc.getDebugConsoleOutput((0, utils_1.fillDefaults)(this.test, {
                program: emptyProgram,
                openGdbConsole: false,
                initCommands: ['break _fini'],
                preRunCommands: [`shell echo "Hello World!" > /tmp/ttyV1`],
                target: {
                    uart: {
                        serialPort: '/tmp/ttyV0',
                        eolCharacter: 'LF',
                        baudRate: 38400,
                    },
                },
            }), 'Serial Port', `Hello World!${os.EOL}`);
            // Kill the spawned process.
            virtualSerialLine.kill();
        });
    });
    it('can show user error on debug console if UART fails to open - Serial Port', function () {
        return __awaiter(this, void 0, void 0, function* () {
            const output = yield dc.getDebugConsoleOutput((0, utils_1.fillDefaults)(this.test, {
                program: emptyProgram,
                openGdbConsole: false,
                initCommands: ['break _fini'],
                target: {
                    uart: {
                        serialPort: '/mistake',
                    },
                },
            }), 'Serial Port', 'error on serial port connection', true);
            (0, chai_1.expect)(output.body.output).contains('mistake');
        });
    });
    it('can show user error on debug console if UART fails to open - Socket', function () {
        return __awaiter(this, void 0, void 0, function* () {
            const output = yield dc.getDebugConsoleOutput((0, utils_1.fillDefaults)(this.test, {
                program: emptyProgram,
                openGdbConsole: false,
                initCommands: ['break _fini'],
                target: {
                    uart: {
                        socketPort: '0',
                    },
                },
            }), 'Socket', 'error on socket connection', true);
            (0, chai_1.expect)(output.body.output).contains('0');
        });
    });
});
//# sourceMappingURL=launchRemote.spec.js.map