"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.getGdbCwd = exports.createEnvValues = exports.compareVersions = exports.parseGdbVersionOutput = exports.getGdbVersion = void 0;
/*********************************************************************
 * Copyright (c) 2022 Kichwa Coders Canada, Inc. and others
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *********************************************************************/
const child_process_1 = require("child_process");
const os_1 = require("os");
const util_1 = require("util");
const path_1 = require("path");
const fs_1 = require("fs");
/**
 * This method actually launches 'gdb --version' to determine the version of
 * the GDB that is being used.
 *
 * @param gdbPath the path to the GDB executable to be called
 * @return the detected version of GDB at gdbPath
 */
function getGdbVersion(gdbPath, gdbCwd, environment) {
    return __awaiter(this, void 0, void 0, function* () {
        const gdbEnvironment = environment
            ? createEnvValues(process.env, environment)
            : process.env;
        const { stdout, stderr } = yield (0, util_1.promisify)(child_process_1.execFile)(gdbPath, ['--version'], { cwd: gdbCwd, env: gdbEnvironment });
        const gdbVersion = parseGdbVersionOutput(stdout);
        if (!gdbVersion) {
            throw new Error(`Failed to get version number from GDB. GDB returned:\nstdout:\n${stdout}\nstderr:\n${stderr}`);
        }
        return gdbVersion;
    });
}
exports.getGdbVersion = getGdbVersion;
/**
 * Find gdb version info from a string object which is supposed to
 * contain output text of "gdb --version" command.
 *
 * @param stdout
 * 		output text from "gdb --version" command .
 * @return
 * 		String representation of version of gdb such as "10.1" on success
 */
function parseGdbVersionOutput(stdout) {
    return stdout.split(/ gdb( \(.*?\))? (\D* )*\(?(\d*(\.\d*)*)/g)[3];
}
exports.parseGdbVersionOutput = parseGdbVersionOutput;
/**
 * Compares two version numbers.
 * Returns -1, 0, or 1 if v1 is less than, equal to, or greater than v2, respectively.
 * @param v1 The first version
 * @param v2 The second version
 * @return -1, 0, or 1 if v1 is less than, equal to, or greater than v2, respectively.
 */
function compareVersions(v1, v2) {
    const v1Parts = v1.split(/\./);
    const v2Parts = v2.split(/\./);
    for (let i = 0; i < v1Parts.length && i < v2Parts.length; i++) {
        const v1PartValue = parseInt(v1Parts[i], 10);
        const v2PartValue = parseInt(v2Parts[i], 10);
        if (isNaN(v1PartValue) || isNaN(v2PartValue)) {
            // Non-integer part, ignore it
            continue;
        }
        if (v1PartValue > v2PartValue) {
            return 1;
        }
        else if (v1PartValue < v2PartValue) {
            return -1;
        }
    }
    // If we get here is means the versions are still equal
    // but there could be extra parts to examine
    if (v1Parts.length < v2Parts.length) {
        // v2 has extra parts, which implies v1 is a lower version (e.g., v1 = 7.9 v2 = 7.9.1)
        // unless each extra part is 0, in which case the two versions are equal (e.g., v1 = 7.9 v2 = 7.9.0)
        for (let i = v1Parts.length; i < v2Parts.length; i++) {
            const v2PartValue = parseInt(v2Parts[i], 10);
            if (isNaN(v2PartValue)) {
                // Non-integer part, ignore it
                continue;
            }
            if (v2PartValue != 0) {
                return -1;
            }
        }
    }
    if (v1Parts.length > v2Parts.length) {
        // v1 has extra parts, which implies v1 is a higher version (e.g., v1 = 7.9.1 v2 = 7.9)
        // unless each extra part is 0, in which case the two versions are equal (e.g., v1 = 7.9.0 v2 = 7.9)
        for (let i = v2Parts.length; i < v1Parts.length; i++) {
            const v1PartValue = parseInt(v1Parts[i], 10);
            if (isNaN(v1PartValue)) {
                // Non-integer part, ignore it
                continue;
            }
            if (v1PartValue != 0) {
                return 1;
            }
        }
    }
    return 0;
}
exports.compareVersions = compareVersions;
/**
 * This method is providing an automatic operation to including new variables to process.env.
 * Method is not injecting the new variables to current thread, rather it is returning a new
 * object with included parameters.
 *
 * @param source
 * 		Source environment variables to include.
 * @param valuesToMerge
 * 		Key-Value dictionary to include.
 * @return
 * 		New environment variables dictionary.
 */
function createEnvValues(source, valuesToMerge) {
    const findTarget = (obj, key) => {
        if ((0, os_1.platform)() === 'win32') {
            return (Object.keys(obj).find((i) => i.localeCompare(key, undefined, {
                sensitivity: 'accent',
            }) === 0) || key);
        }
        return key;
    };
    const result = Object.assign({}, source);
    for (const [key, value] of Object.entries(valuesToMerge)) {
        const target = findTarget(result, key);
        if (value === null) {
            delete result[target];
        }
        else {
            result[target] = value;
        }
    }
    return result;
}
exports.createEnvValues = createEnvValues;
/**
 * Calculate the CWD that should be used to launch gdb based on the program
 * being debugged or the explicitly set cwd in the launch arguments.
 *
 * Note that launchArgs.program is optional here in preparation for
 * debugging where no program is specified. See #262
 *
 * @param launchArgs Launch Arguments to compute GDB cwd from
 * @returns effective cwd to use
 */
function getGdbCwd(launchArgs) {
    const cwd = launchArgs.cwd ||
        (launchArgs.program && (0, fs_1.existsSync)(launchArgs.program)
            ? (0, path_1.dirname)(launchArgs.program)
            : process.cwd());
    return (0, fs_1.existsSync)(cwd) ? cwd : process.cwd();
}
exports.getGdbCwd = getGdbCwd;
//# sourceMappingURL=util.js.map