"use strict";
/*********************************************************************
 * Copyright (c) 2019 Ericsson and others
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *********************************************************************/
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
const chai_1 = require("chai");
const path = require("path");
const utils_1 = require("./utils");
describe('evaluate request', function () {
    let dc;
    let scope;
    const evaluateProgram = path.join(utils_1.testProgramsDir, 'evaluate');
    const evaluateSrc = path.join(utils_1.testProgramsDir, 'evaluate.cpp');
    beforeEach(function () {
        return __awaiter(this, void 0, void 0, function* () {
            dc = yield (0, utils_1.standardBeforeEach)();
            yield dc.hitBreakpoint((0, utils_1.fillDefaults)(this.currentTest, {
                program: evaluateProgram,
            }), {
                path: evaluateSrc,
                line: 2,
            });
            scope = yield (0, utils_1.getScopes)(dc);
        });
    });
    afterEach(function () {
        return __awaiter(this, void 0, void 0, function* () {
            yield dc.stop();
        });
    });
    it('should evaluate a simple literal expression', function () {
        return __awaiter(this, void 0, void 0, function* () {
            const res = yield dc.evaluateRequest({
                context: 'repl',
                expression: '2 + 2',
                frameId: scope.frame.id,
            });
            (0, chai_1.expect)(res.body.result).eq('4');
        });
    });
    it('should reject evaluation of expression without a frame', function () {
        return __awaiter(this, void 0, void 0, function* () {
            const err = yield (0, utils_1.expectRejection)(dc.evaluateRequest({
                context: 'repl',
                expression: '2 + 2',
            }));
            (0, chai_1.expect)(err.message).eq('Evaluation of expression without frameId is not supported.');
        });
    });
    it('should reject evaluation of invalid expression', function () {
        return __awaiter(this, void 0, void 0, function* () {
            const err = yield (0, utils_1.expectRejection)(dc.evaluateRequest({
                context: 'repl',
                expression: '2 +',
                frameId: scope.frame.id,
            }));
            (0, chai_1.expect)(err.message).eq('-var-create: unable to create variable object');
        });
    });
    it('should be able to update the value of a variable named monitor and that variable has local scope', function () {
        return __awaiter(this, void 0, void 0, function* () {
            const res1 = yield dc.evaluateRequest({
                context: 'repl',
                expression: 'monitor = 10',
                frameId: scope.frame.id,
            });
            (0, chai_1.expect)(res1.body.result).eq('10');
            const res2 = yield dc.evaluateRequest({
                context: 'repl',
                expression: 'monitor',
                frameId: scope.frame.id,
            });
            (0, chai_1.expect)(res2.body.result).eq('10');
        });
    });
    it('should be able to use GDB command', function () {
        return __awaiter(this, void 0, void 0, function* () {
            const res1 = yield dc.evaluateRequest({
                context: 'repl',
                expression: '>help',
                frameId: scope.frame.id,
            });
            (0, chai_1.expect)(res1.body.result).eq('\r');
            const res2 = yield dc.evaluateRequest({
                context: 'repl',
                expression: '>-gdb-version',
                frameId: scope.frame.id,
            });
            (0, chai_1.expect)(res2.body.result).eq('\r');
        });
    });
    it('should reject entering an invalid MI command', function () {
        return __awaiter(this, void 0, void 0, function* () {
            const err = yield (0, utils_1.expectRejection)(dc.evaluateRequest({
                context: 'repl',
                expression: '>-a',
                frameId: scope.frame.id,
            }));
            (0, chai_1.expect)(err.message).eq('Undefined MI command: a');
        });
    });
});
//# sourceMappingURL=evaluate.spec.js.map