"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.numberFormat = exports.getClosestPointForScatterPlot = exports.drawSelection = exports.xyChartFactory = void 0;
function xyChartFactory(params) {
    const baseOptions = getDefaultChartOptions(params);
    // If the chart is a line chart
    if (!params.isScatterPlot) {
        return getLineChartOptions(baseOptions);
    }
    return getScatterChartOptions(baseOptions);
}
exports.xyChartFactory = xyChartFactory;
function getLineChartOptions(lineOptions) {
    if (lineOptions.elements && lineOptions.elements.point && lineOptions.elements.line && lineOptions.scales) {
        lineOptions.elements.point.radius = 0;
        lineOptions.elements.line.borderWidth = 0;
        lineOptions.scales.xAxes = [{ id: 'time-axis', display: false }];
    }
    return lineOptions;
}
function getScatterChartOptions(scatterOptions) {
    if (scatterOptions.elements && scatterOptions.elements.point) {
        scatterOptions.elements.point.radius = 2;
    }
    return scatterOptions;
}
function getDefaultChartOptions(params) {
    var _a, _b, _c, _d;
    const offset = (_b = (_a = params.viewRange) === null || _a === void 0 ? void 0 : _a.getOffset()) !== null && _b !== void 0 ? _b : BigInt(0);
    const lineOptions = {
        responsive: true,
        elements: {
            point: { radius: 1 },
            line: {
                tension: 0,
                borderWidth: 2
            }
        },
        maintainAspectRatio: false,
        legend: { display: false },
        tooltips: {
            intersect: false,
            mode: 'index',
            enabled: false
        },
        layout: {
            padding: {
                left: 0,
                right: 0,
                top: 15,
                bottom: 5
            }
        },
        scales: {
            xAxes: [
                {
                    id: 'time-axis',
                    display: false,
                    ticks: {
                        min: Number(((_c = params.viewRange) === null || _c === void 0 ? void 0 : _c.getStart()) - offset),
                        max: Number(((_d = params.viewRange) === null || _d === void 0 ? void 0 : _d.getEnd()) - offset)
                    }
                }
            ],
            yAxes: [
                {
                    display: false,
                    stacked: false,
                    ticks: {
                        max: params.allMax > 0 ? params.allMax : 100,
                        min: params.allMin
                    }
                }
            ]
        },
        animation: { duration: 0 },
        events: ['mousedown']
    };
    return lineOptions;
}
function drawSelection(params) {
    var _a, _b;
    const { startPixel, endPixel, isBarPlot, chartArea, props, ctx, invertSelection } = params;
    const minPixel = Math.min(startPixel, endPixel);
    const maxPixel = Math.max(startPixel, endPixel);
    const initialPoint = isBarPlot ? 0 : (_a = chartArea === null || chartArea === void 0 ? void 0 : chartArea.left) !== null && _a !== void 0 ? _a : 0;
    const chartHeight = parseInt(props.style.height.toString());
    const finalPoint = isBarPlot ? chartHeight : (_b = chartArea === null || chartArea === void 0 ? void 0 : chartArea.bottom) !== null && _b !== void 0 ? _b : 0;
    if (ctx) {
        ctx.save();
        ctx.lineWidth = 1;
        // Selection borders
        if (startPixel > initialPoint) {
            ctx.beginPath();
            ctx.moveTo(minPixel, 0);
            ctx.lineTo(minPixel, finalPoint);
            ctx.stroke();
        }
        if (endPixel < props.viewRange.getEnd()) {
            ctx.beginPath();
            ctx.moveTo(maxPixel, 0);
            ctx.lineTo(maxPixel, finalPoint);
            ctx.stroke();
        }
        // Selection fill
        ctx.globalAlpha = 0.2;
        if (!invertSelection) {
            ctx.fillRect(minPixel, 0, maxPixel - minPixel, finalPoint);
        }
        else {
            const leftSideWidth = -minPixel;
            const rightSideWidth = ((chartArea === null || chartArea === void 0 ? void 0 : chartArea.right) ? chartArea.right : 0) - maxPixel;
            ctx.fillRect(minPixel, 0, leftSideWidth, finalPoint);
            ctx.fillRect(maxPixel, 0, rightSideWidth, finalPoint);
        }
        ctx.restore();
    }
}
exports.drawSelection = drawSelection;
function getClosestPointForScatterPlot(params) {
    var _a;
    let min_hypotenuse = Number.MAX_VALUE;
    let closestPoint = undefined;
    const offset = (_a = params.range.getOffset()) !== null && _a !== void 0 ? _a : BigInt(0);
    const start = params.range.getStart();
    const end = params.range.getEnd();
    const xRatio = params.chartWidth / Number(end - start);
    const yRatio = (params.chartHeight - params.margin.top - params.margin.bottom) / (params.allMax - params.allMin);
    params.dataPoints.forEach((point) => {
        const x = (point.x - Number(start - offset)) * xRatio;
        const y = (point.y - params.allMin) * yRatio + params.margin.bottom;
        const distX = params.mousePosition.x - x;
        const distY = params.chartHeight - params.mousePosition.y - y;
        const hypotenuse = distX * distX + distY * distY;
        if (min_hypotenuse > hypotenuse) {
            closestPoint = point;
            min_hypotenuse = hypotenuse;
        }
    });
    // Return closest point only if it is in a circle with a radius of 20 pixels
    if (min_hypotenuse < 400) {
        return closestPoint;
    }
    return undefined;
}
exports.getClosestPointForScatterPlot = getClosestPointForScatterPlot;
function numberFormat(rawNumber) {
    return new Intl.NumberFormat().format(rawNumber);
}
exports.numberFormat = numberFormat;
//# sourceMappingURL=xy-output-component-utils.js.map