"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StyleProvider = void 0;
const query_helper_1 = require("tsp-typescript-client/lib/models/query/query-helper");
const style_properties_1 = require("./style-properties");
class StyleProvider {
    constructor(outputId, traceId, tspClient) {
        this.outputId = outputId;
        this.tspClient = tspClient;
        this.traceId = traceId;
        const threadStyleObject = {
            '0': {
                color: '646464',
                height: 0.33
            },
            '2': {
                color: '00C800',
                height: 1
            },
            '3': {
                color: '0000C8',
                height: 1
            },
            '4': {
                color: 'C80064',
                height: 0.75
            },
            '1': {
                color: 'C8C800',
                height: 0.5
            },
            '5': {
                color: 'C86400',
                height: 0.5
            },
            '6': {
                color: 'C8C8C8',
                height: 0.5
            }
        };
        const resourceStyleObject = {
            '0': {
                color: 'C8C8C8',
                height: 0.66
            },
            '2': {
                color: '00C800',
                height: 1
            },
            '4': {
                color: '0000C8',
                height: 1
            },
            '16': {
                color: 'C80064',
                height: 0.75
            },
            '8': {
                color: 'C89664',
                height: 1
            },
            '1': {
                color: 'C8C800',
                height: 1
            }
        };
        this.tmpStyleObject = {
            'org.eclipse.tracecompass.internal.analysis.os.linux.core.threadstatus.ThreadStatusDataProvider': threadStyleObject,
            'org.eclipse.tracecompass.internal.analysis.os.linux.core.threadstatus.ResourcesStatusDataProvider': resourceStyleObject
        };
    }
    /**
     * Get the style model for a specific output
     * @param forceUpdate Force the update of the current cached style model from the server
     */
    async getStyleModel(forceUpdate) {
        if (!this.styleModel || forceUpdate) {
            const tspClientResponse = await this.tspClient.fetchStyles(this.traceId, this.outputId, query_helper_1.QueryHelper.query());
            const styleResponse = tspClientResponse.getModel();
            if (tspClientResponse.isOk() && styleResponse) {
                this.styleModel = styleResponse.model;
            }
        }
        return this.styleModel;
    }
    getStylesTmp(_forceUpdate) {
        const styles = this.tmpStyleObject[this.outputId];
        return styles ? styles : {};
    }
    /**
     * Get the style property value for the specified element style. The style
     * hierarchy is traversed until a value is found.
     *
     * @param elementStyle
     *            the style
     * @param property
     *            the style property
     * @return the style property value, or undefined
     */
    getStyle(elementStyle, property) {
        let style = elementStyle;
        const styleQueue = [];
        while (style !== undefined) {
            const styleValues = style.values ? style.values : {};
            const value = styleValues[property];
            if (value) {
                return value;
            }
            // Get the next style
            style = this.popNextStyle(style, styleQueue);
        }
        return undefined;
    }
    /**
     * Get the style property number value for the specified element style. The
     * style hierarchy is traversed until a number value is found, and the
     * returned number value will be multiplied by the first
     * StyleProperties.FACTOR suffixed modifier style that was found
     * along the way, if any.
     *
     * @param elementStyle
     *            the style
     * @param property
     *            the style property
     * @return the style property number value, or undefined
     */
    getNumberStyle(elementStyle, property) {
        let factor = undefined;
        let style = elementStyle;
        const styleQueue = [];
        while (style) {
            const styleValues = style.values ? style.values : {};
            if (factor === undefined) {
                const factorValue = styleValues[property + style_properties_1.StyleProperties.FACTOR];
                if (typeof factorValue === 'number') {
                    factor = factorValue;
                }
            }
            const value = styleValues[property];
            if (typeof value === 'number') {
                const numberValue = value;
                return factor === undefined ? numberValue : factor * numberValue;
            }
            // Get the next style
            style = this.popNextStyle(style, styleQueue);
        }
        return factor;
    }
    /**
     * Get the style property color value for the specified element style. The
     * style hierarchy is traversed until a color and opacity value is found,
     * and the returned color value will be blended with the first
     * StyleProperties.BLEND suffixed modifier style that was found
     * along the way, if any.
     *
     * @param elementStyle
     *            the style
     * @param property
     *            the style property
     * @return the style property color value, or undefined
     */
    getColorStyle(elementStyle, property) {
        let color = undefined;
        let opacity = undefined;
        let blend = undefined;
        let style = elementStyle;
        const styleQueue = [];
        while (style) {
            const styleValues = style.values ? style.values : {};
            if (blend === undefined) {
                const blendValue = styleValues[property + style_properties_1.StyleProperties.BLEND];
                if (typeof blendValue === 'string') {
                    blend = this.rgbaStringToColor(blendValue);
                }
            }
            if (opacity === undefined) {
                const opacityValue = styleValues[style_properties_1.StyleProperties.OPACITY];
                if (typeof opacityValue === 'number') {
                    opacity = opacityValue;
                    if (color) {
                        break;
                    }
                }
            }
            if (color === undefined) {
                const value = styleValues[property];
                if (typeof value === 'string') {
                    color = value;
                    if (opacity) {
                        break;
                    }
                }
            }
            // Get the next style
            style = this.popNextStyle(style, styleQueue);
        }
        const alpha = opacity === undefined ? 1.0 : opacity;
        const rgba = color === undefined
            ? opacity === undefined
                ? undefined
                : this.rgbaToColor(0, 0, 0, alpha)
            : this.rgbStringToColor(color, alpha);
        return rgba === undefined ? undefined : blend === undefined ? rgba : this.blend(rgba, blend);
    }
    popNextStyle(style, styleQueue) {
        var _a;
        // Get the next style
        let nextStyle = undefined;
        const parentKey = style.parentKey;
        if (parentKey) {
            const split = parentKey.split(',');
            split.forEach(styleKey => styleQueue.push(styleKey));
        }
        while (nextStyle === undefined && styleQueue.length !== 0) {
            const nextKey = styleQueue.pop();
            if (nextKey) {
                nextStyle = (_a = this.styleModel) === null || _a === void 0 ? void 0 : _a.styles[nextKey];
            }
        }
        return nextStyle;
    }
    blend(color1, color2) {
        /**
         * If a color component 'c' with alpha 'a' is blended with color
         * component 'd' with alpha 'b', the blended color and alpha are:
         *
         * <pre>
         * color = (a*(1-b)*c + b*d) / (a + b - a*b)
         * alpha = (a + b - a*b)
         * </pre>
         */
        const alpha = color1.alpha + color2.alpha - color1.alpha * color2.alpha;
        const r = this.blendComponent(color1.alpha, (color1.color >> 16) & 0xff, color2.alpha, (color2.color >> 16) & 0xff, alpha);
        const g = this.blendComponent(color1.alpha, (color1.color >> 8) & 0xff, color2.alpha, (color2.color >> 8) & 0xff, alpha);
        const b = this.blendComponent(color1.alpha, color1.color & 0xff, color2.alpha, color2.color & 0xff, alpha);
        return this.rgbaToColor(r, g, b, alpha);
    }
    blendComponent(alpha1, color1, alpha2, color2, alpha) {
        return Math.floor((alpha1 * (1.0 - alpha2) * color1 + alpha2 * color2) / alpha);
    }
    rgbStringToColor(rgbString, alpha) {
        const color = parseInt(rgbString.replace(/^#/, ''), 16);
        return { color, alpha };
    }
    rgbaStringToColor(rgbaString) {
        const int = parseInt(rgbaString.replace(/^#/, ''), 16);
        const color = (int >> 8) & 0xffffff;
        const alpha = (int & 0xff) / 255;
        return { color, alpha };
    }
    rgbaToColor(r, g, b, alpha) {
        const color = (r << 16) + (g << 8) + b;
        return { color, alpha };
    }
}
exports.StyleProvider = StyleProvider;
//# sourceMappingURL=style-provider.js.map