"use strict";
/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
Object.defineProperty(exports, "__esModule", { value: true });
exports.renderMarkdownAsPlaintext = exports.renderStringAsPlaintext = exports.allowedMarkdownAttr = exports.renderMarkdown = void 0;
const DOM = require("./dom");
const dompurify = require("./dompurify/dompurify");
const event_1 = require("./event");
const formattedTextRenderer_1 = require("./formattedTextRenderer");
const mouseEvent_1 = require("./mouseEvent");
const iconLabels_1 = require("./ui/iconLabel/iconLabels");
const errors_1 = require("../common/errors");
const event_2 = require("../common/event");
const htmlContent_1 = require("../common/htmlContent");
const iconLabels_2 = require("../common/iconLabels");
const idGenerator_1 = require("../common/idGenerator");
const lazy_1 = require("../common/lazy");
const lifecycle_1 = require("../common/lifecycle");
const marked_1 = require("../common/marked/marked");
const marshalling_1 = require("../common/marshalling");
const network_1 = require("../common/network");
const objects_1 = require("../common/objects");
const resources_1 = require("../common/resources");
const strings_1 = require("../common/strings");
const uri_1 = require("../common/uri");
const defaultMarkedRenderers = Object.freeze({
    image: (href, title, text) => {
        let dimensions = [];
        let attributes = [];
        if (href) {
            ({ href, dimensions } = (0, htmlContent_1.parseHrefAndDimensions)(href));
            attributes.push(`src="${(0, htmlContent_1.escapeDoubleQuotes)(href)}"`);
        }
        if (text) {
            attributes.push(`alt="${(0, htmlContent_1.escapeDoubleQuotes)(text)}"`);
        }
        if (title) {
            attributes.push(`title="${(0, htmlContent_1.escapeDoubleQuotes)(title)}"`);
        }
        if (dimensions.length) {
            attributes = attributes.concat(dimensions);
        }
        return '<img ' + attributes.join(' ') + '>';
    },
    paragraph: (text) => {
        return `<p>${text}</p>`;
    },
    link: (href, title, text) => {
        if (typeof href !== 'string') {
            return '';
        }
        // Remove markdown escapes. Workaround for https://github.com/chjj/marked/issues/829
        if (href === text) { // raw link case
            text = (0, htmlContent_1.removeMarkdownEscapes)(text);
        }
        title = typeof title === 'string' ? (0, htmlContent_1.escapeDoubleQuotes)((0, htmlContent_1.removeMarkdownEscapes)(title)) : '';
        href = (0, htmlContent_1.removeMarkdownEscapes)(href);
        // HTML Encode href
        href = href.replace(/&/g, '&amp;')
            .replace(/</g, '&lt;')
            .replace(/>/g, '&gt;')
            .replace(/"/g, '&quot;')
            .replace(/'/g, '&#39;');
        return `<a href="${href}" title="${title || href}">${text}</a>`;
    },
});
/**
 * Low-level way create a html element from a markdown string.
 *
 * **Note** that for most cases you should be using [`MarkdownRenderer`](./src/vs/editor/contrib/markdownRenderer/browser/markdownRenderer.ts)
 * which comes with support for pretty code block rendering and which uses the default way of handling links.
 */
function renderMarkdown(markdown, options = {}, markedOptions = {}) {
    var _a;
    const disposables = new lifecycle_1.DisposableStore();
    let isDisposed = false;
    const element = (0, formattedTextRenderer_1.createElement)(options);
    const _uriMassage = function (part) {
        let data;
        try {
            data = (0, marshalling_1.parse)(decodeURIComponent(part));
        }
        catch (e) {
            // ignore
        }
        if (!data) {
            return part;
        }
        data = (0, objects_1.cloneAndChange)(data, value => {
            if (markdown.uris && markdown.uris[value]) {
                return uri_1.URI.revive(markdown.uris[value]);
            }
            else {
                return undefined;
            }
        });
        return encodeURIComponent(JSON.stringify(data));
    };
    const _href = function (href, isDomUri) {
        const data = markdown.uris && markdown.uris[href];
        let uri = uri_1.URI.revive(data);
        if (isDomUri) {
            if (href.startsWith(network_1.Schemas.data + ':')) {
                return href;
            }
            if (!uri) {
                uri = uri_1.URI.parse(href);
            }
            // this URI will end up as "src"-attribute of a dom node
            // and because of that special rewriting needs to be done
            // so that the URI uses a protocol that's understood by
            // browsers (like http or https)
            return network_1.FileAccess.asBrowserUri(uri).toString(true);
        }
        if (!uri) {
            return href;
        }
        if (uri_1.URI.parse(href).toString() === uri.toString()) {
            return href; // no transformation performed
        }
        if (uri.query) {
            uri = uri.with({ query: _uriMassage(uri.query) });
        }
        return uri.toString();
    };
    const renderer = new marked_1.marked.Renderer();
    renderer.image = defaultMarkedRenderers.image;
    renderer.link = defaultMarkedRenderers.link;
    renderer.paragraph = defaultMarkedRenderers.paragraph;
    // Will collect [id, renderedElement] tuples
    const codeBlocks = [];
    if (options.codeBlockRenderer) {
        renderer.code = (code, lang) => {
            const id = idGenerator_1.defaultGenerator.nextId();
            const value = options.codeBlockRenderer(postProcessCodeBlockLanguageId(lang), code);
            codeBlocks.push(value.then(element => [id, element]));
            return `<div class="code" data-code="${id}">${(0, strings_1.escape)(code)}</div>`;
        };
    }
    if (options.actionHandler) {
        const onClick = options.actionHandler.disposables.add(new event_1.DomEmitter(element, 'click'));
        const onAuxClick = options.actionHandler.disposables.add(new event_1.DomEmitter(element, 'auxclick'));
        options.actionHandler.disposables.add(event_2.Event.any(onClick.event, onAuxClick.event)(e => {
            const mouseEvent = new mouseEvent_1.StandardMouseEvent(e);
            if (!mouseEvent.leftButton && !mouseEvent.middleButton) {
                return;
            }
            let target = mouseEvent.target;
            if (target.tagName !== 'A') {
                target = target.parentElement;
                if (!target || target.tagName !== 'A') {
                    return;
                }
            }
            try {
                let href = target.dataset['href'];
                if (href) {
                    if (markdown.baseUri) {
                        href = resolveWithBaseUri(uri_1.URI.from(markdown.baseUri), href);
                    }
                    options.actionHandler.callback(href, mouseEvent);
                }
            }
            catch (err) {
                (0, errors_1.onUnexpectedError)(err);
            }
            finally {
                mouseEvent.preventDefault();
            }
        }));
    }
    if (!markdown.supportHtml) {
        // TODO: Can we deprecated this in favor of 'supportHtml'?
        // Use our own sanitizer so that we can let through only spans.
        // Otherwise, we'd be letting all html be rendered.
        // If we want to allow markdown permitted tags, then we can delete sanitizer and sanitize.
        // We always pass the output through dompurify after this so that we don't rely on
        // marked for sanitization.
        markedOptions.sanitizer = (html) => {
            const match = markdown.isTrusted ? html.match(/^(<span[^>]+>)|(<\/\s*span>)$/) : undefined;
            return match ? html : '';
        };
        markedOptions.sanitize = true;
        markedOptions.silent = true;
    }
    markedOptions.renderer = renderer;
    // values that are too long will freeze the UI
    let value = (_a = markdown.value) !== null && _a !== void 0 ? _a : '';
    if (value.length > 100000) {
        value = `${value.substr(0, 100000)}…`;
    }
    // escape theme icons
    if (markdown.supportThemeIcons) {
        value = (0, iconLabels_2.markdownEscapeEscapedIcons)(value);
    }
    let renderedMarkdown = marked_1.marked.parse(value, markedOptions);
    // Rewrite theme icons
    if (markdown.supportThemeIcons) {
        const elements = (0, iconLabels_1.renderLabelWithIcons)(renderedMarkdown);
        renderedMarkdown = elements.map(e => typeof e === 'string' ? e : e.outerHTML).join('');
    }
    const htmlParser = new DOMParser();
    const markdownHtmlDoc = htmlParser.parseFromString(sanitizeRenderedMarkdown(markdown, renderedMarkdown), 'text/html');
    markdownHtmlDoc.body.querySelectorAll('img')
        .forEach(img => {
        const src = img.getAttribute('src'); // Get the raw 'src' attribute value as text, not the resolved 'src'
        if (src) {
            let href = src;
            try {
                if (markdown.baseUri) { // absolute or relative local path, or file: uri
                    href = resolveWithBaseUri(uri_1.URI.from(markdown.baseUri), href);
                }
            }
            catch (err) { }
            img.src = _href(href, true);
        }
    });
    markdownHtmlDoc.body.querySelectorAll('a')
        .forEach(a => {
        const href = a.getAttribute('href'); // Get the raw 'href' attribute value as text, not the resolved 'href'
        a.setAttribute('href', ''); // Clear out href. We use the `data-href` for handling clicks instead
        if (!href
            || /^data:|javascript:/i.test(href)
            || (/^command:/i.test(href) && !markdown.isTrusted)
            || /^command:(\/\/\/)?_workbench\.downloadResource/i.test(href)) {
            // drop the link
            a.replaceWith(...a.childNodes);
        }
        else {
            let resolvedHref = _href(href, false);
            if (markdown.baseUri) {
                resolvedHref = resolveWithBaseUri(uri_1.URI.from(markdown.baseUri), href);
            }
            a.dataset.href = resolvedHref;
        }
    });
    element.innerHTML = sanitizeRenderedMarkdown(markdown, markdownHtmlDoc.body.innerHTML);
    if (codeBlocks.length > 0) {
        Promise.all(codeBlocks).then((tuples) => {
            var _a, _b;
            if (isDisposed) {
                return;
            }
            const renderedElements = new Map(tuples);
            const placeholderElements = element.querySelectorAll(`div[data-code]`);
            for (const placeholderElement of placeholderElements) {
                const renderedElement = renderedElements.get((_a = placeholderElement.dataset['code']) !== null && _a !== void 0 ? _a : '');
                if (renderedElement) {
                    DOM.reset(placeholderElement, renderedElement);
                }
            }
            (_b = options.asyncRenderCallback) === null || _b === void 0 ? void 0 : _b.call(options);
        });
    }
    // signal size changes for image tags
    if (options.asyncRenderCallback) {
        for (const img of element.getElementsByTagName('img')) {
            const listener = disposables.add(DOM.addDisposableListener(img, 'load', () => {
                listener.dispose();
                options.asyncRenderCallback();
            }));
        }
    }
    return {
        element,
        dispose: () => {
            isDisposed = true;
            disposables.dispose();
        }
    };
}
exports.renderMarkdown = renderMarkdown;
function postProcessCodeBlockLanguageId(lang) {
    if (!lang) {
        return '';
    }
    const parts = lang.split(/[\s+|:|,|\{|\?]/, 1);
    if (parts.length) {
        return parts[0];
    }
    return lang;
}
function resolveWithBaseUri(baseUri, href) {
    const hasScheme = /^\w[\w\d+.-]*:/.test(href);
    if (hasScheme) {
        return href;
    }
    if (baseUri.path.endsWith('/')) {
        return (0, resources_1.resolvePath)(baseUri, href).toString();
    }
    else {
        return (0, resources_1.resolvePath)((0, resources_1.dirname)(baseUri), href).toString();
    }
}
function sanitizeRenderedMarkdown(options, renderedMarkdown) {
    const { config, allowedSchemes } = getSanitizerOptions(options);
    dompurify.addHook('uponSanitizeAttribute', (element, e) => {
        if (e.attrName === 'style' || e.attrName === 'class') {
            if (element.tagName === 'SPAN') {
                if (e.attrName === 'style') {
                    e.keepAttr = /^(color\:#[0-9a-fA-F]+;)?(background-color\:#[0-9a-fA-F]+;)?$/.test(e.attrValue);
                    return;
                }
                else if (e.attrName === 'class') {
                    e.keepAttr = /^codicon codicon-[a-z\-]+( codicon-modifier-[a-z\-]+)?$/.test(e.attrValue);
                    return;
                }
            }
            e.keepAttr = false;
            return;
        }
    });
    const hook = DOM.hookDomPurifyHrefAndSrcSanitizer(allowedSchemes);
    try {
        return dompurify.sanitize(renderedMarkdown, Object.assign(Object.assign({}, config), { RETURN_TRUSTED_TYPE: true }));
    }
    finally {
        dompurify.removeHook('uponSanitizeAttribute');
        hook.dispose();
    }
}
exports.allowedMarkdownAttr = [
    'align',
    'alt',
    'class',
    'controls',
    'data-code',
    'data-href',
    'height',
    'href',
    'loop',
    'muted',
    'playsinline',
    'poster',
    'src',
    'style',
    'target',
    'title',
    'width',
];
function getSanitizerOptions(options) {
    const allowedSchemes = [
        network_1.Schemas.http,
        network_1.Schemas.https,
        network_1.Schemas.mailto,
        network_1.Schemas.data,
        network_1.Schemas.file,
        network_1.Schemas.vscodeFileResource,
        network_1.Schemas.vscodeRemote,
        network_1.Schemas.vscodeRemoteResource,
    ];
    if (options.isTrusted) {
        allowedSchemes.push(network_1.Schemas.command);
    }
    return {
        config: {
            // allowedTags should included everything that markdown renders to.
            // Since we have our own sanitize function for marked, it's possible we missed some tag so let dompurify make sure.
            // HTML tags that can result from markdown are from reading https://spec.commonmark.org/0.29/
            // HTML table tags that can result from markdown are from https://github.github.com/gfm/#tables-extension-
            ALLOWED_TAGS: [...DOM.basicMarkupHtmlTags],
            ALLOWED_ATTR: exports.allowedMarkdownAttr,
            ALLOW_UNKNOWN_PROTOCOLS: true,
        },
        allowedSchemes
    };
}
/**
 * Strips all markdown from `string`, if it's an IMarkdownString. For example
 * `# Header` would be output as `Header`. If it's not, the string is returned.
 */
function renderStringAsPlaintext(string) {
    return typeof string === 'string' ? string : renderMarkdownAsPlaintext(string);
}
exports.renderStringAsPlaintext = renderStringAsPlaintext;
/**
 * Strips all markdown from `markdown`. For example `# Header` would be output as `Header`.
 */
function renderMarkdownAsPlaintext(markdown) {
    var _a;
    // values that are too long will freeze the UI
    let value = (_a = markdown.value) !== null && _a !== void 0 ? _a : '';
    if (value.length > 100000) {
        value = `${value.substr(0, 100000)}…`;
    }
    const html = marked_1.marked.parse(value, { renderer: plainTextRenderer.getValue() }).replace(/&(#\d+|[a-zA-Z]+);/g, m => { var _a; return (_a = unescapeInfo.get(m)) !== null && _a !== void 0 ? _a : m; });
    return sanitizeRenderedMarkdown({ isTrusted: false }, html).toString();
}
exports.renderMarkdownAsPlaintext = renderMarkdownAsPlaintext;
const unescapeInfo = new Map([
    ['&quot;', '"'],
    ['&nbsp;', ' '],
    ['&amp;', '&'],
    ['&#39;', '\''],
    ['&lt;', '<'],
    ['&gt;', '>'],
]);
const plainTextRenderer = new lazy_1.Lazy(() => {
    const renderer = new marked_1.marked.Renderer();
    renderer.code = (code) => {
        return code;
    };
    renderer.blockquote = (quote) => {
        return quote;
    };
    renderer.html = (_html) => {
        return '';
    };
    renderer.heading = (text, _level, _raw) => {
        return text + '\n';
    };
    renderer.hr = () => {
        return '';
    };
    renderer.list = (body, _ordered) => {
        return body;
    };
    renderer.listitem = (text) => {
        return text + '\n';
    };
    renderer.paragraph = (text) => {
        return text + '\n';
    };
    renderer.table = (header, body) => {
        return header + body + '\n';
    };
    renderer.tablerow = (content) => {
        return content;
    };
    renderer.tablecell = (content, _flags) => {
        return content + ' ';
    };
    renderer.strong = (text) => {
        return text;
    };
    renderer.em = (text) => {
        return text;
    };
    renderer.codespan = (code) => {
        return code;
    };
    renderer.br = () => {
        return '\n';
    };
    renderer.del = (text) => {
        return text;
    };
    renderer.image = (_href, _title, _text) => {
        return '';
    };
    renderer.text = (text) => {
        return text;
    };
    renderer.link = (_href, _title, text) => {
        return text;
    };
    return renderer;
});
//# sourceMappingURL=markdownRenderer.js.map