"use strict";
/*********************************************************************
 * Copyright (c) 2019 Arm and others
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *********************************************************************/
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const chai_1 = require("chai");
const utils_1 = require("./utils");
const os = require("os");
describe('breakpoints', function () {
    return __awaiter(this, void 0, void 0, function* () {
        let dc;
        beforeEach(function () {
            return __awaiter(this, void 0, void 0, function* () {
                dc = yield (0, utils_1.standardBeforeEach)();
                yield dc.launchRequest((0, utils_1.fillDefaults)(this.currentTest, {
                    program: path.join(utils_1.testProgramsDir, 'count'),
                }));
            });
        });
        afterEach(() => __awaiter(this, void 0, void 0, function* () {
            yield dc.stop();
        }));
        it('set type of standard breakpoint', () => __awaiter(this, void 0, void 0, function* () {
            const bpResp = yield dc.setBreakpointsRequest({
                source: {
                    name: 'count.c',
                    path: path.join(utils_1.testProgramsDir, 'count.c'),
                },
                breakpoints: [
                    {
                        column: 1,
                        line: 4,
                    },
                ],
            });
            (0, chai_1.expect)(bpResp.body.breakpoints.length).eq(1);
            (0, chai_1.expect)(bpResp.body.breakpoints[0].verified).eq(true);
            (0, chai_1.expect)(bpResp.body.breakpoints[0].message).eq(undefined);
            yield dc.configurationDoneRequest();
            let isCorrect;
            let outputs;
            while (!isCorrect) {
                // Cover the case of getting event in Linux environment.
                // If cannot get correct event, program timeout and test case failed.
                outputs = yield dc.waitForEvent('output', this.timeout());
                isCorrect = outputs.body.output.includes('breakpoint-modified');
            }
            let substring;
            if (utils_1.hardwareBreakpoint) {
                substring = 'type="hw breakpoint"';
            }
            else {
                substring = 'type="breakpoint"';
            }
            (0, chai_1.expect)(outputs === null || outputs === void 0 ? void 0 : outputs.body.output).includes(substring);
        }));
        it('hits a standard breakpoint', () => __awaiter(this, void 0, void 0, function* () {
            const bpResp = yield dc.setBreakpointsRequest({
                source: {
                    name: 'count.c',
                    path: path.join(utils_1.testProgramsDir, 'count.c'),
                },
                breakpoints: [
                    {
                        column: 1,
                        line: 4,
                    },
                ],
            });
            (0, chai_1.expect)(bpResp.body.breakpoints.length).eq(1);
            (0, chai_1.expect)(bpResp.body.breakpoints[0].verified).eq(true);
            (0, chai_1.expect)(bpResp.body.breakpoints[0].message).eq(undefined);
            yield dc.configurationDoneRequest();
            yield dc.waitForEvent('stopped');
            const scope = yield (0, utils_1.getScopes)(dc);
            const vr = scope.scopes.body.scopes[0].variablesReference;
            const vars = yield dc.variablesRequest({ variablesReference: vr });
            (0, utils_1.verifyVariable)(vars.body.variables[0], 'count', 'int', '0');
        }));
        it('can set breakpoints while program is running', function () {
            return __awaiter(this, void 0, void 0, function* () {
                if (os.platform() === 'win32' && (!utils_1.isRemoteTest || !utils_1.gdbAsync)) {
                    // win32 host can only pause remote + mi-async targets
                    this.skip();
                }
                let response = yield dc.setBreakpointsRequest({
                    source: {
                        name: 'count.c',
                        path: path.join(utils_1.testProgramsDir, 'count.c'),
                    },
                    breakpoints: [
                        {
                            column: 1,
                            line: 2,
                        },
                    ],
                });
                (0, chai_1.expect)(response.body.breakpoints.length).to.eq(1);
                yield dc.configurationDoneRequest();
                yield dc.waitForEvent('stopped');
                const scope = yield (0, utils_1.getScopes)(dc);
                yield dc.continueRequest({ threadId: scope.thread.id });
                // start listening for stopped events before we issue the
                // setBreakpointsRequest to ensure we don't get extra
                // stopped events
                const stoppedEventWaitor = dc.waitForEvent('stopped');
                response = yield dc.setBreakpointsRequest({
                    source: {
                        name: 'count.c',
                        path: path.join(utils_1.testProgramsDir, 'count.c'),
                    },
                    breakpoints: [
                        {
                            column: 1,
                            line: 4,
                        },
                    ],
                });
                (0, chai_1.expect)(response.body.breakpoints.length).to.eq(1);
                yield dc.assertStoppedLocation('breakpoint', { line: 4 });
                const stoppedEvent = yield stoppedEventWaitor;
                (0, chai_1.expect)(stoppedEvent).to.have.property('body');
                (0, chai_1.expect)(stoppedEvent.body).to.have.property('reason', 'breakpoint');
            });
        });
        it('handles breakpoints in multiple files', () => __awaiter(this, void 0, void 0, function* () {
            yield dc.setBreakpointsRequest({
                source: {
                    name: 'count.c',
                    path: path.join(utils_1.testProgramsDir, 'count.c'),
                },
                breakpoints: [
                    {
                        column: 1,
                        line: 4,
                    },
                ],
            });
            yield dc.setBreakpointsRequest({
                source: {
                    name: 'count_other.c',
                    path: path.join(utils_1.testProgramsDir, 'count_other.c'),
                },
                breakpoints: [
                    {
                        column: 1,
                        line: 2,
                    },
                ],
            });
            yield dc.configurationDoneRequest();
            yield dc.waitForEvent('stopped');
            const scope = yield (0, utils_1.getScopes)(dc);
            const vr = scope.scopes.body.scopes[0].variablesReference;
            const vars = yield dc.variablesRequest({ variablesReference: vr });
            (0, utils_1.verifyVariable)(vars.body.variables[0], 'count', 'int', '0');
        }));
        it('fails gracefully on breakpoint on unknown file', () => __awaiter(this, void 0, void 0, function* () {
            const bpResp = yield dc.setBreakpointsRequest({
                source: {
                    name: 'countBAD.c',
                    path: path.join(utils_1.testProgramsDir, 'countBAD.c'),
                },
                breakpoints: [
                    {
                        column: 1,
                        line: 4,
                    },
                ],
            });
            (0, chai_1.expect)(bpResp.body.breakpoints.length).to.eq(1);
            (0, chai_1.expect)(bpResp.body.breakpoints[0].verified).to.eq(false);
            (0, chai_1.expect)(bpResp.body.breakpoints[0].message).not.eq(undefined);
        }));
        it('fails gracefully on breakpoint on bad line in otherwise good source', () => __awaiter(this, void 0, void 0, function* () {
            const bpResp = yield dc.setBreakpointsRequest({
                source: {
                    name: 'count.c',
                    path: path.join(utils_1.testProgramsDir, 'count.c'),
                },
                breakpoints: [
                    {
                        column: 1,
                        line: 4 + 100000000,
                    },
                ],
            });
            (0, chai_1.expect)(bpResp.body.breakpoints.length).to.eq(1);
            (0, chai_1.expect)(bpResp.body.breakpoints[0].verified).to.eq(false);
            (0, chai_1.expect)(bpResp.body.breakpoints[0].message).not.eq(undefined);
        }));
        it('maintains breakpoint order when modifying breakpoints in a file', () => __awaiter(this, void 0, void 0, function* () {
            const bpResp1 = yield dc.setBreakpointsRequest({
                source: {
                    name: 'count.c',
                    path: path.join(utils_1.testProgramsDir, 'count.c'),
                },
                breakpoints: [
                    {
                        column: 1,
                        line: 6,
                    },
                ],
            });
            (0, chai_1.expect)(bpResp1.body.breakpoints.length).to.eq(1);
            (0, chai_1.expect)(bpResp1.body.breakpoints[0].line).eq(6);
            const bpResp2 = yield dc.setBreakpointsRequest({
                source: {
                    name: 'count.c',
                    path: path.join(utils_1.testProgramsDir, 'count.c'),
                },
                breakpoints: [
                    {
                        column: 1,
                        line: 4,
                    },
                    {
                        column: 1,
                        line: 6,
                    },
                ],
            });
            (0, chai_1.expect)(bpResp2.body.breakpoints.length).to.eq(2);
            (0, chai_1.expect)(bpResp2.body.breakpoints[0].line).eq(4);
            (0, chai_1.expect)(bpResp2.body.breakpoints[1].line).eq(6);
            // Make sure the GDB id of the breakpoint on line 6 is unchanged
            (0, chai_1.expect)(bpResp2.body.breakpoints[1].id).eq(bpResp1.body.breakpoints[0].id);
        }));
        it('maintains breakpoint order when modifying breakpoints in a file with space', () => __awaiter(this, void 0, void 0, function* () {
            const bpResp1 = yield dc.setBreakpointsRequest({
                source: {
                    name: 'count space.c',
                    path: path.join(utils_1.testProgramsDir, 'count space.c'),
                },
                breakpoints: [
                    {
                        column: 1,
                        line: 5,
                    },
                ],
            });
            (0, chai_1.expect)(bpResp1.body.breakpoints.length).to.eq(1);
            (0, chai_1.expect)(bpResp1.body.breakpoints[0].line).eq(5);
            const bpResp2 = yield dc.setBreakpointsRequest({
                source: {
                    name: 'count space.c',
                    path: path.join(utils_1.testProgramsDir, 'count space.c'),
                },
                breakpoints: [
                    {
                        column: 1,
                        line: 2,
                    },
                    {
                        column: 1,
                        line: 5,
                    },
                ],
            });
            (0, chai_1.expect)(bpResp2.body.breakpoints.length).to.eq(2);
            (0, chai_1.expect)(bpResp2.body.breakpoints[0].line).eq(2);
            (0, chai_1.expect)(bpResp2.body.breakpoints[1].line).eq(5);
            // Make sure the GDB id of the breakpoint on line 5 is unchanged
            (0, chai_1.expect)(bpResp2.body.breakpoints[1].id).eq(bpResp1.body.breakpoints[0].id);
        }));
        it('reports back relocated line number', function () {
            return __awaiter(this, void 0, void 0, function* () {
                if (!(yield (0, utils_1.gdbVersionAtLeast)('8.2'))) {
                    this.skip();
                }
                const args = {
                    source: {
                        name: 'count.c',
                        path: path.join(utils_1.testProgramsDir, 'count.c'),
                    },
                    breakpoints: [
                        {
                            column: 1,
                            line: 5, // this will be relocated to line 6 as no code on line 5
                        },
                    ],
                };
                const bpResp = yield dc.setBreakpointsRequest(args);
                // Make sure the GDB id of the breakpoint is unchanged
                (0, chai_1.expect)(bpResp.body.breakpoints[0].line).eq(6);
            });
        });
        it('maintains gdb breakpoint when relocated', function () {
            return __awaiter(this, void 0, void 0, function* () {
                if (!(yield (0, utils_1.gdbVersionAtLeast)('8.2'))) {
                    this.skip();
                }
                const args = {
                    source: {
                        name: 'count.c',
                        path: path.join(utils_1.testProgramsDir, 'count.c'),
                    },
                    breakpoints: [
                        {
                            column: 1,
                            line: 5, // this will be relocated to line 6 as no code on line 5
                        },
                    ],
                };
                const bpResp1 = yield dc.setBreakpointsRequest(args);
                (0, chai_1.expect)(bpResp1.body.breakpoints.length).to.eq(1);
                (0, chai_1.expect)(bpResp1.body.breakpoints[0].line).eq(6);
                const bpResp2 = yield dc.setBreakpointsRequest(args);
                (0, chai_1.expect)(bpResp2.body.breakpoints.length).to.eq(1);
                (0, chai_1.expect)(bpResp2.body.breakpoints[0].line).eq(6);
                // Make sure the GDB id of the breakpoint is unchanged
                (0, chai_1.expect)(bpResp2.body.breakpoints[0].id).eq(bpResp1.body.breakpoints[0].id);
            });
        });
        it('maintains gdb breakpoint when relocated - files with spaces', function () {
            return __awaiter(this, void 0, void 0, function* () {
                if (!(yield (0, utils_1.gdbVersionAtLeast)('8.2'))) {
                    this.skip();
                }
                const args = {
                    source: {
                        name: 'count space.c',
                        path: path.join(utils_1.testProgramsDir, 'count space.c'),
                    },
                    breakpoints: [
                        {
                            column: 1,
                            line: 7, // this will be relocated to line 9 as no code on line 7
                        },
                    ],
                };
                const bpResp1 = yield dc.setBreakpointsRequest(args);
                (0, chai_1.expect)(bpResp1.body.breakpoints.length).to.eq(1);
                (0, chai_1.expect)(bpResp1.body.breakpoints[0].line).eq(9);
                const bpResp2 = yield dc.setBreakpointsRequest(args);
                (0, chai_1.expect)(bpResp2.body.breakpoints.length).to.eq(1);
                (0, chai_1.expect)(bpResp2.body.breakpoints[0].line).eq(9);
                // Make sure the GDB id of the breakpoint is unchanged
                (0, chai_1.expect)(bpResp2.body.breakpoints[0].id).eq(bpResp1.body.breakpoints[0].id);
            });
        });
        it('hits a conditional breakpoint', () => __awaiter(this, void 0, void 0, function* () {
            yield dc.setBreakpointsRequest({
                source: {
                    name: 'count.c',
                    path: path.join(utils_1.testProgramsDir, 'count.c'),
                },
                breakpoints: [
                    {
                        column: 1,
                        line: 4,
                        condition: 'count == 5',
                    },
                ],
            });
            yield dc.configurationDoneRequest();
            yield dc.waitForEvent('stopped');
            const scope = yield (0, utils_1.getScopes)(dc);
            const vr = scope.scopes.body.scopes[0].variablesReference;
            const vars = yield dc.variablesRequest({ variablesReference: vr });
            (0, utils_1.verifyVariable)(vars.body.variables[0], 'count', 'int', '5');
        }));
        it('hits a hit conditional breakpoint with >', () => __awaiter(this, void 0, void 0, function* () {
            yield dc.setBreakpointsRequest({
                source: {
                    name: 'count.c',
                    path: path.join(utils_1.testProgramsDir, 'count.c'),
                },
                breakpoints: [
                    {
                        column: 1,
                        line: 4,
                        hitCondition: '> 5',
                    },
                ],
            });
            yield dc.configurationDoneRequest();
            yield dc.waitForEvent('stopped');
            const scope = yield (0, utils_1.getScopes)(dc);
            const vr = scope.scopes.body.scopes[0].variablesReference;
            const vars = yield dc.variablesRequest({ variablesReference: vr });
            (0, utils_1.verifyVariable)(vars.body.variables[0], 'count', 'int', '5');
        }));
        it('hits a hit conditional breakpoint without >', () => __awaiter(this, void 0, void 0, function* () {
            yield dc.setBreakpointsRequest({
                source: {
                    name: 'count.c',
                    path: path.join(utils_1.testProgramsDir, 'count.c'),
                },
                breakpoints: [
                    {
                        column: 1,
                        line: 4,
                        hitCondition: '5',
                    },
                ],
            });
            yield dc.configurationDoneRequest();
            yield dc.waitForEvent('stopped');
            const scope = yield (0, utils_1.getScopes)(dc);
            const vr = scope.scopes.body.scopes[0].variablesReference;
            const vars = yield dc.variablesRequest({ variablesReference: vr });
            (0, utils_1.verifyVariable)(vars.body.variables[0], 'count', 'int', '4');
        }));
        it('resolves breakpoints', () => __awaiter(this, void 0, void 0, function* () {
            let response = yield dc.setBreakpointsRequest({
                source: {
                    name: 'count.c',
                    path: path.join(utils_1.testProgramsDir, 'count.c'),
                },
                breakpoints: [
                    {
                        column: 1,
                        line: 2,
                    },
                ],
            });
            (0, chai_1.expect)(response.body.breakpoints.length).to.eq(1);
            yield dc.configurationDoneRequest();
            yield dc.waitForEvent('stopped');
            response = yield dc.setBreakpointsRequest({
                source: {
                    name: 'count.c',
                    path: path.join(utils_1.testProgramsDir, 'count.c'),
                },
                breakpoints: [
                    {
                        column: 1,
                        line: 2,
                    },
                    {
                        column: 1,
                        line: 3,
                    },
                ],
            });
            (0, chai_1.expect)(response.body.breakpoints.length).to.eq(2);
            response = yield dc.setBreakpointsRequest({
                source: {
                    name: 'count.c',
                    path: path.join(utils_1.testProgramsDir, 'count.c'),
                },
                breakpoints: [
                    {
                        column: 1,
                        line: 2,
                        condition: 'count == 5',
                    },
                    {
                        column: 1,
                        line: 3,
                    },
                ],
            });
            (0, chai_1.expect)(response.body.breakpoints.length).to.eq(2);
            response = yield dc.setBreakpointsRequest({
                source: {
                    name: 'count.c',
                    path: path.join(utils_1.testProgramsDir, 'count.c'),
                },
                breakpoints: [
                    {
                        column: 1,
                        line: 2,
                        condition: 'count == 3',
                    },
                ],
            });
            (0, chai_1.expect)(response.body.breakpoints.length).to.eq(1);
        }));
    });
});
//# sourceMappingURL=breakpoints.spec.js.map