"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*-----------------------------------------------------------------------------
| Copyright (c) 2014-2017, PhosphorJS Contributors
|
| Distributed under the terms of the BSD 3-Clause License.
|
| The full license is in the file LICENSE, distributed with this software.
|----------------------------------------------------------------------------*/
var iter_1 = require("./iter");
/**
 * Enumerate an iterable object.
 *
 * @param object - The iterable or array-like object of interest.
 *
 * @param start - The starting enum value. The default is `0`.
 *
 * @returns An iterator which yields the enumerated values.
 *
 * #### Example
 * ```typescript
 * import { enumerate, toArray } from '@phosphor/algorithm';
 *
 * let data = ['foo', 'bar', 'baz'];
 *
 * let stream = enumerate(data, 1);
 *
 * toArray(stream);  // [[1, 'foo'], [2, 'bar'], [3, 'baz']]
 * ```
 */
function enumerate(object, start) {
    if (start === void 0) { start = 0; }
    return new EnumerateIterator(iter_1.iter(object), start);
}
exports.enumerate = enumerate;
/**
 * An iterator which enumerates the source values.
 */
var EnumerateIterator = /** @class */ (function () {
    /**
     * Construct a new enumerate iterator.
     *
     * @param source - The iterator of values of interest.
     *
     * @param start - The starting enum value.
     */
    function EnumerateIterator(source, start) {
        this._source = source;
        this._index = start;
    }
    /**
     * Get an iterator over the object's values.
     *
     * @returns An iterator which yields the object's values.
     */
    EnumerateIterator.prototype.iter = function () {
        return this;
    };
    /**
     * Create an independent clone of the iterator.
     *
     * @returns A new independent clone of the iterator.
     */
    EnumerateIterator.prototype.clone = function () {
        return new EnumerateIterator(this._source.clone(), this._index);
    };
    /**
     * Get the next value from the iterator.
     *
     * @returns The next value from the iterator, or `undefined`.
     */
    EnumerateIterator.prototype.next = function () {
        var value = this._source.next();
        if (value === undefined) {
            return undefined;
        }
        return [this._index++, value];
    };
    return EnumerateIterator;
}());
exports.EnumerateIterator = EnumerateIterator;
