"use strict";
// *****************************************************************************
// Copyright (C) 2022 Red Hat, Inc. and others.
//
// This program and the accompanying materials are made available under the
// terms of the Eclipse Public License v. 2.0 which is available at
// http://www.eclipse.org/legal/epl-2.0.
//
// This Source Code may also be made available under the following Secondary
// Licenses when the conditions for such availability set forth in the Eclipse
// Public License v. 2.0 are satisfied: GNU General Public License, version 2
// with the GNU Classpath Exception which is available at
// https://www.gnu.org/software/classpath/license.html.
//
// SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
// *****************************************************************************
/* eslint-disable @typescript-eslint/no-explicit-any */
Object.defineProperty(exports, "__esModule", { value: true });
exports.RpcMessageEncoder = exports.RpcMessageDecoder = exports.EncodingError = exports.ObjectType = exports.ResponseError = void 0;
/**
 * A special error that can be returned in case a request
 * has failed. Provides additional information i.e. an error code
 * and additional error data.
 */
class ResponseError extends Error {
    constructor(code, message, data) {
        super(message);
        this.code = code;
        this.data = data;
    }
}
exports.ResponseError = ResponseError;
/**
 * The tag values for the default {@link ValueEncoder}s & {@link ValueDecoder}s
 */
var ObjectType;
(function (ObjectType) {
    ObjectType[ObjectType["JSON"] = 0] = "JSON";
    ObjectType[ObjectType["ByteArray"] = 1] = "ByteArray";
    ObjectType[ObjectType["ObjectArray"] = 2] = "ObjectArray";
    ObjectType[ObjectType["Undefined"] = 3] = "Undefined";
    ObjectType[ObjectType["Object"] = 4] = "Object";
    ObjectType[ObjectType["String"] = 5] = "String";
    ObjectType[ObjectType["Boolean"] = 6] = "Boolean";
    ObjectType[ObjectType["Number"] = 7] = "Number";
    // eslint-disable-next-line @typescript-eslint/no-shadow
    ObjectType[ObjectType["ResponseError"] = 8] = "ResponseError";
    ObjectType[ObjectType["Error"] = 9] = "Error";
    ObjectType[ObjectType["Map"] = 10] = "Map";
    ObjectType[ObjectType["Set"] = 11] = "Set";
    ObjectType[ObjectType["Function"] = 12] = "Function";
})(ObjectType = exports.ObjectType || (exports.ObjectType = {}));
/**
 * Custom error thrown by the {@link RpcMessageEncoder} if an error occurred during the encoding and the
 * object could not be written to the given {@link WriteBuffer}
 */
class EncodingError extends Error {
    constructor(msg) {
        super(msg);
    }
}
exports.EncodingError = EncodingError;
/**
 * A `RpcMessageDecoder` parses a a binary message received via {@link ReadBuffer} into a {@link RpcMessage}
 */
class RpcMessageDecoder {
    constructor() {
        this.decoders = new Map();
        this.registerDecoders();
    }
    registerDecoders() {
        this.registerDecoder(ObjectType.JSON, {
            read: buf => {
                const json = buf.readString();
                return JSON.parse(json);
            }
        });
        this.registerDecoder(ObjectType.Error, {
            read: buf => {
                const serializedError = JSON.parse(buf.readString());
                const error = new Error(serializedError.message);
                Object.assign(error, serializedError);
                return error;
            }
        });
        this.registerDecoder(ObjectType.ResponseError, {
            read: buf => {
                const error = JSON.parse(buf.readString());
                return new ResponseError(error.code, error.message, error.data);
            }
        });
        this.registerDecoder(ObjectType.ByteArray, {
            read: buf => buf.readBytes()
        });
        this.registerDecoder(ObjectType.ObjectArray, {
            read: buf => this.readArray(buf)
        });
        this.registerDecoder(ObjectType.Undefined, {
            read: () => undefined
        });
        this.registerDecoder(ObjectType.Object, {
            read: (buf, recursiveRead) => {
                const propertyCount = buf.readLength();
                const result = Object.create({});
                for (let i = 0; i < propertyCount; i++) {
                    const key = buf.readString();
                    const value = recursiveRead(buf);
                    result[key] = value;
                }
                return result;
            }
        });
        this.registerDecoder(ObjectType.String, {
            read: (buf, recursiveRead) => buf.readString()
        });
        this.registerDecoder(ObjectType.Boolean, {
            read: buf => buf.readUint8() === 1
        });
        this.registerDecoder(ObjectType.Number, {
            read: buf => buf.readNumber()
        });
        this.registerDecoder(ObjectType.Map, {
            read: buf => new Map(this.readArray(buf))
        });
        this.registerDecoder(ObjectType.Set, {
            read: buf => new Set(this.readArray(buf))
        });
        this.registerDecoder(ObjectType.Function, {
            read: () => ({})
        });
    }
    /**
     * Registers a new {@link ValueDecoder} for the given tag.
     * After the successful registration the {@link tagIntType} is recomputed
     * by retrieving the highest tag value and calculating the required Uint size to store it.
     * @param tag the tag for which the decoder should be registered.
     * @param decoder the decoder that should be registered.
     */
    registerDecoder(tag, decoder) {
        if (this.decoders.has(tag)) {
            throw new Error(`Decoder already registered: ${tag}`);
        }
        this.decoders.set(tag, decoder);
    }
    parse(buf) {
        try {
            const msgType = buf.readUint8();
            switch (msgType) {
                case 1 /* Request */:
                    return this.parseRequest(buf);
                case 2 /* Notification */:
                    return this.parseNotification(buf);
                case 3 /* Reply */:
                    return this.parseReply(buf);
                case 4 /* ReplyErr */:
                    return this.parseReplyErr(buf);
                case 5 /* Cancel */:
                    return this.parseCancel(buf);
            }
            throw new Error(`Unknown message type: ${msgType}`);
        }
        catch (e) {
            // exception does not show problematic content: log it!
            console.log('failed to parse message: ' + buf);
            throw e;
        }
    }
    parseCancel(msg) {
        const callId = msg.readUint32();
        return {
            type: 5 /* Cancel */,
            id: callId
        };
    }
    parseRequest(msg) {
        const callId = msg.readUint32();
        const method = msg.readString();
        const args = this.readArray(msg);
        return {
            type: 1 /* Request */,
            id: callId,
            method: method,
            args: args
        };
    }
    parseNotification(msg) {
        const callId = msg.readUint32();
        const method = msg.readString();
        const args = this.readArray(msg);
        return {
            type: 2 /* Notification */,
            id: callId,
            method: method,
            args: args
        };
    }
    parseReply(msg) {
        const callId = msg.readUint32();
        const value = this.readTypedValue(msg);
        return {
            type: 3 /* Reply */,
            id: callId,
            res: value
        };
    }
    parseReplyErr(msg) {
        const callId = msg.readUint32();
        const err = this.readTypedValue(msg);
        return {
            type: 4 /* ReplyErr */,
            id: callId,
            err
        };
    }
    readArray(buf) {
        const length = buf.readLength();
        const result = new Array(length);
        for (let i = 0; i < length; i++) {
            result[i] = this.readTypedValue(buf);
        }
        return result;
    }
    readTypedValue(buf) {
        const type = buf.readUint8();
        const decoder = this.decoders.get(type);
        if (!decoder) {
            throw new Error(`No decoder for tag ${type}`);
        }
        return decoder.read(buf, innerBuffer => this.readTypedValue(innerBuffer));
    }
}
exports.RpcMessageDecoder = RpcMessageDecoder;
/**
 * A `RpcMessageEncoder` writes {@link RpcMessage} objects to a {@link WriteBuffer}. Note that it is
 * up to clients to commit the message. This allows for multiple messages being
 * encoded before sending.
 */
class RpcMessageEncoder {
    constructor() {
        this.encoders = [];
        this.registeredTags = new Set();
        this.registerEncoders();
    }
    registerEncoders() {
        // encoders will be consulted in reverse order of registration, so the JSON fallback needs to be last
        this.registerEncoder(ObjectType.JSON, {
            is: () => true,
            write: (buf, value) => {
                buf.writeString(JSON.stringify(value));
            }
        });
        this.registerEncoder(ObjectType.Function, {
            is: value => typeof value === 'function',
            write: () => { }
        });
        this.registerEncoder(ObjectType.Object, {
            is: value => typeof value === 'object',
            write: (buf, object, visitedObjects, recursiveEncode) => {
                const properties = Object.keys(object);
                const relevant = [];
                for (const property of properties) {
                    const value = object[property];
                    if (typeof value !== 'function') {
                        relevant.push([property, value]);
                    }
                }
                buf.writeLength(relevant.length);
                for (const [property, value] of relevant) {
                    buf.writeString(property);
                    recursiveEncode === null || recursiveEncode === void 0 ? void 0 : recursiveEncode(buf, value, visitedObjects);
                }
            }
        });
        this.registerEncoder(ObjectType.Error, {
            is: value => value instanceof Error,
            write: (buf, error) => {
                const { name, message } = error;
                const stack = error.stacktrace || error.stack;
                const serializedError = {
                    $isError: true,
                    name,
                    message,
                    stack
                };
                buf.writeString(JSON.stringify(serializedError));
            }
        });
        this.registerEncoder(ObjectType.ResponseError, {
            is: value => value instanceof ResponseError,
            write: (buf, value) => buf.writeString(JSON.stringify(value))
        });
        this.registerEncoder(ObjectType.Map, {
            is: value => value instanceof Map,
            write: (buf, value, visitedObjects) => this.writeArray(buf, Array.from(value.entries()), visitedObjects)
        });
        this.registerEncoder(ObjectType.Set, {
            is: value => value instanceof Set,
            write: (buf, value, visitedObjects) => this.writeArray(buf, [...value], visitedObjects)
        });
        this.registerEncoder(ObjectType.Undefined, {
            // eslint-disable-next-line no-null/no-null
            is: value => value == null,
            write: () => { }
        });
        this.registerEncoder(ObjectType.ObjectArray, {
            is: value => Array.isArray(value),
            write: (buf, value, visitedObjects) => {
                this.writeArray(buf, value, visitedObjects);
            }
        });
        this.registerEncoder(ObjectType.ByteArray, {
            is: value => value instanceof Uint8Array,
            write: (buf, value) => {
                buf.writeBytes(value);
            }
        });
        this.registerEncoder(ObjectType.String, {
            is: value => typeof value === 'string',
            write: (buf, value) => {
                buf.writeString(value);
            }
        });
        this.registerEncoder(ObjectType.Boolean, {
            is: value => typeof value === 'boolean',
            write: (buf, value) => {
                buf.writeUint8(value === true ? 1 : 0);
            }
        });
        this.registerEncoder(ObjectType.Number, {
            is: value => typeof value === 'number',
            write: (buf, value) => {
                buf.writeNumber(value);
            }
        });
    }
    /**
     * Registers a new {@link ValueEncoder} for the given tag.
     * After the successful registration the {@link tagIntType} is recomputed
     * by retrieving the highest tag value and calculating the required Uint size to store it.
     * @param tag the tag for which the encoder should be registered.
     * @param decoder the encoder that should be registered.
     */
    registerEncoder(tag, encoder) {
        if (this.registeredTags.has(tag)) {
            throw new Error(`Tag already registered: ${tag}`);
        }
        this.registeredTags.add(tag);
        this.encoders.push([tag, encoder]);
    }
    cancel(buf, requestId) {
        buf.writeUint8(5 /* Cancel */);
        buf.writeUint32(requestId);
    }
    notification(buf, requestId, method, args) {
        buf.writeUint8(2 /* Notification */);
        buf.writeUint32(requestId);
        buf.writeString(method);
        this.writeArray(buf, args, new WeakSet());
    }
    request(buf, requestId, method, args) {
        buf.writeUint8(1 /* Request */);
        buf.writeUint32(requestId);
        buf.writeString(method);
        this.writeArray(buf, args, new WeakSet());
    }
    replyOK(buf, requestId, res) {
        buf.writeUint8(3 /* Reply */);
        buf.writeUint32(requestId);
        this.writeTypedValue(buf, res, new WeakSet());
    }
    replyErr(buf, requestId, err) {
        buf.writeUint8(4 /* ReplyErr */);
        buf.writeUint32(requestId);
        this.writeTypedValue(buf, err, new WeakSet());
    }
    writeTypedValue(buf, value, visitedObjects) {
        if (value && typeof value === 'object') {
            if (visitedObjects.has(value)) {
                throw new EncodingError('Object to encode contains circular references!');
            }
            visitedObjects.add(value);
        }
        try {
            for (let i = this.encoders.length - 1; i >= 0; i--) {
                if (this.encoders[i][1].is(value)) {
                    buf.writeUint8(this.encoders[i][0]);
                    this.encoders[i][1].write(buf, value, visitedObjects, (innerBuffer, innerValue, _visitedObjects) => {
                        this.writeTypedValue(innerBuffer, innerValue, _visitedObjects);
                    });
                    return;
                }
            }
            throw new EncodingError(`No suitable value encoder found for ${value}`);
        }
        finally {
            if (value && typeof value === 'object') {
                visitedObjects.delete(value);
            }
        }
    }
    writeArray(buf, value, visitedObjects) {
        buf.writeLength(value.length);
        for (let i = 0; i < value.length; i++) {
            this.writeTypedValue(buf, value[i], visitedObjects);
        }
    }
}
exports.RpcMessageEncoder = RpcMessageEncoder;
//# sourceMappingURL=rpc-message-encoder.js.map