"use strict";
/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
Object.defineProperty(exports, "__esModule", { value: true });
exports.Extensions = exports.KeybindingsRegistry = void 0;
const keybindings_1 = require("../../../base/common/keybindings");
const platform_1 = require("../../../base/common/platform");
const commands_1 = require("../../commands/common/commands");
const platform_2 = require("../../registry/common/platform");
class KeybindingsRegistryImpl {
    constructor() {
        this._coreKeybindings = [];
        this._extensionKeybindings = [];
        this._cachedMergedKeybindings = null;
    }
    /**
     * Take current platform into account and reduce to primary & secondary.
     */
    static bindToCurrentPlatform(kb) {
        if (platform_1.OS === 1 /* Windows */) {
            if (kb && kb.win) {
                return kb.win;
            }
        }
        else if (platform_1.OS === 2 /* Macintosh */) {
            if (kb && kb.mac) {
                return kb.mac;
            }
        }
        else {
            if (kb && kb.linux) {
                return kb.linux;
            }
        }
        return kb;
    }
    registerKeybindingRule(rule) {
        const actualKb = KeybindingsRegistryImpl.bindToCurrentPlatform(rule);
        if (actualKb && actualKb.primary) {
            const kk = (0, keybindings_1.createKeybinding)(actualKb.primary, platform_1.OS);
            if (kk) {
                this._registerDefaultKeybinding(kk, rule.id, rule.args, rule.weight, 0, rule.when);
            }
        }
        if (actualKb && Array.isArray(actualKb.secondary)) {
            for (let i = 0, len = actualKb.secondary.length; i < len; i++) {
                const k = actualKb.secondary[i];
                const kk = (0, keybindings_1.createKeybinding)(k, platform_1.OS);
                if (kk) {
                    this._registerDefaultKeybinding(kk, rule.id, rule.args, rule.weight, -i - 1, rule.when);
                }
            }
        }
    }
    setExtensionKeybindings(rules) {
        let result = [], keybindingsLen = 0;
        for (const rule of rules) {
            if (rule.keybinding.length > 0) {
                result[keybindingsLen++] = {
                    keybinding: rule.keybinding,
                    command: rule.id,
                    commandArgs: rule.args,
                    when: rule.when,
                    weight1: rule.weight,
                    weight2: 0,
                    extensionId: rule.extensionId || null,
                    isBuiltinExtension: rule.isBuiltinExtension || false
                };
            }
        }
        this._extensionKeybindings = result;
        this._cachedMergedKeybindings = null;
    }
    registerCommandAndKeybindingRule(desc) {
        this.registerKeybindingRule(desc);
        commands_1.CommandsRegistry.registerCommand(desc);
    }
    static _mightProduceChar(keyCode) {
        if (keyCode >= 21 /* Digit0 */ && keyCode <= 30 /* Digit9 */) {
            return true;
        }
        if (keyCode >= 31 /* KeyA */ && keyCode <= 56 /* KeyZ */) {
            return true;
        }
        return (keyCode === 80 /* Semicolon */
            || keyCode === 81 /* Equal */
            || keyCode === 82 /* Comma */
            || keyCode === 83 /* Minus */
            || keyCode === 84 /* Period */
            || keyCode === 85 /* Slash */
            || keyCode === 86 /* Backquote */
            || keyCode === 110 /* ABNT_C1 */
            || keyCode === 111 /* ABNT_C2 */
            || keyCode === 87 /* BracketLeft */
            || keyCode === 88 /* Backslash */
            || keyCode === 89 /* BracketRight */
            || keyCode === 90 /* Quote */
            || keyCode === 91 /* OEM_8 */
            || keyCode === 92 /* IntlBackslash */);
    }
    _assertNoCtrlAlt(keybinding, commandId) {
        if (keybinding.ctrlKey && keybinding.altKey && !keybinding.metaKey) {
            if (KeybindingsRegistryImpl._mightProduceChar(keybinding.keyCode)) {
                console.warn('Ctrl+Alt+ keybindings should not be used by default under Windows. Offender: ', keybinding, ' for ', commandId);
            }
        }
    }
    _registerDefaultKeybinding(keybinding, commandId, commandArgs, weight1, weight2, when) {
        if (platform_1.OS === 1 /* Windows */) {
            this._assertNoCtrlAlt(keybinding.parts[0], commandId);
        }
        this._coreKeybindings.push({
            keybinding: keybinding.parts,
            command: commandId,
            commandArgs: commandArgs,
            when: when,
            weight1: weight1,
            weight2: weight2,
            extensionId: null,
            isBuiltinExtension: false
        });
        this._cachedMergedKeybindings = null;
    }
    getDefaultKeybindings() {
        if (!this._cachedMergedKeybindings) {
            this._cachedMergedKeybindings = [].concat(this._coreKeybindings).concat(this._extensionKeybindings);
            this._cachedMergedKeybindings.sort(sorter);
        }
        return this._cachedMergedKeybindings.slice(0);
    }
}
exports.KeybindingsRegistry = new KeybindingsRegistryImpl();
// Define extension point ids
exports.Extensions = {
    EditorModes: 'platform.keybindingsRegistry'
};
platform_2.Registry.add(exports.Extensions.EditorModes, exports.KeybindingsRegistry);
function sorter(a, b) {
    if (a.weight1 !== b.weight1) {
        return a.weight1 - b.weight1;
    }
    if (a.command < b.command) {
        return -1;
    }
    if (a.command > b.command) {
        return 1;
    }
    return a.weight2 - b.weight2;
}
//# sourceMappingURL=keybindingsRegistry.js.map