"use strict";
// *****************************************************************************
// Copyright (C) 2018 TypeFox and others.
//
// This program and the accompanying materials are made available under the
// terms of the Eclipse Public License v. 2.0 which is available at
// http://www.eclipse.org/legal/epl-2.0.
//
// This Source Code may also be made available under the following Secondary
// Licenses when the conditions for such availability set forth in the Eclipse
// Public License v. 2.0 are satisfied: GNU General Public License, version 2
// with the GNU Classpath Exception which is available at
// https://www.gnu.org/software/classpath/license.html.
//
// SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
// *****************************************************************************
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
var MessagingContribution_1;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MessagingContribution = exports.MessagingContainer = void 0;
const socket_io_1 = require("socket.io");
const inversify_1 = require("inversify");
const connection_1 = require("vscode-ws-jsonrpc/lib/socket/connection");
const launch = require("vscode-ws-jsonrpc/lib/server/launch");
const common_1 = require("../../common");
const web_socket_channel_1 = require("../../common/messaging/web-socket-channel");
const messaging_service_1 = require("./messaging-service");
const logger_1 = require("./logger");
const connection_container_module_1 = require("./connection-container-module");
const Route = require("route-parser");
const ws_request_validators_1 = require("../ws-request-validators");
const messaging_listeners_1 = require("./messaging-listeners");
exports.MessagingContainer = Symbol('MessagingContainer');
let MessagingContribution = MessagingContribution_1 = class MessagingContribution {
    constructor() {
        this.wsHandlers = new MessagingContribution_1.ConnectionHandlers();
        this.channelHandlers = new MessagingContribution_1.ConnectionHandlers();
        this.checkAliveTimeout = 30000; // 30 seconds
        this.maxHttpBufferSize = 1e8; // 100 MB
    }
    init() {
        this.ws(web_socket_channel_1.WebSocketChannel.wsPath, (_, socket) => this.handleChannels(socket));
        for (const contribution of this.contributions.getContributions()) {
            contribution.configure(this);
        }
    }
    listen(spec, callback) {
        this.wsChannel(spec, (params, channel) => {
            const connection = (0, connection_1.createWebSocketConnection)(channel, new logger_1.ConsoleLogger());
            callback(params, connection);
        });
    }
    forward(spec, callback) {
        this.wsChannel(spec, (params, channel) => {
            const connection = launch.createWebSocketConnection(channel);
            callback(params, messaging_service_1.WebSocketChannelConnection.create(connection, channel));
        });
    }
    wsChannel(spec, callback) {
        this.channelHandlers.push(spec, (params, channel) => callback(params, channel));
    }
    ws(spec, callback) {
        this.wsHandlers.push(spec, callback);
    }
    onStart(server) {
        const socketServer = new socket_io_1.Server(server, {
            pingInterval: this.checkAliveTimeout,
            pingTimeout: this.checkAliveTimeout * 2,
            maxHttpBufferSize: this.maxHttpBufferSize
        });
        // Accept every namespace by using /.*/
        socketServer.of(/.*/).on('connection', async (socket) => {
            const request = socket.request;
            // Socket.io strips the `origin` header of the incoming request
            // We provide a `fix-origin` header in the `WebSocketConnectionProvider`
            request.headers.origin = request.headers['fix-origin'];
            if (await this.allowConnect(socket.request)) {
                this.handleConnection(socket);
                this.messagingListener.onDidWebSocketUpgrade(socket.request, socket);
            }
            else {
                socket.disconnect(true);
            }
        });
    }
    handleConnection(socket) {
        const pathname = socket.nsp.name;
        if (pathname && !this.wsHandlers.route(pathname, socket)) {
            console.error('Cannot find a ws handler for the path: ' + pathname);
        }
    }
    async allowConnect(request) {
        try {
            return this.wsRequestValidator.allowWsUpgrade(request);
        }
        catch (e) {
            return false;
        }
    }
    handleChannels(socket) {
        const channelHandlers = this.getConnectionChannelHandlers(socket);
        const channels = new Map();
        socket.on('message', data => {
            try {
                const message = JSON.parse(data.toString());
                if (message.kind === 'open') {
                    const { id, path } = message;
                    const channel = this.createChannel(id, socket);
                    if (channelHandlers.route(path, channel)) {
                        channel.ready();
                        console.debug(`Opening channel for service path '${path}'. [ID: ${id}]`);
                        channels.set(id, channel);
                        channel.onClose(() => {
                            console.debug(`Closing channel on service path '${path}'. [ID: ${id}]`);
                            channels.delete(id);
                        });
                    }
                    else {
                        console.error('Cannot find a service for the path: ' + path);
                    }
                }
                else {
                    const { id } = message;
                    const channel = channels.get(id);
                    if (channel) {
                        channel.handleMessage(message);
                    }
                    else {
                        console.error('The ws channel does not exist', id);
                    }
                }
            }
            catch (error) {
                console.error('Failed to handle message', { error, data });
            }
        });
        socket.on('error', err => {
            for (const channel of channels.values()) {
                channel.fireError(err);
            }
        });
        socket.on('disconnect', reason => {
            for (const channel of channels.values()) {
                channel.close(undefined, reason);
            }
            channels.clear();
        });
    }
    createSocketContainer(socket) {
        const connectionContainer = this.container.createChild();
        connectionContainer.bind(socket_io_1.Socket).toConstantValue(socket);
        return connectionContainer;
    }
    getConnectionChannelHandlers(socket) {
        const connectionContainer = this.createSocketContainer(socket);
        (0, common_1.bindContributionProvider)(connectionContainer, common_1.ConnectionHandler);
        connectionContainer.load(...this.connectionModules.getContributions());
        const connectionChannelHandlers = new MessagingContribution_1.ConnectionHandlers(this.channelHandlers);
        const connectionHandlers = connectionContainer.getNamed(common_1.ContributionProvider, common_1.ConnectionHandler);
        for (const connectionHandler of connectionHandlers.getContributions(true)) {
            connectionChannelHandlers.push(connectionHandler.path, (_, channel) => {
                const connection = (0, connection_1.createWebSocketConnection)(channel, new logger_1.ConsoleLogger());
                connectionHandler.onConnection(connection);
            });
        }
        return connectionChannelHandlers;
    }
    createChannel(id, socket) {
        return new web_socket_channel_1.WebSocketChannel(id, content => {
            if (socket.connected) {
                socket.send(content);
            }
        });
    }
};
__decorate([
    (0, inversify_1.inject)(exports.MessagingContainer),
    __metadata("design:type", Object)
], MessagingContribution.prototype, "container", void 0);
__decorate([
    (0, inversify_1.inject)(common_1.ContributionProvider),
    (0, inversify_1.named)(connection_container_module_1.ConnectionContainerModule),
    __metadata("design:type", Object)
], MessagingContribution.prototype, "connectionModules", void 0);
__decorate([
    (0, inversify_1.inject)(common_1.ContributionProvider),
    (0, inversify_1.named)(messaging_service_1.MessagingService.Contribution),
    __metadata("design:type", Object)
], MessagingContribution.prototype, "contributions", void 0);
__decorate([
    (0, inversify_1.inject)(ws_request_validators_1.WsRequestValidator),
    __metadata("design:type", ws_request_validators_1.WsRequestValidator)
], MessagingContribution.prototype, "wsRequestValidator", void 0);
__decorate([
    (0, inversify_1.inject)(messaging_listeners_1.MessagingListener),
    __metadata("design:type", messaging_listeners_1.MessagingListener)
], MessagingContribution.prototype, "messagingListener", void 0);
__decorate([
    (0, inversify_1.postConstruct)(),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", []),
    __metadata("design:returntype", void 0)
], MessagingContribution.prototype, "init", null);
MessagingContribution = MessagingContribution_1 = __decorate([
    (0, inversify_1.injectable)()
], MessagingContribution);
exports.MessagingContribution = MessagingContribution;
(function (MessagingContribution) {
    class ConnectionHandlers {
        constructor(parent) {
            this.parent = parent;
            this.handlers = [];
        }
        push(spec, callback) {
            const route = new Route(spec);
            this.handlers.push((path, channel) => {
                const params = route.match(path);
                if (!params) {
                    return false;
                }
                callback(params, channel);
                return route.reverse(params);
            });
        }
        route(path, connection) {
            for (const handler of this.handlers) {
                try {
                    const result = handler(path, connection);
                    if (result) {
                        return result;
                    }
                }
                catch (e) {
                    console.error(e);
                }
            }
            if (this.parent) {
                return this.parent.route(path, connection);
            }
            return false;
        }
    }
    MessagingContribution.ConnectionHandlers = ConnectionHandlers;
})(MessagingContribution = exports.MessagingContribution || (exports.MessagingContribution = {}));
exports.MessagingContribution = MessagingContribution;
//# sourceMappingURL=messaging-contribution.js.map