/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
var __values = (this && this.__values) || function(o) {
    var s = typeof Symbol === "function" && Symbol.iterator, m = s && o[s], i = 0;
    if (m) return m.call(o);
    if (o && typeof o.length === "number") return {
        next: function () {
            if (o && i >= o.length) o = void 0;
            return { value: o && o[i++], done: !o };
        }
    };
    throw new TypeError(s ? "Object is not iterable." : "Symbol.iterator is not defined.");
};
import { createDecorator } from '../../instantiation/common/instantiation';
import { localize } from '../../../nls';
import { WorkspaceFolder } from '../../workspace/common/workspace';
import { URI } from '../../../base/common/uri';
import { isWindows, isLinux, isMacintosh } from '../../../base/common/platform';
import { extname, isAbsolute } from '../../../base/common/path';
import { extname as resourceExtname, extUriBiasedIgnorePathCase } from '../../../base/common/resources';
import * as jsonEdit from '../../../base/common/jsonEdit';
import * as json from '../../../base/common/json';
import { Schemas } from '../../../base/common/network';
import { normalizeDriveLetter } from '../../../base/common/labels';
import { toSlashes } from '../../../base/common/extpath';
import { getRemoteAuthority } from '../../remote/common/remoteHosts';
export var WORKSPACE_EXTENSION = 'code-workspace';
var WORKSPACE_SUFFIX = "." + WORKSPACE_EXTENSION;
export var WORKSPACE_FILTER = [{ name: localize('codeWorkspace', "Code Workspace"), extensions: [WORKSPACE_EXTENSION] }];
export var UNTITLED_WORKSPACE_NAME = 'workspace.json';
export function hasWorkspaceFileExtension(path) {
    var ext = (typeof path === 'string') ? extname(path) : resourceExtname(path);
    return ext === WORKSPACE_SUFFIX;
}
export var IWorkspacesService = createDecorator('workspacesService');
export function isRecentWorkspace(curr) {
    return curr.hasOwnProperty('workspace');
}
export function isRecentFolder(curr) {
    return curr.hasOwnProperty('folderUri');
}
export function isRecentFile(curr) {
    return curr.hasOwnProperty('fileUri');
}
export function isSingleFolderWorkspaceIdentifier(obj) {
    var singleFolderIdentifier = obj;
    return typeof (singleFolderIdentifier === null || singleFolderIdentifier === void 0 ? void 0 : singleFolderIdentifier.id) === 'string' && URI.isUri(singleFolderIdentifier.uri);
}
export function toWorkspaceIdentifier(workspace) {
    // Multi root
    if (workspace.configuration) {
        return {
            id: workspace.id,
            configPath: workspace.configuration
        };
    }
    // Single folder
    if (workspace.folders.length === 1) {
        return {
            id: workspace.id,
            uri: workspace.folders[0].uri
        };
    }
    // Empty workspace
    return undefined;
}
export function isWorkspaceIdentifier(obj) {
    var workspaceIdentifier = obj;
    return typeof (workspaceIdentifier === null || workspaceIdentifier === void 0 ? void 0 : workspaceIdentifier.id) === 'string' && URI.isUri(workspaceIdentifier.configPath);
}
export function reviveIdentifier(identifier) {
    // Single Folder
    var singleFolderIdentifierCandidate = identifier;
    if (singleFolderIdentifierCandidate === null || singleFolderIdentifierCandidate === void 0 ? void 0 : singleFolderIdentifierCandidate.uri) {
        return { id: singleFolderIdentifierCandidate.id, uri: URI.revive(singleFolderIdentifierCandidate.uri) };
    }
    // Multi folder
    var workspaceIdentifierCandidate = identifier;
    if (workspaceIdentifierCandidate === null || workspaceIdentifierCandidate === void 0 ? void 0 : workspaceIdentifierCandidate.configPath) {
        return { id: workspaceIdentifierCandidate.id, configPath: URI.revive(workspaceIdentifierCandidate.configPath) };
    }
    // Empty
    if (identifier === null || identifier === void 0 ? void 0 : identifier.id) {
        return { id: identifier.id };
    }
    return undefined;
}
export function isUntitledWorkspace(path, environmentService) {
    return extUriBiasedIgnorePathCase.isEqualOrParent(path, environmentService.untitledWorkspacesHome);
}
//#endregion
//#region Workspace File Utilities
export function isStoredWorkspaceFolder(obj) {
    return isRawFileWorkspaceFolder(obj) || isRawUriWorkspaceFolder(obj);
}
export function isRawFileWorkspaceFolder(obj) {
    var candidate = obj;
    return typeof (candidate === null || candidate === void 0 ? void 0 : candidate.path) === 'string' && (!candidate.name || typeof candidate.name === 'string');
}
export function isRawUriWorkspaceFolder(obj) {
    var candidate = obj;
    return typeof (candidate === null || candidate === void 0 ? void 0 : candidate.uri) === 'string' && (!candidate.name || typeof candidate.name === 'string');
}
/**
 * Given a folder URI and the workspace config folder, computes the IStoredWorkspaceFolder using
* a relative or absolute path or a uri.
 * Undefined is returned if the folderURI and the targetConfigFolderURI don't have the same schema or authority
 *
 * @param folderURI a workspace folder
 * @param forceAbsolute if set, keep the path absolute
 * @param folderName a workspace name
 * @param targetConfigFolderURI the folder where the workspace is living in
 * @param useSlashForPath if set, use forward slashes for file paths on windows
 */
export function getStoredWorkspaceFolder(folderURI, forceAbsolute, folderName, targetConfigFolderURI, useSlashForPath, extUri) {
    if (useSlashForPath === void 0) { useSlashForPath = !isWindows; }
    if (folderURI.scheme !== targetConfigFolderURI.scheme) {
        return { name: folderName, uri: folderURI.toString(true) };
    }
    var folderPath = !forceAbsolute ? extUri.relativePath(targetConfigFolderURI, folderURI) : undefined;
    if (folderPath !== undefined) {
        if (folderPath.length === 0) {
            folderPath = '.';
        }
        else if (isWindows && folderURI.scheme === Schemas.file && !useSlashForPath) {
            // Windows gets special treatment:
            // - use backslahes unless slash is used by other existing folders
            folderPath = folderPath.replace(/\//g, '\\');
        }
    }
    else {
        // use absolute path
        if (folderURI.scheme === Schemas.file) {
            folderPath = folderURI.fsPath;
            if (isWindows) {
                // Windows gets special treatment:
                // - normalize all paths to get nice casing of drive letters
                // - use backslahes unless slash is used by other existing folders
                folderPath = normalizeDriveLetter(folderPath);
                if (useSlashForPath) {
                    folderPath = toSlashes(folderPath);
                }
            }
        }
        else {
            if (!extUri.isEqualAuthority(folderURI.authority, targetConfigFolderURI.authority)) {
                return { name: folderName, uri: folderURI.toString(true) };
            }
            folderPath = folderURI.path;
        }
    }
    return { name: folderName, path: folderPath };
}
export function toWorkspaceFolders(configuredFolders, workspaceConfigFile, extUri) {
    var e_1, _a;
    var result = [];
    var seen = new Set();
    var relativeTo = extUri.dirname(workspaceConfigFile);
    try {
        for (var configuredFolders_1 = __values(configuredFolders), configuredFolders_1_1 = configuredFolders_1.next(); !configuredFolders_1_1.done; configuredFolders_1_1 = configuredFolders_1.next()) {
            var configuredFolder = configuredFolders_1_1.value;
            var uri = null;
            if (isRawFileWorkspaceFolder(configuredFolder)) {
                if (configuredFolder.path) {
                    uri = extUri.resolvePath(relativeTo, configuredFolder.path);
                }
            }
            else if (isRawUriWorkspaceFolder(configuredFolder)) {
                try {
                    uri = URI.parse(configuredFolder.uri);
                    // this makes sure all workspace folder are absolute
                    if (uri.path[0] !== '/') {
                        uri = uri.with({ path: '/' + uri.path });
                    }
                }
                catch (e) {
                    console.warn(e);
                    // ignore
                }
            }
            if (uri) {
                // remove duplicates
                var comparisonKey = extUri.getComparisonKey(uri);
                if (!seen.has(comparisonKey)) {
                    seen.add(comparisonKey);
                    var name_1 = configuredFolder.name || extUri.basenameOrAuthority(uri);
                    result.push(new WorkspaceFolder({ uri: uri, name: name_1, index: result.length }, configuredFolder));
                }
            }
        }
    }
    catch (e_1_1) { e_1 = { error: e_1_1 }; }
    finally {
        try {
            if (configuredFolders_1_1 && !configuredFolders_1_1.done && (_a = configuredFolders_1.return)) _a.call(configuredFolders_1);
        }
        finally { if (e_1) throw e_1.error; }
    }
    return result;
}
/**
 * Rewrites the content of a workspace file to be saved at a new location.
 * Throws an exception if file is not a valid workspace file
 */
export function rewriteWorkspaceFileForNewLocation(rawWorkspaceContents, configPathURI, isFromUntitledWorkspace, targetConfigPathURI, extUri) {
    var e_2, _a;
    var storedWorkspace = doParseStoredWorkspace(configPathURI, rawWorkspaceContents);
    var sourceConfigFolder = extUri.dirname(configPathURI);
    var targetConfigFolder = extUri.dirname(targetConfigPathURI);
    var rewrittenFolders = [];
    var slashForPath = useSlashForPath(storedWorkspace.folders);
    try {
        for (var _b = __values(storedWorkspace.folders), _c = _b.next(); !_c.done; _c = _b.next()) {
            var folder = _c.value;
            var folderURI = isRawFileWorkspaceFolder(folder) ? extUri.resolvePath(sourceConfigFolder, folder.path) : URI.parse(folder.uri);
            var absolute = void 0;
            if (isFromUntitledWorkspace) {
                // if it was an untitled workspace, try to make paths relative
                absolute = false;
            }
            else {
                // for existing workspaces, preserve whether a path was absolute or relative
                absolute = !isRawFileWorkspaceFolder(folder) || isAbsolute(folder.path);
            }
            rewrittenFolders.push(getStoredWorkspaceFolder(folderURI, absolute, folder.name, targetConfigFolder, slashForPath, extUri));
        }
    }
    catch (e_2_1) { e_2 = { error: e_2_1 }; }
    finally {
        try {
            if (_c && !_c.done && (_a = _b.return)) _a.call(_b);
        }
        finally { if (e_2) throw e_2.error; }
    }
    // Preserve as much of the existing workspace as possible by using jsonEdit
    // and only changing the folders portion.
    var formattingOptions = { insertSpaces: false, tabSize: 4, eol: (isLinux || isMacintosh) ? '\n' : '\r\n' };
    var edits = jsonEdit.setProperty(rawWorkspaceContents, ['folders'], rewrittenFolders, formattingOptions);
    var newContent = jsonEdit.applyEdits(rawWorkspaceContents, edits);
    if (storedWorkspace.remoteAuthority === getRemoteAuthority(targetConfigPathURI)) {
        // unsaved remote workspaces have the remoteAuthority set. Remove it when no longer nexessary.
        newContent = jsonEdit.applyEdits(newContent, jsonEdit.removeProperty(newContent, ['remoteAuthority'], formattingOptions));
    }
    return newContent;
}
function doParseStoredWorkspace(path, contents) {
    // Parse workspace file
    var storedWorkspace = json.parse(contents); // use fault tolerant parser
    // Filter out folders which do not have a path or uri set
    if (storedWorkspace && Array.isArray(storedWorkspace.folders)) {
        storedWorkspace.folders = storedWorkspace.folders.filter(function (folder) { return isStoredWorkspaceFolder(folder); });
    }
    else {
        throw new Error(path + " looks like an invalid workspace file.");
    }
    return storedWorkspace;
}
export function useSlashForPath(storedFolders) {
    if (isWindows) {
        return storedFolders.some(function (folder) { return isRawFileWorkspaceFolder(folder) && folder.path.indexOf('/') >= 0; });
    }
    return true;
}
export function restoreRecentlyOpened(data, logService) {
    var result = { workspaces: [], files: [] };
    if (data) {
        var restoreGracefully = function (entries, func) {
            for (var i = 0; i < entries.length; i++) {
                try {
                    func(entries[i], i);
                }
                catch (e) {
                    logService.warn("Error restoring recent entry " + JSON.stringify(entries[i]) + ": " + e.toString() + ". Skip entry.");
                }
            }
        };
        var storedRecents_1 = data;
        if (Array.isArray(storedRecents_1.workspaces3)) {
            restoreGracefully(storedRecents_1.workspaces3, function (workspace, i) {
                var label = (Array.isArray(storedRecents_1.workspaceLabels) && storedRecents_1.workspaceLabels[i]) || undefined;
                if (typeof workspace === 'object' && typeof workspace.id === 'string' && typeof workspace.configURIPath === 'string') {
                    result.workspaces.push({ label: label, workspace: { id: workspace.id, configPath: URI.parse(workspace.configURIPath) } });
                }
                else if (typeof workspace === 'string') {
                    result.workspaces.push({ label: label, folderUri: URI.parse(workspace) });
                }
            });
        }
        if (Array.isArray(storedRecents_1.files2)) {
            restoreGracefully(storedRecents_1.files2, function (file, i) {
                var label = (Array.isArray(storedRecents_1.fileLabels) && storedRecents_1.fileLabels[i]) || undefined;
                if (typeof file === 'string') {
                    result.files.push({ label: label, fileUri: URI.parse(file) });
                }
            });
        }
    }
    return result;
}
export function toStoreData(recents) {
    var e_3, _a, e_4, _b;
    var serialized = { workspaces3: [], files2: [] };
    var hasLabel = false;
    var workspaceLabels = [];
    try {
        for (var _c = __values(recents.workspaces), _d = _c.next(); !_d.done; _d = _c.next()) {
            var recent = _d.value;
            if (isRecentFolder(recent)) {
                serialized.workspaces3.push(recent.folderUri.toString());
            }
            else {
                serialized.workspaces3.push({ id: recent.workspace.id, configURIPath: recent.workspace.configPath.toString() });
            }
            workspaceLabels.push(recent.label || null);
            hasLabel = hasLabel || !!recent.label;
        }
    }
    catch (e_3_1) { e_3 = { error: e_3_1 }; }
    finally {
        try {
            if (_d && !_d.done && (_a = _c.return)) _a.call(_c);
        }
        finally { if (e_3) throw e_3.error; }
    }
    if (hasLabel) {
        serialized.workspaceLabels = workspaceLabels;
    }
    hasLabel = false;
    var fileLabels = [];
    try {
        for (var _e = __values(recents.files), _f = _e.next(); !_f.done; _f = _e.next()) {
            var recent = _f.value;
            serialized.files2.push(recent.fileUri.toString());
            fileLabels.push(recent.label || null);
            hasLabel = hasLabel || !!recent.label;
        }
    }
    catch (e_4_1) { e_4 = { error: e_4_1 }; }
    finally {
        try {
            if (_f && !_f.done && (_b = _e.return)) _b.call(_e);
        }
        finally { if (e_4) throw e_4.error; }
    }
    if (hasLabel) {
        serialized.fileLabels = fileLabels;
    }
    return serialized;
}
//#endregion
