/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
var __read = (this && this.__read) || function (o, n) {
    var m = typeof Symbol === "function" && o[Symbol.iterator];
    if (!m) return o;
    var i = m.call(o), r, ar = [], e;
    try {
        while ((n === void 0 || n-- > 0) && !(r = i.next()).done) ar.push(r.value);
    }
    catch (error) { e = { error: error }; }
    finally {
        try {
            if (r && !r.done && (m = i["return"])) m.call(i);
        }
        finally { if (e) throw e.error; }
    }
    return ar;
};
var __spreadArray = (this && this.__spreadArray) || function (to, from) {
    for (var i = 0, il = from.length, j = to.length; i < il; i++, j++)
        to[j] = from[i];
    return to;
};
import * as extpath from './extpath';
import * as paths from './path';
import { URI, uriToFsPath } from './uri';
import { equalsIgnoreCase, compare as strCompare } from './strings';
import { Schemas } from './network';
import { isWindows, isLinux } from './platform';
export function originalFSPath(uri) {
    return uriToFsPath(uri, true);
}
var ExtUri = /** @class */ (function () {
    function ExtUri(_ignorePathCasing) {
        this._ignorePathCasing = _ignorePathCasing;
    }
    ExtUri.prototype.compare = function (uri1, uri2, ignoreFragment) {
        if (ignoreFragment === void 0) { ignoreFragment = false; }
        if (uri1 === uri2) {
            return 0;
        }
        return strCompare(this.getComparisonKey(uri1, ignoreFragment), this.getComparisonKey(uri2, ignoreFragment));
    };
    ExtUri.prototype.isEqual = function (uri1, uri2, ignoreFragment) {
        if (ignoreFragment === void 0) { ignoreFragment = false; }
        if (uri1 === uri2) {
            return true;
        }
        if (!uri1 || !uri2) {
            return false;
        }
        return this.getComparisonKey(uri1, ignoreFragment) === this.getComparisonKey(uri2, ignoreFragment);
    };
    ExtUri.prototype.getComparisonKey = function (uri, ignoreFragment) {
        if (ignoreFragment === void 0) { ignoreFragment = false; }
        return uri.with({
            path: this._ignorePathCasing(uri) ? uri.path.toLowerCase() : undefined,
            fragment: ignoreFragment ? null : undefined
        }).toString();
    };
    ExtUri.prototype.ignorePathCasing = function (uri) {
        return this._ignorePathCasing(uri);
    };
    ExtUri.prototype.isEqualOrParent = function (base, parentCandidate, ignoreFragment) {
        if (ignoreFragment === void 0) { ignoreFragment = false; }
        if (base.scheme === parentCandidate.scheme) {
            if (base.scheme === Schemas.file) {
                return extpath.isEqualOrParent(originalFSPath(base), originalFSPath(parentCandidate), this._ignorePathCasing(base)) && base.query === parentCandidate.query && (ignoreFragment || base.fragment === parentCandidate.fragment);
            }
            if (isEqualAuthority(base.authority, parentCandidate.authority)) {
                return extpath.isEqualOrParent(base.path, parentCandidate.path, this._ignorePathCasing(base), '/') && base.query === parentCandidate.query && (ignoreFragment || base.fragment === parentCandidate.fragment);
            }
        }
        return false;
    };
    // --- path math
    ExtUri.prototype.joinPath = function (resource) {
        var pathFragment = [];
        for (var _i = 1; _i < arguments.length; _i++) {
            pathFragment[_i - 1] = arguments[_i];
        }
        return URI.joinPath.apply(URI, __spreadArray([resource], __read(pathFragment)));
    };
    ExtUri.prototype.basenameOrAuthority = function (resource) {
        return basename(resource) || resource.authority;
    };
    ExtUri.prototype.basename = function (resource) {
        return paths.posix.basename(resource.path);
    };
    ExtUri.prototype.extname = function (resource) {
        return paths.posix.extname(resource.path);
    };
    ExtUri.prototype.dirname = function (resource) {
        if (resource.path.length === 0) {
            return resource;
        }
        var dirname;
        if (resource.scheme === Schemas.file) {
            dirname = URI.file(paths.dirname(originalFSPath(resource))).path;
        }
        else {
            dirname = paths.posix.dirname(resource.path);
            if (resource.authority && dirname.length && dirname.charCodeAt(0) !== 47 /* Slash */) {
                console.error("dirname(\"" + resource.toString + ")) resulted in a relative path");
                dirname = '/'; // If a URI contains an authority component, then the path component must either be empty or begin with a CharCode.Slash ("/") character
            }
        }
        return resource.with({
            path: dirname
        });
    };
    ExtUri.prototype.normalizePath = function (resource) {
        if (!resource.path.length) {
            return resource;
        }
        var normalizedPath;
        if (resource.scheme === Schemas.file) {
            normalizedPath = URI.file(paths.normalize(originalFSPath(resource))).path;
        }
        else {
            normalizedPath = paths.posix.normalize(resource.path);
        }
        return resource.with({
            path: normalizedPath
        });
    };
    ExtUri.prototype.relativePath = function (from, to) {
        if (from.scheme !== to.scheme || !isEqualAuthority(from.authority, to.authority)) {
            return undefined;
        }
        if (from.scheme === Schemas.file) {
            var relativePath_1 = paths.relative(originalFSPath(from), originalFSPath(to));
            return isWindows ? extpath.toSlashes(relativePath_1) : relativePath_1;
        }
        var fromPath = from.path || '/', toPath = to.path || '/';
        if (this._ignorePathCasing(from)) {
            // make casing of fromPath match toPath
            var i = 0;
            for (var len = Math.min(fromPath.length, toPath.length); i < len; i++) {
                if (fromPath.charCodeAt(i) !== toPath.charCodeAt(i)) {
                    if (fromPath.charAt(i).toLowerCase() !== toPath.charAt(i).toLowerCase()) {
                        break;
                    }
                }
            }
            fromPath = toPath.substr(0, i) + fromPath.substr(i);
        }
        return paths.posix.relative(fromPath, toPath);
    };
    ExtUri.prototype.resolvePath = function (base, path) {
        if (base.scheme === Schemas.file) {
            var newURI = URI.file(paths.resolve(originalFSPath(base), path));
            return base.with({
                authority: newURI.authority,
                path: newURI.path
            });
        }
        if (path.indexOf('/') === -1) { // no slashes? it's likely a Windows path
            path = extpath.toSlashes(path);
            if (/^[a-zA-Z]:(\/|$)/.test(path)) { // starts with a drive letter
                path = '/' + path;
            }
        }
        return base.with({
            path: paths.posix.resolve(base.path, path)
        });
    };
    // --- misc
    ExtUri.prototype.isAbsolutePath = function (resource) {
        return !!resource.path && resource.path[0] === '/';
    };
    ExtUri.prototype.isEqualAuthority = function (a1, a2) {
        return a1 === a2 || equalsIgnoreCase(a1, a2);
    };
    ExtUri.prototype.hasTrailingPathSeparator = function (resource, sep) {
        if (sep === void 0) { sep = paths.sep; }
        if (resource.scheme === Schemas.file) {
            var fsp = originalFSPath(resource);
            return fsp.length > extpath.getRoot(fsp).length && fsp[fsp.length - 1] === sep;
        }
        else {
            var p = resource.path;
            return (p.length > 1 && p.charCodeAt(p.length - 1) === 47 /* Slash */) && !(/^[a-zA-Z]:(\/$|\\$)/.test(resource.fsPath)); // ignore the slash at offset 0
        }
    };
    ExtUri.prototype.removeTrailingPathSeparator = function (resource, sep) {
        if (sep === void 0) { sep = paths.sep; }
        // Make sure that the path isn't a drive letter. A trailing separator there is not removable.
        if (hasTrailingPathSeparator(resource, sep)) {
            return resource.with({ path: resource.path.substr(0, resource.path.length - 1) });
        }
        return resource;
    };
    ExtUri.prototype.addTrailingPathSeparator = function (resource, sep) {
        if (sep === void 0) { sep = paths.sep; }
        var isRootSep = false;
        if (resource.scheme === Schemas.file) {
            var fsp = originalFSPath(resource);
            isRootSep = ((fsp !== undefined) && (fsp.length === extpath.getRoot(fsp).length) && (fsp[fsp.length - 1] === sep));
        }
        else {
            sep = '/';
            var p = resource.path;
            isRootSep = p.length === 1 && p.charCodeAt(p.length - 1) === 47 /* Slash */;
        }
        if (!isRootSep && !hasTrailingPathSeparator(resource, sep)) {
            return resource.with({ path: resource.path + '/' });
        }
        return resource;
    };
    return ExtUri;
}());
export { ExtUri };
/**
 * Unbiased utility that takes uris "as they are". This means it can be interchanged with
 * uri#toString() usages. The following is true
 * ```
 * assertEqual(aUri.toString() === bUri.toString(), exturi.isEqual(aUri, bUri))
 * ```
 */
export var extUri = new ExtUri(function () { return false; });
/**
 * BIASED utility that _mostly_ ignored the case of urs paths. ONLY use this util if you
 * understand what you are doing.
 *
 * This utility is INCOMPATIBLE with `uri.toString()`-usages and both CANNOT be used interchanged.
 *
 * When dealing with uris from files or documents, `extUri` (the unbiased friend)is sufficient
 * because those uris come from a "trustworthy source". When creating unknown uris it's always
 * better to use `IUriIdentityService` which exposes an `IExtUri`-instance which knows when path
 * casing matters.
 */
export var extUriBiasedIgnorePathCase = new ExtUri(function (uri) {
    // A file scheme resource is in the same platform as code, so ignore case for non linux platforms
    // Resource can be from another platform. Lowering the case as an hack. Should come from File system provider
    return uri.scheme === Schemas.file ? !isLinux : true;
});
/**
 * BIASED utility that always ignores the casing of uris paths. ONLY use this util if you
 * understand what you are doing.
 *
 * This utility is INCOMPATIBLE with `uri.toString()`-usages and both CANNOT be used interchanged.
 *
 * When dealing with uris from files or documents, `extUri` (the unbiased friend)is sufficient
 * because those uris come from a "trustworthy source". When creating unknown uris it's always
 * better to use `IUriIdentityService` which exposes an `IExtUri`-instance which knows when path
 * casing matters.
 */
export var extUriIgnorePathCase = new ExtUri(function (_) { return true; });
export var isEqual = extUri.isEqual.bind(extUri);
export var isEqualOrParent = extUri.isEqualOrParent.bind(extUri);
export var getComparisonKey = extUri.getComparisonKey.bind(extUri);
export var basenameOrAuthority = extUri.basenameOrAuthority.bind(extUri);
export var basename = extUri.basename.bind(extUri);
export var extname = extUri.extname.bind(extUri);
export var dirname = extUri.dirname.bind(extUri);
export var joinPath = extUri.joinPath.bind(extUri);
export var normalizePath = extUri.normalizePath.bind(extUri);
export var relativePath = extUri.relativePath.bind(extUri);
export var resolvePath = extUri.resolvePath.bind(extUri);
export var isAbsolutePath = extUri.isAbsolutePath.bind(extUri);
export var isEqualAuthority = extUri.isEqualAuthority.bind(extUri);
export var hasTrailingPathSeparator = extUri.hasTrailingPathSeparator.bind(extUri);
export var removeTrailingPathSeparator = extUri.removeTrailingPathSeparator.bind(extUri);
export var addTrailingPathSeparator = extUri.addTrailingPathSeparator.bind(extUri);
//#endregion
export function distinctParents(items, resourceAccessor) {
    var distinctParents = [];
    var _loop_1 = function (i) {
        var candidateResource = resourceAccessor(items[i]);
        if (items.some(function (otherItem, index) {
            if (index === i) {
                return false;
            }
            return isEqualOrParent(candidateResource, resourceAccessor(otherItem));
        })) {
            return "continue";
        }
        distinctParents.push(items[i]);
    };
    for (var i = 0; i < items.length; i++) {
        _loop_1(i);
    }
    return distinctParents;
}
/**
 * Data URI related helpers.
 */
export var DataUri;
(function (DataUri) {
    DataUri.META_DATA_LABEL = 'label';
    DataUri.META_DATA_DESCRIPTION = 'description';
    DataUri.META_DATA_SIZE = 'size';
    DataUri.META_DATA_MIME = 'mime';
    function parseMetaData(dataUri) {
        var metadata = new Map();
        // Given a URI of:  data:image/png;size:2313;label:SomeLabel;description:SomeDescription;base64,77+9UE5...
        // the metadata is: size:2313;label:SomeLabel;description:SomeDescription
        var meta = dataUri.path.substring(dataUri.path.indexOf(';') + 1, dataUri.path.lastIndexOf(';'));
        meta.split(';').forEach(function (property) {
            var _a = __read(property.split(':'), 2), key = _a[0], value = _a[1];
            if (key && value) {
                metadata.set(key, value);
            }
        });
        // Given a URI of:  data:image/png;size:2313;label:SomeLabel;description:SomeDescription;base64,77+9UE5...
        // the mime is: image/png
        var mime = dataUri.path.substring(0, dataUri.path.indexOf(';'));
        if (mime) {
            metadata.set(DataUri.META_DATA_MIME, mime);
        }
        return metadata;
    }
    DataUri.parseMetaData = parseMetaData;
})(DataUri || (DataUri = {}));
export function toLocalResource(resource, authority, localScheme) {
    if (authority) {
        var path = resource.path;
        if (path && path[0] !== paths.posix.sep) {
            path = paths.posix.sep + path;
        }
        return resource.with({ scheme: localScheme, authority: authority, path: path });
    }
    return resource.with({ scheme: localScheme });
}
