"use strict";
/********************************************************************************
 * Copyright (C) 2019 RedHat and others.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 ********************************************************************************/
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.HostedPluginDeployerHandler = void 0;
const fs = require("@theia/core/shared/fs-extra");
const inversify_1 = require("@theia/core/shared/inversify");
const core_1 = require("@theia/core");
const plugin_protocol_1 = require("../../common/plugin-protocol");
const plugin_reader_1 = require("./plugin-reader");
const promise_util_1 = require("@theia/core/lib/common/promise-util");
const hosted_plugin_localization_service_1 = require("./hosted-plugin-localization-service");
let HostedPluginDeployerHandler = class HostedPluginDeployerHandler {
    constructor() {
        this.deployedLocations = new Map();
        /**
         * Managed plugin metadata backend entries.
         */
        this.deployedBackendPlugins = new Map();
        /**
         * Managed plugin metadata frontend entries.
         */
        this.deployedFrontendPlugins = new Map();
        this.backendPluginsMetadataDeferred = new promise_util_1.Deferred();
        this.frontendPluginsMetadataDeferred = new promise_util_1.Deferred();
    }
    async getDeployedFrontendPluginIds() {
        // await first deploy
        await this.frontendPluginsMetadataDeferred.promise;
        // fetch the last deployed state
        return [...this.deployedFrontendPlugins.keys()];
    }
    async getDeployedBackendPluginIds() {
        // await first deploy
        await this.backendPluginsMetadataDeferred.promise;
        // fetch the last deployed state
        return [...this.deployedBackendPlugins.keys()];
    }
    getDeployedPlugin(pluginId) {
        const metadata = this.deployedBackendPlugins.get(pluginId);
        if (metadata) {
            return metadata;
        }
        return this.deployedFrontendPlugins.get(pluginId);
    }
    /**
     * @throws never! in order to isolate plugin deployment
     */
    async getPluginDependencies(entry) {
        const pluginPath = entry.path();
        try {
            const manifest = await this.reader.readPackage(pluginPath);
            if (!manifest) {
                return undefined;
            }
            const metadata = this.reader.readMetadata(manifest);
            const dependencies = { metadata };
            // Do not resolve system (aka builtin) plugins because it should be done statically at build time.
            if (entry.type !== plugin_protocol_1.PluginType.System) {
                dependencies.mapping = this.reader.readDependencies(manifest);
            }
            return dependencies;
        }
        catch (e) {
            console.error(`Failed to load plugin dependencies from '${pluginPath}' path`, e);
            return undefined;
        }
    }
    async deployFrontendPlugins(frontendPlugins) {
        for (const plugin of frontendPlugins) {
            await this.deployPlugin(plugin, 'frontend');
        }
        // resolve on first deploy
        this.frontendPluginsMetadataDeferred.resolve(undefined);
    }
    async deployBackendPlugins(backendPlugins) {
        for (const plugin of backendPlugins) {
            await this.deployPlugin(plugin, 'backend');
        }
        // rebuild translation config after deployment
        this.localizationService.buildTranslationConfig([...this.deployedBackendPlugins.values()]);
        // resolve on first deploy
        this.backendPluginsMetadataDeferred.resolve(undefined);
    }
    /**
     * @throws never! in order to isolate plugin deployment
     */
    async deployPlugin(entry, entryPoint) {
        const pluginPath = entry.path();
        try {
            const manifest = await this.reader.readPackage(pluginPath);
            if (!manifest) {
                return;
            }
            const metadata = this.reader.readMetadata(manifest);
            const deployedLocations = this.deployedLocations.get(metadata.model.id) || new Set();
            deployedLocations.add(entry.rootPath);
            this.deployedLocations.set(metadata.model.id, deployedLocations);
            const deployedPlugins = entryPoint === 'backend' ? this.deployedBackendPlugins : this.deployedFrontendPlugins;
            if (deployedPlugins.has(metadata.model.id)) {
                return;
            }
            const { type } = entry;
            const deployed = { metadata, type };
            deployed.contributes = this.reader.readContribution(manifest);
            this.localizationService.deployLocalizations(deployed);
            deployedPlugins.set(metadata.model.id, deployed);
            this.logger.info(`Deploying ${entryPoint} plugin "${metadata.model.name}@${metadata.model.version}" from "${metadata.model.entryPoint[entryPoint] || pluginPath}"`);
        }
        catch (e) {
            console.error(`Failed to deploy ${entryPoint} plugin from '${pluginPath}' path`, e);
        }
    }
    async undeployPlugin(pluginId) {
        this.deployedBackendPlugins.delete(pluginId);
        this.deployedFrontendPlugins.delete(pluginId);
        const deployedLocations = this.deployedLocations.get(pluginId);
        if (!deployedLocations) {
            return false;
        }
        this.deployedLocations.delete(pluginId);
        for (const location of deployedLocations) {
            try {
                await fs.remove(location);
            }
            catch (e) {
                console.error(`[${pluginId}]: failed to undeploy from "${location}", reason`, e);
            }
        }
        return true;
    }
};
__decorate([
    inversify_1.inject(core_1.ILogger),
    __metadata("design:type", Object)
], HostedPluginDeployerHandler.prototype, "logger", void 0);
__decorate([
    inversify_1.inject(plugin_reader_1.HostedPluginReader),
    __metadata("design:type", plugin_reader_1.HostedPluginReader)
], HostedPluginDeployerHandler.prototype, "reader", void 0);
__decorate([
    inversify_1.inject(hosted_plugin_localization_service_1.HostedPluginLocalizationService),
    __metadata("design:type", hosted_plugin_localization_service_1.HostedPluginLocalizationService)
], HostedPluginDeployerHandler.prototype, "localizationService", void 0);
HostedPluginDeployerHandler = __decorate([
    inversify_1.injectable()
], HostedPluginDeployerHandler);
exports.HostedPluginDeployerHandler = HostedPluginDeployerHandler;
//# sourceMappingURL=hosted-plugin-deployer-handler.js.map