/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
var __read = (this && this.__read) || function (o, n) {
    var m = typeof Symbol === "function" && o[Symbol.iterator];
    if (!m) return o;
    var i = m.call(o), r, ar = [], e;
    try {
        while ((n === void 0 || n-- > 0) && !(r = i.next()).done) ar.push(r.value);
    }
    catch (error) { e = { error: error }; }
    finally {
        try {
            if (r && !r.done && (m = i["return"])) m.call(i);
        }
        finally { if (e) throw e.error; }
    }
    return ar;
};
import * as buffer from '../../../base/common/buffer';
import { decodeUTF16LE } from '../core/stringBuilder';
function escapeNewLine(str) {
    return (str
        .replace(/\n/g, '\\n')
        .replace(/\r/g, '\\r'));
}
var TextChange = /** @class */ (function () {
    function TextChange(oldPosition, oldText, newPosition, newText) {
        this.oldPosition = oldPosition;
        this.oldText = oldText;
        this.newPosition = newPosition;
        this.newText = newText;
    }
    Object.defineProperty(TextChange.prototype, "oldLength", {
        get: function () {
            return this.oldText.length;
        },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(TextChange.prototype, "oldEnd", {
        get: function () {
            return this.oldPosition + this.oldText.length;
        },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(TextChange.prototype, "newLength", {
        get: function () {
            return this.newText.length;
        },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(TextChange.prototype, "newEnd", {
        get: function () {
            return this.newPosition + this.newText.length;
        },
        enumerable: false,
        configurable: true
    });
    TextChange.prototype.toString = function () {
        if (this.oldText.length === 0) {
            return "(insert@" + this.oldPosition + " \"" + escapeNewLine(this.newText) + "\")";
        }
        if (this.newText.length === 0) {
            return "(delete@" + this.oldPosition + " \"" + escapeNewLine(this.oldText) + "\")";
        }
        return "(replace@" + this.oldPosition + " \"" + escapeNewLine(this.oldText) + "\" with \"" + escapeNewLine(this.newText) + "\")";
    };
    TextChange._writeStringSize = function (str) {
        return (4 + 2 * str.length);
    };
    TextChange._writeString = function (b, str, offset) {
        var len = str.length;
        buffer.writeUInt32BE(b, len, offset);
        offset += 4;
        for (var i = 0; i < len; i++) {
            buffer.writeUInt16LE(b, str.charCodeAt(i), offset);
            offset += 2;
        }
        return offset;
    };
    TextChange._readString = function (b, offset) {
        var len = buffer.readUInt32BE(b, offset);
        offset += 4;
        return decodeUTF16LE(b, offset, len);
    };
    TextChange.prototype.writeSize = function () {
        return (+4 // oldPosition
            + 4 // newPosition
            + TextChange._writeStringSize(this.oldText)
            + TextChange._writeStringSize(this.newText));
    };
    TextChange.prototype.write = function (b, offset) {
        buffer.writeUInt32BE(b, this.oldPosition, offset);
        offset += 4;
        buffer.writeUInt32BE(b, this.newPosition, offset);
        offset += 4;
        offset = TextChange._writeString(b, this.oldText, offset);
        offset = TextChange._writeString(b, this.newText, offset);
        return offset;
    };
    TextChange.read = function (b, offset, dest) {
        var oldPosition = buffer.readUInt32BE(b, offset);
        offset += 4;
        var newPosition = buffer.readUInt32BE(b, offset);
        offset += 4;
        var oldText = TextChange._readString(b, offset);
        offset += TextChange._writeStringSize(oldText);
        var newText = TextChange._readString(b, offset);
        offset += TextChange._writeStringSize(newText);
        dest.push(new TextChange(oldPosition, oldText, newPosition, newText));
        return offset;
    };
    return TextChange;
}());
export { TextChange };
export function compressConsecutiveTextChanges(prevEdits, currEdits) {
    if (prevEdits === null || prevEdits.length === 0) {
        return currEdits;
    }
    var compressor = new TextChangeCompressor(prevEdits, currEdits);
    return compressor.compress();
}
var TextChangeCompressor = /** @class */ (function () {
    function TextChangeCompressor(prevEdits, currEdits) {
        this._prevEdits = prevEdits;
        this._currEdits = currEdits;
        this._result = [];
        this._resultLen = 0;
        this._prevLen = this._prevEdits.length;
        this._prevDeltaOffset = 0;
        this._currLen = this._currEdits.length;
        this._currDeltaOffset = 0;
    }
    TextChangeCompressor.prototype.compress = function () {
        var prevIndex = 0;
        var currIndex = 0;
        var prevEdit = this._getPrev(prevIndex);
        var currEdit = this._getCurr(currIndex);
        while (prevIndex < this._prevLen || currIndex < this._currLen) {
            if (prevEdit === null) {
                this._acceptCurr(currEdit);
                currEdit = this._getCurr(++currIndex);
                continue;
            }
            if (currEdit === null) {
                this._acceptPrev(prevEdit);
                prevEdit = this._getPrev(++prevIndex);
                continue;
            }
            if (currEdit.oldEnd <= prevEdit.newPosition) {
                this._acceptCurr(currEdit);
                currEdit = this._getCurr(++currIndex);
                continue;
            }
            if (prevEdit.newEnd <= currEdit.oldPosition) {
                this._acceptPrev(prevEdit);
                prevEdit = this._getPrev(++prevIndex);
                continue;
            }
            if (currEdit.oldPosition < prevEdit.newPosition) {
                var _a = __read(TextChangeCompressor._splitCurr(currEdit, prevEdit.newPosition - currEdit.oldPosition), 2), e1 = _a[0], e2 = _a[1];
                this._acceptCurr(e1);
                currEdit = e2;
                continue;
            }
            if (prevEdit.newPosition < currEdit.oldPosition) {
                var _b = __read(TextChangeCompressor._splitPrev(prevEdit, currEdit.oldPosition - prevEdit.newPosition), 2), e1 = _b[0], e2 = _b[1];
                this._acceptPrev(e1);
                prevEdit = e2;
                continue;
            }
            // At this point, currEdit.oldPosition === prevEdit.newPosition
            var mergePrev = void 0;
            var mergeCurr = void 0;
            if (currEdit.oldEnd === prevEdit.newEnd) {
                mergePrev = prevEdit;
                mergeCurr = currEdit;
                prevEdit = this._getPrev(++prevIndex);
                currEdit = this._getCurr(++currIndex);
            }
            else if (currEdit.oldEnd < prevEdit.newEnd) {
                var _c = __read(TextChangeCompressor._splitPrev(prevEdit, currEdit.oldLength), 2), e1 = _c[0], e2 = _c[1];
                mergePrev = e1;
                mergeCurr = currEdit;
                prevEdit = e2;
                currEdit = this._getCurr(++currIndex);
            }
            else {
                var _d = __read(TextChangeCompressor._splitCurr(currEdit, prevEdit.newLength), 2), e1 = _d[0], e2 = _d[1];
                mergePrev = prevEdit;
                mergeCurr = e1;
                prevEdit = this._getPrev(++prevIndex);
                currEdit = e2;
            }
            this._result[this._resultLen++] = new TextChange(mergePrev.oldPosition, mergePrev.oldText, mergeCurr.newPosition, mergeCurr.newText);
            this._prevDeltaOffset += mergePrev.newLength - mergePrev.oldLength;
            this._currDeltaOffset += mergeCurr.newLength - mergeCurr.oldLength;
        }
        var merged = TextChangeCompressor._merge(this._result);
        var cleaned = TextChangeCompressor._removeNoOps(merged);
        return cleaned;
    };
    TextChangeCompressor.prototype._acceptCurr = function (currEdit) {
        this._result[this._resultLen++] = TextChangeCompressor._rebaseCurr(this._prevDeltaOffset, currEdit);
        this._currDeltaOffset += currEdit.newLength - currEdit.oldLength;
    };
    TextChangeCompressor.prototype._getCurr = function (currIndex) {
        return (currIndex < this._currLen ? this._currEdits[currIndex] : null);
    };
    TextChangeCompressor.prototype._acceptPrev = function (prevEdit) {
        this._result[this._resultLen++] = TextChangeCompressor._rebasePrev(this._currDeltaOffset, prevEdit);
        this._prevDeltaOffset += prevEdit.newLength - prevEdit.oldLength;
    };
    TextChangeCompressor.prototype._getPrev = function (prevIndex) {
        return (prevIndex < this._prevLen ? this._prevEdits[prevIndex] : null);
    };
    TextChangeCompressor._rebaseCurr = function (prevDeltaOffset, currEdit) {
        return new TextChange(currEdit.oldPosition - prevDeltaOffset, currEdit.oldText, currEdit.newPosition, currEdit.newText);
    };
    TextChangeCompressor._rebasePrev = function (currDeltaOffset, prevEdit) {
        return new TextChange(prevEdit.oldPosition, prevEdit.oldText, prevEdit.newPosition + currDeltaOffset, prevEdit.newText);
    };
    TextChangeCompressor._splitPrev = function (edit, offset) {
        var preText = edit.newText.substr(0, offset);
        var postText = edit.newText.substr(offset);
        return [
            new TextChange(edit.oldPosition, edit.oldText, edit.newPosition, preText),
            new TextChange(edit.oldEnd, '', edit.newPosition + offset, postText)
        ];
    };
    TextChangeCompressor._splitCurr = function (edit, offset) {
        var preText = edit.oldText.substr(0, offset);
        var postText = edit.oldText.substr(offset);
        return [
            new TextChange(edit.oldPosition, preText, edit.newPosition, edit.newText),
            new TextChange(edit.oldPosition + offset, postText, edit.newEnd, '')
        ];
    };
    TextChangeCompressor._merge = function (edits) {
        if (edits.length === 0) {
            return edits;
        }
        var result = [], resultLen = 0;
        var prev = edits[0];
        for (var i = 1; i < edits.length; i++) {
            var curr = edits[i];
            if (prev.oldEnd === curr.oldPosition) {
                // Merge into `prev`
                prev = new TextChange(prev.oldPosition, prev.oldText + curr.oldText, prev.newPosition, prev.newText + curr.newText);
            }
            else {
                result[resultLen++] = prev;
                prev = curr;
            }
        }
        result[resultLen++] = prev;
        return result;
    };
    TextChangeCompressor._removeNoOps = function (edits) {
        if (edits.length === 0) {
            return edits;
        }
        var result = [], resultLen = 0;
        for (var i = 0; i < edits.length; i++) {
            var edit = edits[i];
            if (edit.oldText === edit.newText) {
                continue;
            }
            result[resultLen++] = edit;
        }
        return result;
    };
    return TextChangeCompressor;
}());
