/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
var __values = (this && this.__values) || function(o) {
    var s = typeof Symbol === "function" && Symbol.iterator, m = s && o[s], i = 0;
    if (m) return m.call(o);
    if (o && typeof o.length === "number") return {
        next: function () {
            if (o && i >= o.length) o = void 0;
            return { value: o && o[i++], done: !o };
        }
    };
    throw new TypeError(s ? "Object is not iterable." : "Symbol.iterator is not defined.");
};
var __read = (this && this.__read) || function (o, n) {
    var m = typeof Symbol === "function" && o[Symbol.iterator];
    if (!m) return o;
    var i = m.call(o), r, ar = [], e;
    try {
        while ((n === void 0 || n-- > 0) && !(r = i.next()).done) ar.push(r.value);
    }
    catch (error) { e = { error: error }; }
    finally {
        try {
            if (r && !r.done && (m = i["return"])) m.call(i);
        }
        finally { if (e) throw e.error; }
    }
    return ar;
};
var __spreadArray = (this && this.__spreadArray) || function (to, from) {
    for (var i = 0, il = from.length, j = to.length; i < il; i++, j++)
        to[j] = from[i];
    return to;
};
import { compareAnything } from './comparers';
import { matchesPrefix, isUpper, fuzzyScore, createMatches as createFuzzyMatches } from './filters';
import { sep } from './path';
import { isWindows, isLinux } from './platform';
import { stripWildcards, equalsIgnoreCase } from './strings';
var NO_MATCH = 0;
var NO_SCORE = [NO_MATCH, []];
// const DEBUG = false;
// const DEBUG_MATRIX = false;
export function scoreFuzzy(target, query, queryLower, fuzzy) {
    if (!target || !query) {
        return NO_SCORE; // return early if target or query are undefined
    }
    var targetLength = target.length;
    var queryLength = query.length;
    if (targetLength < queryLength) {
        return NO_SCORE; // impossible for query to be contained in target
    }
    // if (DEBUG) {
    // 	console.group(`Target: ${target}, Query: ${query}`);
    // }
    var targetLower = target.toLowerCase();
    // When not searching fuzzy, we require the query to be contained fully
    // in the target string contiguously.
    if (!fuzzy) {
        if (!targetLower.includes(queryLower)) {
            // if (DEBUG) {
            // 	console.log(`Characters not matching consecutively ${queryLower} within ${targetLower}`);
            // }
            return NO_SCORE;
        }
    }
    var res = doScoreFuzzy(query, queryLower, queryLength, target, targetLower, targetLength);
    // if (DEBUG) {
    // 	console.log(`%cFinal Score: ${res[0]}`, 'font-weight: bold');
    // 	console.groupEnd();
    // }
    return res;
}
function doScoreFuzzy(query, queryLower, queryLength, target, targetLower, targetLength) {
    var scores = [];
    var matches = [];
    //
    // Build Scorer Matrix:
    //
    // The matrix is composed of query q and target t. For each index we score
    // q[i] with t[i] and compare that with the previous score. If the score is
    // equal or larger, we keep the match. In addition to the score, we also keep
    // the length of the consecutive matches to use as boost for the score.
    //
    //      t   a   r   g   e   t
    //  q
    //  u
    //  e
    //  r
    //  y
    //
    for (var queryIndex_1 = 0; queryIndex_1 < queryLength; queryIndex_1++) {
        var queryIndexOffset = queryIndex_1 * targetLength;
        var queryIndexPreviousOffset = queryIndexOffset - targetLength;
        var queryIndexGtNull = queryIndex_1 > 0;
        var queryCharAtIndex = query[queryIndex_1];
        var queryLowerCharAtIndex = queryLower[queryIndex_1];
        for (var targetIndex_1 = 0; targetIndex_1 < targetLength; targetIndex_1++) {
            var targetIndexGtNull = targetIndex_1 > 0;
            var currentIndex = queryIndexOffset + targetIndex_1;
            var leftIndex = currentIndex - 1;
            var diagIndex = queryIndexPreviousOffset + targetIndex_1 - 1;
            var leftScore = targetIndexGtNull ? scores[leftIndex] : 0;
            var diagScore = queryIndexGtNull && targetIndexGtNull ? scores[diagIndex] : 0;
            var matchesSequenceLength = queryIndexGtNull && targetIndexGtNull ? matches[diagIndex] : 0;
            // If we are not matching on the first query character any more, we only produce a
            // score if we had a score previously for the last query index (by looking at the diagScore).
            // This makes sure that the query always matches in sequence on the target. For example
            // given a target of "ede" and a query of "de", we would otherwise produce a wrong high score
            // for query[1] ("e") matching on target[0] ("e") because of the "beginning of word" boost.
            var score = void 0;
            if (!diagScore && queryIndexGtNull) {
                score = 0;
            }
            else {
                score = computeCharScore(queryCharAtIndex, queryLowerCharAtIndex, target, targetLower, targetIndex_1, matchesSequenceLength);
            }
            // We have a score and its equal or larger than the left score
            // Match: sequence continues growing from previous diag value
            // Score: increases by diag score value
            if (score && diagScore + score >= leftScore) {
                matches[currentIndex] = matchesSequenceLength + 1;
                scores[currentIndex] = diagScore + score;
            }
            // We either have no score or the score is lower than the left score
            // Match: reset to 0
            // Score: pick up from left hand side
            else {
                matches[currentIndex] = NO_MATCH;
                scores[currentIndex] = leftScore;
            }
        }
    }
    // Restore Positions (starting from bottom right of matrix)
    var positions = [];
    var queryIndex = queryLength - 1;
    var targetIndex = targetLength - 1;
    while (queryIndex >= 0 && targetIndex >= 0) {
        var currentIndex = queryIndex * targetLength + targetIndex;
        var match = matches[currentIndex];
        if (match === NO_MATCH) {
            targetIndex--; // go left
        }
        else {
            positions.push(targetIndex);
            // go up and left
            queryIndex--;
            targetIndex--;
        }
    }
    // Print matrix
    // if (DEBUG_MATRIX) {
    // printMatrix(query, target, matches, scores);
    // }
    return [scores[queryLength * targetLength - 1], positions.reverse()];
}
function computeCharScore(queryCharAtIndex, queryLowerCharAtIndex, target, targetLower, targetIndex, matchesSequenceLength) {
    var score = 0;
    if (!considerAsEqual(queryLowerCharAtIndex, targetLower[targetIndex])) {
        return score; // no match of characters
    }
    // Character match bonus
    score += 1;
    // if (DEBUG) {
    // console.groupCollapsed(`%cCharacter match bonus: +1 (char: ${queryLowerCharAtIndex} at index ${targetIndex}, total score: ${score})`, 'font-weight: normal');
    // }
    // Consecutive match bonus
    if (matchesSequenceLength > 0) {
        score += (matchesSequenceLength * 5);
        // if (DEBUG) {
        // console.log(`Consecutive match bonus: +${matchesSequenceLength * 5}`);
        // }
    }
    // Same case bonus
    if (queryCharAtIndex === target[targetIndex]) {
        score += 1;
        // if (DEBUG) {
        // 	console.log('Same case bonus: +1');
        // }
    }
    // Start of word bonus
    if (targetIndex === 0) {
        score += 8;
        // if (DEBUG) {
        // 	console.log('Start of word bonus: +8');
        // }
    }
    else {
        // After separator bonus
        var separatorBonus = scoreSeparatorAtPos(target.charCodeAt(targetIndex - 1));
        if (separatorBonus) {
            score += separatorBonus;
            // if (DEBUG) {
            // console.log(`After separtor bonus: +${separatorBonus}`);
            // }
        }
        // Inside word upper case bonus (camel case)
        else if (isUpper(target.charCodeAt(targetIndex))) {
            score += 2;
            // if (DEBUG) {
            // 	console.log('Inside word upper case bonus: +2');
            // }
        }
    }
    // if (DEBUG) {
    // 	console.groupEnd();
    // }
    return score;
}
function considerAsEqual(a, b) {
    if (a === b) {
        return true;
    }
    // Special case path spearators: ignore platform differences
    if (a === '/' || a === '\\') {
        return b === '/' || b === '\\';
    }
    return false;
}
function scoreSeparatorAtPos(charCode) {
    switch (charCode) {
        case 47 /* Slash */:
        case 92 /* Backslash */:
            return 5; // prefer path separators...
        case 95 /* Underline */:
        case 45 /* Dash */:
        case 46 /* Period */:
        case 32 /* Space */:
        case 39 /* SingleQuote */:
        case 34 /* DoubleQuote */:
        case 58 /* Colon */:
            return 4; // ...over other separators
        default:
            return 0;
    }
}
var NO_SCORE2 = [undefined, []];
export function scoreFuzzy2(target, query, patternStart, wordStart) {
    if (patternStart === void 0) { patternStart = 0; }
    if (wordStart === void 0) { wordStart = 0; }
    // Score: multiple inputs
    var preparedQuery = query;
    if (preparedQuery.values && preparedQuery.values.length > 1) {
        return doScoreFuzzy2Multiple(target, preparedQuery.values, patternStart, wordStart);
    }
    // Score: single input
    return doScoreFuzzy2Single(target, query, patternStart, wordStart);
}
function doScoreFuzzy2Multiple(target, query, patternStart, wordStart) {
    var e_1, _a;
    var totalScore = 0;
    var totalMatches = [];
    try {
        for (var query_1 = __values(query), query_1_1 = query_1.next(); !query_1_1.done; query_1_1 = query_1.next()) {
            var queryPiece = query_1_1.value;
            var _b = __read(doScoreFuzzy2Single(target, queryPiece, patternStart, wordStart), 2), score = _b[0], matches = _b[1];
            if (typeof score !== 'number') {
                // if a single query value does not match, return with
                // no score entirely, we require all queries to match
                return NO_SCORE2;
            }
            totalScore += score;
            totalMatches.push.apply(totalMatches, __spreadArray([], __read(matches)));
        }
    }
    catch (e_1_1) { e_1 = { error: e_1_1 }; }
    finally {
        try {
            if (query_1_1 && !query_1_1.done && (_a = query_1.return)) _a.call(query_1);
        }
        finally { if (e_1) throw e_1.error; }
    }
    // if we have a score, ensure that the positions are
    // sorted in ascending order and distinct
    return [totalScore, normalizeMatches(totalMatches)];
}
function doScoreFuzzy2Single(target, query, patternStart, wordStart) {
    var score = fuzzyScore(query.original, query.originalLowercase, patternStart, target, target.toLowerCase(), wordStart, true);
    if (!score) {
        return NO_SCORE2;
    }
    return [score[0], createFuzzyMatches(score)];
}
var NO_ITEM_SCORE = Object.freeze({ score: 0 });
var PATH_IDENTITY_SCORE = 1 << 18;
var LABEL_PREFIX_SCORE_THRESHOLD = 1 << 17;
var LABEL_SCORE_THRESHOLD = 1 << 16;
export function scoreItemFuzzy(item, query, fuzzy, accessor, cache) {
    if (!item || !query.normalized) {
        return NO_ITEM_SCORE; // we need an item and query to score on at least
    }
    var label = accessor.getItemLabel(item);
    if (!label) {
        return NO_ITEM_SCORE; // we need a label at least
    }
    var description = accessor.getItemDescription(item);
    // in order to speed up scoring, we cache the score with a unique hash based on:
    // - label
    // - description (if provided)
    // - query (normalized)
    // - number of query pieces (i.e. 'hello world' and 'helloworld' are different)
    // - whether fuzzy matching is enabled or not
    var cacheHash;
    if (description) {
        cacheHash = "" + label + description + query.normalized + (Array.isArray(query.values) ? query.values.length : '') + fuzzy;
    }
    else {
        cacheHash = "" + label + query.normalized + (Array.isArray(query.values) ? query.values.length : '') + fuzzy;
    }
    var cached = cache[cacheHash];
    if (cached) {
        return cached;
    }
    var itemScore = doScoreItemFuzzy(label, description, accessor.getItemPath(item), query, fuzzy);
    cache[cacheHash] = itemScore;
    return itemScore;
}
function doScoreItemFuzzy(label, description, path, query, fuzzy) {
    var preferLabelMatches = !path || !query.containsPathSeparator;
    // Treat identity matches on full path highest
    if (path && (isLinux ? query.pathNormalized === path : equalsIgnoreCase(query.pathNormalized, path))) {
        return { score: PATH_IDENTITY_SCORE, labelMatch: [{ start: 0, end: label.length }], descriptionMatch: description ? [{ start: 0, end: description.length }] : undefined };
    }
    // Score: multiple inputs
    if (query.values && query.values.length > 1) {
        return doScoreItemFuzzyMultiple(label, description, path, query.values, preferLabelMatches, fuzzy);
    }
    // Score: single input
    return doScoreItemFuzzySingle(label, description, path, query, preferLabelMatches, fuzzy);
}
function doScoreItemFuzzyMultiple(label, description, path, query, preferLabelMatches, fuzzy) {
    var e_2, _a;
    var totalScore = 0;
    var totalLabelMatches = [];
    var totalDescriptionMatches = [];
    try {
        for (var query_2 = __values(query), query_2_1 = query_2.next(); !query_2_1.done; query_2_1 = query_2.next()) {
            var queryPiece = query_2_1.value;
            var _b = doScoreItemFuzzySingle(label, description, path, queryPiece, preferLabelMatches, fuzzy), score = _b.score, labelMatch = _b.labelMatch, descriptionMatch = _b.descriptionMatch;
            if (score === NO_MATCH) {
                // if a single query value does not match, return with
                // no score entirely, we require all queries to match
                return NO_ITEM_SCORE;
            }
            totalScore += score;
            if (labelMatch) {
                totalLabelMatches.push.apply(totalLabelMatches, __spreadArray([], __read(labelMatch)));
            }
            if (descriptionMatch) {
                totalDescriptionMatches.push.apply(totalDescriptionMatches, __spreadArray([], __read(descriptionMatch)));
            }
        }
    }
    catch (e_2_1) { e_2 = { error: e_2_1 }; }
    finally {
        try {
            if (query_2_1 && !query_2_1.done && (_a = query_2.return)) _a.call(query_2);
        }
        finally { if (e_2) throw e_2.error; }
    }
    // if we have a score, ensure that the positions are
    // sorted in ascending order and distinct
    return {
        score: totalScore,
        labelMatch: normalizeMatches(totalLabelMatches),
        descriptionMatch: normalizeMatches(totalDescriptionMatches)
    };
}
function doScoreItemFuzzySingle(label, description, path, query, preferLabelMatches, fuzzy) {
    // Prefer label matches if told so or we have no description
    if (preferLabelMatches || !description) {
        var _a = __read(scoreFuzzy(label, query.normalized, query.normalizedLowercase, fuzzy), 2), labelScore = _a[0], labelPositions = _a[1];
        if (labelScore) {
            // If we have a prefix match on the label, we give a much
            // higher baseScore to elevate these matches over others
            // This ensures that typing a file name wins over results
            // that are present somewhere in the label, but not the
            // beginning.
            var labelPrefixMatch = matchesPrefix(query.normalized, label);
            var baseScore = void 0;
            if (labelPrefixMatch) {
                baseScore = LABEL_PREFIX_SCORE_THRESHOLD;
                // We give another boost to labels that are short, e.g. given
                // files "window.ts" and "windowActions.ts" and a query of
                // "window", we want "window.ts" to receive a higher score.
                // As such we compute the percentage the query has within the
                // label and add that to the baseScore.
                var prefixLengthBoost = Math.round((query.normalized.length / label.length) * 100);
                baseScore += prefixLengthBoost;
            }
            else {
                baseScore = LABEL_SCORE_THRESHOLD;
            }
            return { score: baseScore + labelScore, labelMatch: labelPrefixMatch || createMatches(labelPositions) };
        }
    }
    // Finally compute description + label scores if we have a description
    if (description) {
        var descriptionPrefix = description;
        if (!!path) {
            descriptionPrefix = "" + description + sep; // assume this is a file path
        }
        var descriptionPrefixLength_1 = descriptionPrefix.length;
        var descriptionAndLabel = "" + descriptionPrefix + label;
        var _b = __read(scoreFuzzy(descriptionAndLabel, query.normalized, query.normalizedLowercase, fuzzy), 2), labelDescriptionScore = _b[0], labelDescriptionPositions = _b[1];
        if (labelDescriptionScore) {
            var labelDescriptionMatches = createMatches(labelDescriptionPositions);
            var labelMatch_1 = [];
            var descriptionMatch_1 = [];
            // We have to split the matches back onto the label and description portions
            labelDescriptionMatches.forEach(function (h) {
                // Match overlaps label and description part, we need to split it up
                if (h.start < descriptionPrefixLength_1 && h.end > descriptionPrefixLength_1) {
                    labelMatch_1.push({ start: 0, end: h.end - descriptionPrefixLength_1 });
                    descriptionMatch_1.push({ start: h.start, end: descriptionPrefixLength_1 });
                }
                // Match on label part
                else if (h.start >= descriptionPrefixLength_1) {
                    labelMatch_1.push({ start: h.start - descriptionPrefixLength_1, end: h.end - descriptionPrefixLength_1 });
                }
                // Match on description part
                else {
                    descriptionMatch_1.push(h);
                }
            });
            return { score: labelDescriptionScore, labelMatch: labelMatch_1, descriptionMatch: descriptionMatch_1 };
        }
    }
    return NO_ITEM_SCORE;
}
function createMatches(offsets) {
    var e_3, _a;
    var ret = [];
    if (!offsets) {
        return ret;
    }
    var last;
    try {
        for (var offsets_1 = __values(offsets), offsets_1_1 = offsets_1.next(); !offsets_1_1.done; offsets_1_1 = offsets_1.next()) {
            var pos = offsets_1_1.value;
            if (last && last.end === pos) {
                last.end += 1;
            }
            else {
                last = { start: pos, end: pos + 1 };
                ret.push(last);
            }
        }
    }
    catch (e_3_1) { e_3 = { error: e_3_1 }; }
    finally {
        try {
            if (offsets_1_1 && !offsets_1_1.done && (_a = offsets_1.return)) _a.call(offsets_1);
        }
        finally { if (e_3) throw e_3.error; }
    }
    return ret;
}
function normalizeMatches(matches) {
    var e_4, _a;
    // sort matches by start to be able to normalize
    var sortedMatches = matches.sort(function (matchA, matchB) {
        return matchA.start - matchB.start;
    });
    // merge matches that overlap
    var normalizedMatches = [];
    var currentMatch = undefined;
    try {
        for (var sortedMatches_1 = __values(sortedMatches), sortedMatches_1_1 = sortedMatches_1.next(); !sortedMatches_1_1.done; sortedMatches_1_1 = sortedMatches_1.next()) {
            var match = sortedMatches_1_1.value;
            // if we have no current match or the matches
            // do not overlap, we take it as is and remember
            // it for future merging
            if (!currentMatch || !matchOverlaps(currentMatch, match)) {
                currentMatch = match;
                normalizedMatches.push(match);
            }
            // otherwise we merge the matches
            else {
                currentMatch.start = Math.min(currentMatch.start, match.start);
                currentMatch.end = Math.max(currentMatch.end, match.end);
            }
        }
    }
    catch (e_4_1) { e_4 = { error: e_4_1 }; }
    finally {
        try {
            if (sortedMatches_1_1 && !sortedMatches_1_1.done && (_a = sortedMatches_1.return)) _a.call(sortedMatches_1);
        }
        finally { if (e_4) throw e_4.error; }
    }
    return normalizedMatches;
}
function matchOverlaps(matchA, matchB) {
    if (matchA.end < matchB.start) {
        return false; // A ends before B starts
    }
    if (matchB.end < matchA.start) {
        return false; // B ends before A starts
    }
    return true;
}
//#endregion
//#region Comparers
export function compareItemsByFuzzyScore(itemA, itemB, query, fuzzy, accessor, cache) {
    var itemScoreA = scoreItemFuzzy(itemA, query, fuzzy, accessor, cache);
    var itemScoreB = scoreItemFuzzy(itemB, query, fuzzy, accessor, cache);
    var scoreA = itemScoreA.score;
    var scoreB = itemScoreB.score;
    // 1.) identity matches have highest score
    if (scoreA === PATH_IDENTITY_SCORE || scoreB === PATH_IDENTITY_SCORE) {
        if (scoreA !== scoreB) {
            return scoreA === PATH_IDENTITY_SCORE ? -1 : 1;
        }
    }
    // 2.) matches on label are considered higher compared to label+description matches
    if (scoreA > LABEL_SCORE_THRESHOLD || scoreB > LABEL_SCORE_THRESHOLD) {
        if (scoreA !== scoreB) {
            return scoreA > scoreB ? -1 : 1;
        }
        // prefer more compact matches over longer in label (unless this is a prefix match where
        // longer prefix matches are actually preferred)
        if (scoreA < LABEL_PREFIX_SCORE_THRESHOLD && scoreB < LABEL_PREFIX_SCORE_THRESHOLD) {
            var comparedByMatchLength = compareByMatchLength(itemScoreA.labelMatch, itemScoreB.labelMatch);
            if (comparedByMatchLength !== 0) {
                return comparedByMatchLength;
            }
        }
        // prefer shorter labels over longer labels
        var labelA = accessor.getItemLabel(itemA) || '';
        var labelB = accessor.getItemLabel(itemB) || '';
        if (labelA.length !== labelB.length) {
            return labelA.length - labelB.length;
        }
    }
    // 3.) compare by score in label+description
    if (scoreA !== scoreB) {
        return scoreA > scoreB ? -1 : 1;
    }
    // 4.) scores are identical: prefer matches in label over non-label matches
    var itemAHasLabelMatches = Array.isArray(itemScoreA.labelMatch) && itemScoreA.labelMatch.length > 0;
    var itemBHasLabelMatches = Array.isArray(itemScoreB.labelMatch) && itemScoreB.labelMatch.length > 0;
    if (itemAHasLabelMatches && !itemBHasLabelMatches) {
        return -1;
    }
    else if (itemBHasLabelMatches && !itemAHasLabelMatches) {
        return 1;
    }
    // 5.) scores are identical: prefer more compact matches (label and description)
    var itemAMatchDistance = computeLabelAndDescriptionMatchDistance(itemA, itemScoreA, accessor);
    var itemBMatchDistance = computeLabelAndDescriptionMatchDistance(itemB, itemScoreB, accessor);
    if (itemAMatchDistance && itemBMatchDistance && itemAMatchDistance !== itemBMatchDistance) {
        return itemBMatchDistance > itemAMatchDistance ? -1 : 1;
    }
    // 6.) scores are identical: start to use the fallback compare
    return fallbackCompare(itemA, itemB, query, accessor);
}
function computeLabelAndDescriptionMatchDistance(item, score, accessor) {
    var matchStart = -1;
    var matchEnd = -1;
    // If we have description matches, the start is first of description match
    if (score.descriptionMatch && score.descriptionMatch.length) {
        matchStart = score.descriptionMatch[0].start;
    }
    // Otherwise, the start is the first label match
    else if (score.labelMatch && score.labelMatch.length) {
        matchStart = score.labelMatch[0].start;
    }
    // If we have label match, the end is the last label match
    // If we had a description match, we add the length of the description
    // as offset to the end to indicate this.
    if (score.labelMatch && score.labelMatch.length) {
        matchEnd = score.labelMatch[score.labelMatch.length - 1].end;
        if (score.descriptionMatch && score.descriptionMatch.length) {
            var itemDescription = accessor.getItemDescription(item);
            if (itemDescription) {
                matchEnd += itemDescription.length;
            }
        }
    }
    // If we have just a description match, the end is the last description match
    else if (score.descriptionMatch && score.descriptionMatch.length) {
        matchEnd = score.descriptionMatch[score.descriptionMatch.length - 1].end;
    }
    return matchEnd - matchStart;
}
function compareByMatchLength(matchesA, matchesB) {
    if ((!matchesA && !matchesB) || ((!matchesA || !matchesA.length) && (!matchesB || !matchesB.length))) {
        return 0; // make sure to not cause bad comparing when matches are not provided
    }
    if (!matchesB || !matchesB.length) {
        return -1;
    }
    if (!matchesA || !matchesA.length) {
        return 1;
    }
    // Compute match length of A (first to last match)
    var matchStartA = matchesA[0].start;
    var matchEndA = matchesA[matchesA.length - 1].end;
    var matchLengthA = matchEndA - matchStartA;
    // Compute match length of B (first to last match)
    var matchStartB = matchesB[0].start;
    var matchEndB = matchesB[matchesB.length - 1].end;
    var matchLengthB = matchEndB - matchStartB;
    // Prefer shorter match length
    return matchLengthA === matchLengthB ? 0 : matchLengthB < matchLengthA ? 1 : -1;
}
function fallbackCompare(itemA, itemB, query, accessor) {
    // check for label + description length and prefer shorter
    var labelA = accessor.getItemLabel(itemA) || '';
    var labelB = accessor.getItemLabel(itemB) || '';
    var descriptionA = accessor.getItemDescription(itemA);
    var descriptionB = accessor.getItemDescription(itemB);
    var labelDescriptionALength = labelA.length + (descriptionA ? descriptionA.length : 0);
    var labelDescriptionBLength = labelB.length + (descriptionB ? descriptionB.length : 0);
    if (labelDescriptionALength !== labelDescriptionBLength) {
        return labelDescriptionALength - labelDescriptionBLength;
    }
    // check for path length and prefer shorter
    var pathA = accessor.getItemPath(itemA);
    var pathB = accessor.getItemPath(itemB);
    if (pathA && pathB && pathA.length !== pathB.length) {
        return pathA.length - pathB.length;
    }
    // 7.) finally we have equal scores and equal length, we fallback to comparer
    // compare by label
    if (labelA !== labelB) {
        return compareAnything(labelA, labelB, query.normalized);
    }
    // compare by description
    if (descriptionA && descriptionB && descriptionA !== descriptionB) {
        return compareAnything(descriptionA, descriptionB, query.normalized);
    }
    // compare by path
    if (pathA && pathB && pathA !== pathB) {
        return compareAnything(pathA, pathB, query.normalized);
    }
    // equal
    return 0;
}
/**
 * Helper function to prepare a search value for scoring by removing unwanted characters
 * and allowing to score on multiple pieces separated by whitespace character.
 */
var MULTIPLE_QUERY_VALUES_SEPARATOR = ' ';
export function prepareQuery(original) {
    var e_5, _a;
    if (typeof original !== 'string') {
        original = '';
    }
    var originalLowercase = original.toLowerCase();
    var _b = normalizeQuery(original), pathNormalized = _b.pathNormalized, normalized = _b.normalized, normalizedLowercase = _b.normalizedLowercase;
    var containsPathSeparator = pathNormalized.indexOf(sep) >= 0;
    var values = undefined;
    var originalSplit = original.split(MULTIPLE_QUERY_VALUES_SEPARATOR);
    if (originalSplit.length > 1) {
        try {
            for (var originalSplit_1 = __values(originalSplit), originalSplit_1_1 = originalSplit_1.next(); !originalSplit_1_1.done; originalSplit_1_1 = originalSplit_1.next()) {
                var originalPiece = originalSplit_1_1.value;
                var _c = normalizeQuery(originalPiece), pathNormalizedPiece = _c.pathNormalized, normalizedPiece = _c.normalized, normalizedLowercasePiece = _c.normalizedLowercase;
                if (normalizedPiece) {
                    if (!values) {
                        values = [];
                    }
                    values.push({
                        original: originalPiece,
                        originalLowercase: originalPiece.toLowerCase(),
                        pathNormalized: pathNormalizedPiece,
                        normalized: normalizedPiece,
                        normalizedLowercase: normalizedLowercasePiece
                    });
                }
            }
        }
        catch (e_5_1) { e_5 = { error: e_5_1 }; }
        finally {
            try {
                if (originalSplit_1_1 && !originalSplit_1_1.done && (_a = originalSplit_1.return)) _a.call(originalSplit_1);
            }
            finally { if (e_5) throw e_5.error; }
        }
    }
    return { original: original, originalLowercase: originalLowercase, pathNormalized: pathNormalized, normalized: normalized, normalizedLowercase: normalizedLowercase, values: values, containsPathSeparator: containsPathSeparator };
}
function normalizeQuery(original) {
    var pathNormalized;
    if (isWindows) {
        pathNormalized = original.replace(/\//g, sep); // Help Windows users to search for paths when using slash
    }
    else {
        pathNormalized = original.replace(/\\/g, sep); // Help macOS/Linux users to search for paths when using backslash
    }
    var normalized = stripWildcards(pathNormalized).replace(/\s/g, '');
    return {
        pathNormalized: pathNormalized,
        normalized: normalized,
        normalizedLowercase: normalized.toLowerCase()
    };
}
export function pieceToQuery(arg1) {
    if (Array.isArray(arg1)) {
        return prepareQuery(arg1.map(function (piece) { return piece.original; }).join(MULTIPLE_QUERY_VALUES_SEPARATOR));
    }
    return prepareQuery(arg1.original);
}
//#endregion
