/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
var __extends = (this && this.__extends) || (function () {
    var extendStatics = function (d, b) {
        extendStatics = Object.setPrototypeOf ||
            ({ __proto__: [] } instanceof Array && function (d, b) { d.__proto__ = b; }) ||
            function (d, b) { for (var p in b) if (Object.prototype.hasOwnProperty.call(b, p)) d[p] = b[p]; };
        return extendStatics(d, b);
    };
    return function (d, b) {
        if (typeof b !== "function" && b !== null)
            throw new TypeError("Class extends value " + String(b) + " is not a constructor or null");
        extendStatics(d, b);
        function __() { this.constructor = d; }
        d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
    };
})();
import './actionbar.css';
import { Disposable, dispose } from '../../../common/lifecycle';
import { ActionRunner, Separator } from '../../../common/actions';
import * as DOM from '../../dom';
import * as types from '../../../common/types';
import { StandardKeyboardEvent } from '../../keyboardEvent';
import { Emitter } from '../../../common/event';
import { ActionViewItem, BaseActionViewItem } from './actionViewItems';
var ActionBar = /** @class */ (function (_super) {
    __extends(ActionBar, _super);
    function ActionBar(container, options) {
        if (options === void 0) { options = {}; }
        var _a, _b, _c, _d, _e, _f;
        var _this = _super.call(this) || this;
        // Trigger Key Tracking
        _this.triggerKeyDown = false;
        _this.focusable = true;
        _this._onDidBlur = _this._register(new Emitter());
        _this.onDidBlur = _this._onDidBlur.event;
        _this._onDidCancel = _this._register(new Emitter({ onFirstListenerAdd: function () { return _this.cancelHasListener = true; } }));
        _this.onDidCancel = _this._onDidCancel.event;
        _this.cancelHasListener = false;
        _this._onDidRun = _this._register(new Emitter());
        _this.onDidRun = _this._onDidRun.event;
        _this._onBeforeRun = _this._register(new Emitter());
        _this.onBeforeRun = _this._onBeforeRun.event;
        _this.options = options;
        _this._context = (_a = options.context) !== null && _a !== void 0 ? _a : null;
        _this._orientation = (_b = _this.options.orientation) !== null && _b !== void 0 ? _b : 0 /* HORIZONTAL */;
        _this._triggerKeys = {
            keyDown: (_d = (_c = _this.options.triggerKeys) === null || _c === void 0 ? void 0 : _c.keyDown) !== null && _d !== void 0 ? _d : false,
            keys: (_f = (_e = _this.options.triggerKeys) === null || _e === void 0 ? void 0 : _e.keys) !== null && _f !== void 0 ? _f : [3 /* Enter */, 10 /* Space */]
        };
        if (_this.options.actionRunner) {
            _this._actionRunner = _this.options.actionRunner;
        }
        else {
            _this._actionRunner = new ActionRunner();
            _this._register(_this._actionRunner);
        }
        _this._register(_this._actionRunner.onDidRun(function (e) { return _this._onDidRun.fire(e); }));
        _this._register(_this._actionRunner.onBeforeRun(function (e) { return _this._onBeforeRun.fire(e); }));
        _this._actionIds = [];
        _this.viewItems = [];
        _this.focusedItem = undefined;
        _this.domNode = document.createElement('div');
        _this.domNode.className = 'monaco-action-bar';
        if (options.animated !== false) {
            _this.domNode.classList.add('animated');
        }
        var previousKeys;
        var nextKeys;
        switch (_this._orientation) {
            case 0 /* HORIZONTAL */:
                previousKeys = [15 /* LeftArrow */];
                nextKeys = [17 /* RightArrow */];
                break;
            case 1 /* HORIZONTAL_REVERSE */:
                previousKeys = [17 /* RightArrow */];
                nextKeys = [15 /* LeftArrow */];
                _this.domNode.className += ' reverse';
                break;
            case 2 /* VERTICAL */:
                previousKeys = [16 /* UpArrow */];
                nextKeys = [18 /* DownArrow */];
                _this.domNode.className += ' vertical';
                break;
            case 3 /* VERTICAL_REVERSE */:
                previousKeys = [18 /* DownArrow */];
                nextKeys = [16 /* UpArrow */];
                _this.domNode.className += ' vertical reverse';
                break;
        }
        _this._register(DOM.addDisposableListener(_this.domNode, DOM.EventType.KEY_DOWN, function (e) {
            var event = new StandardKeyboardEvent(e);
            var eventHandled = true;
            var focusedItem = typeof _this.focusedItem === 'number' ? _this.viewItems[_this.focusedItem] : undefined;
            if (previousKeys && (event.equals(previousKeys[0]) || event.equals(previousKeys[1]))) {
                eventHandled = _this.focusPrevious();
            }
            else if (nextKeys && (event.equals(nextKeys[0]) || event.equals(nextKeys[1]))) {
                eventHandled = _this.focusNext();
            }
            else if (event.equals(9 /* Escape */) && _this.cancelHasListener) {
                _this._onDidCancel.fire();
            }
            else if (event.equals(2 /* Tab */) && focusedItem instanceof BaseActionViewItem && focusedItem.trapsArrowNavigation) {
                _this.focusNext();
            }
            else if (_this.isTriggerKeyEvent(event)) {
                // Staying out of the else branch even if not triggered
                if (_this._triggerKeys.keyDown) {
                    _this.doTrigger(event);
                }
                else {
                    _this.triggerKeyDown = true;
                }
            }
            else {
                eventHandled = false;
            }
            if (eventHandled) {
                event.preventDefault();
                event.stopPropagation();
            }
        }));
        _this._register(DOM.addDisposableListener(_this.domNode, DOM.EventType.KEY_UP, function (e) {
            var event = new StandardKeyboardEvent(e);
            // Run action on Enter/Space
            if (_this.isTriggerKeyEvent(event)) {
                if (!_this._triggerKeys.keyDown && _this.triggerKeyDown) {
                    _this.triggerKeyDown = false;
                    _this.doTrigger(event);
                }
                event.preventDefault();
                event.stopPropagation();
            }
            // Recompute focused item
            else if (event.equals(2 /* Tab */) || event.equals(1024 /* Shift */ | 2 /* Tab */)) {
                _this.updateFocusedItem();
            }
        }));
        _this.focusTracker = _this._register(DOM.trackFocus(_this.domNode));
        _this._register(_this.focusTracker.onDidBlur(function () {
            if (DOM.getActiveElement() === _this.domNode || !DOM.isAncestor(DOM.getActiveElement(), _this.domNode)) {
                _this._onDidBlur.fire();
                _this.focusedItem = undefined;
                _this.triggerKeyDown = false;
            }
        }));
        _this._register(_this.focusTracker.onDidFocus(function () { return _this.updateFocusedItem(); }));
        _this.actionsList = document.createElement('ul');
        _this.actionsList.className = 'actions-container';
        _this.actionsList.setAttribute('role', 'toolbar');
        if (_this.options.ariaLabel) {
            _this.actionsList.setAttribute('aria-label', _this.options.ariaLabel);
        }
        _this.domNode.appendChild(_this.actionsList);
        container.appendChild(_this.domNode);
        return _this;
    }
    ActionBar.prototype.setAriaLabel = function (label) {
        if (label) {
            this.actionsList.setAttribute('aria-label', label);
        }
        else {
            this.actionsList.removeAttribute('aria-label');
        }
    };
    // Some action bars should not be focusable at times
    // When an action bar is not focusable make sure to make all the elements inside it not focusable
    // When an action bar is focusable again, make sure the first item can be focused
    ActionBar.prototype.setFocusable = function (focusable) {
        this.focusable = focusable;
        if (this.focusable) {
            var firstEnabled = this.viewItems.find(function (vi) { return vi instanceof BaseActionViewItem && vi.isEnabled(); });
            if (firstEnabled instanceof BaseActionViewItem) {
                firstEnabled.setFocusable(true);
            }
        }
        else {
            this.viewItems.forEach(function (vi) {
                if (vi instanceof BaseActionViewItem) {
                    vi.setFocusable(false);
                }
            });
        }
    };
    ActionBar.prototype.isTriggerKeyEvent = function (event) {
        var ret = false;
        this._triggerKeys.keys.forEach(function (keyCode) {
            ret = ret || event.equals(keyCode);
        });
        return ret;
    };
    ActionBar.prototype.updateFocusedItem = function () {
        for (var i = 0; i < this.actionsList.children.length; i++) {
            var elem = this.actionsList.children[i];
            if (DOM.isAncestor(DOM.getActiveElement(), elem)) {
                this.focusedItem = i;
                break;
            }
        }
    };
    Object.defineProperty(ActionBar.prototype, "context", {
        get: function () {
            return this._context;
        },
        set: function (context) {
            this._context = context;
            this.viewItems.forEach(function (i) { return i.setActionContext(context); });
        },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(ActionBar.prototype, "actionRunner", {
        get: function () {
            return this._actionRunner;
        },
        set: function (actionRunner) {
            if (actionRunner) {
                this._actionRunner = actionRunner;
                this.viewItems.forEach(function (item) { return item.actionRunner = actionRunner; });
            }
        },
        enumerable: false,
        configurable: true
    });
    ActionBar.prototype.getContainer = function () {
        return this.domNode;
    };
    ActionBar.prototype.hasAction = function (action) {
        return this._actionIds.includes(action.id);
    };
    ActionBar.prototype.push = function (arg, options) {
        var _this = this;
        if (options === void 0) { options = {}; }
        var actions = Array.isArray(arg) ? arg : [arg];
        var index = types.isNumber(options.index) ? options.index : null;
        actions.forEach(function (action) {
            var actionViewItemElement = document.createElement('li');
            actionViewItemElement.className = 'action-item';
            actionViewItemElement.setAttribute('role', 'presentation');
            // Prevent native context menu on actions
            if (!_this.options.allowContextMenu) {
                _this._register(DOM.addDisposableListener(actionViewItemElement, DOM.EventType.CONTEXT_MENU, function (e) {
                    DOM.EventHelper.stop(e, true);
                }));
            }
            var item;
            if (_this.options.actionViewItemProvider) {
                item = _this.options.actionViewItemProvider(action);
            }
            if (!item) {
                item = new ActionViewItem(_this.context, action, options);
            }
            item.actionRunner = _this._actionRunner;
            item.setActionContext(_this.context);
            item.render(actionViewItemElement);
            if (_this.focusable && item instanceof BaseActionViewItem && _this.viewItems.length === 0) {
                // We need to allow for the first enabled item to be focused on using tab navigation #106441
                item.setFocusable(true);
            }
            if (index === null || index < 0 || index >= _this.actionsList.children.length) {
                _this.actionsList.appendChild(actionViewItemElement);
                _this.viewItems.push(item);
                _this._actionIds.push(action.id);
            }
            else {
                _this.actionsList.insertBefore(actionViewItemElement, _this.actionsList.children[index]);
                _this.viewItems.splice(index, 0, item);
                _this._actionIds.splice(index, 0, action.id);
                index++;
            }
        });
        if (typeof this.focusedItem === 'number') {
            // After a clear actions might be re-added to simply toggle some actions. We should preserve focus #97128
            this.focus(this.focusedItem);
        }
    };
    ActionBar.prototype.getWidth = function (index) {
        if (index >= 0 && index < this.actionsList.children.length) {
            var item = this.actionsList.children.item(index);
            if (item) {
                return item.clientWidth;
            }
        }
        return 0;
    };
    ActionBar.prototype.getHeight = function (index) {
        if (index >= 0 && index < this.actionsList.children.length) {
            var item = this.actionsList.children.item(index);
            if (item) {
                return item.clientHeight;
            }
        }
        return 0;
    };
    ActionBar.prototype.pull = function (index) {
        if (index >= 0 && index < this.viewItems.length) {
            this.actionsList.removeChild(this.actionsList.childNodes[index]);
            dispose(this.viewItems.splice(index, 1));
            this._actionIds.splice(index, 1);
        }
    };
    ActionBar.prototype.clear = function () {
        dispose(this.viewItems);
        this.viewItems = [];
        this._actionIds = [];
        DOM.clearNode(this.actionsList);
    };
    ActionBar.prototype.length = function () {
        return this.viewItems.length;
    };
    ActionBar.prototype.isEmpty = function () {
        return this.viewItems.length === 0;
    };
    ActionBar.prototype.focus = function (arg) {
        var selectFirst = false;
        var index = undefined;
        if (arg === undefined) {
            selectFirst = true;
        }
        else if (typeof arg === 'number') {
            index = arg;
        }
        else if (typeof arg === 'boolean') {
            selectFirst = arg;
        }
        if (selectFirst && typeof this.focusedItem === 'undefined') {
            var firstEnabled = this.viewItems.findIndex(function (item) { return item.isEnabled(); });
            // Focus the first enabled item
            this.focusedItem = firstEnabled === -1 ? undefined : firstEnabled;
            this.updateFocus();
        }
        else {
            if (index !== undefined) {
                this.focusedItem = index;
            }
            this.updateFocus();
        }
    };
    ActionBar.prototype.focusNext = function () {
        if (typeof this.focusedItem === 'undefined') {
            this.focusedItem = this.viewItems.length - 1;
        }
        var startIndex = this.focusedItem;
        var item;
        do {
            if (this.options.preventLoopNavigation && this.focusedItem + 1 >= this.viewItems.length) {
                this.focusedItem = startIndex;
                return false;
            }
            this.focusedItem = (this.focusedItem + 1) % this.viewItems.length;
            item = this.viewItems[this.focusedItem];
        } while (this.focusedItem !== startIndex && this.options.focusOnlyEnabledItems && !item.isEnabled());
        this.updateFocus();
        return true;
    };
    ActionBar.prototype.focusPrevious = function () {
        if (typeof this.focusedItem === 'undefined') {
            this.focusedItem = 0;
        }
        var startIndex = this.focusedItem;
        var item;
        do {
            this.focusedItem = this.focusedItem - 1;
            if (this.focusedItem < 0) {
                if (this.options.preventLoopNavigation) {
                    this.focusedItem = startIndex;
                    return false;
                }
                this.focusedItem = this.viewItems.length - 1;
            }
            item = this.viewItems[this.focusedItem];
        } while (this.focusedItem !== startIndex && this.options.focusOnlyEnabledItems && !item.isEnabled());
        this.updateFocus(true);
        return true;
    };
    ActionBar.prototype.updateFocus = function (fromRight, preventScroll) {
        if (typeof this.focusedItem === 'undefined') {
            this.actionsList.focus({ preventScroll: preventScroll });
        }
        for (var i = 0; i < this.viewItems.length; i++) {
            var item = this.viewItems[i];
            var actionViewItem = item;
            if (i === this.focusedItem) {
                var focusItem = true;
                if (!types.isFunction(actionViewItem.focus)) {
                    focusItem = false;
                }
                if (this.options.focusOnlyEnabledItems && types.isFunction(item.isEnabled) && !item.isEnabled()) {
                    focusItem = false;
                }
                if (focusItem) {
                    actionViewItem.focus(fromRight);
                }
                else {
                    this.actionsList.focus({ preventScroll: preventScroll });
                }
            }
            else {
                if (types.isFunction(actionViewItem.blur)) {
                    actionViewItem.blur();
                }
            }
        }
    };
    ActionBar.prototype.doTrigger = function (event) {
        if (typeof this.focusedItem === 'undefined') {
            return; //nothing to focus
        }
        // trigger action
        var actionViewItem = this.viewItems[this.focusedItem];
        if (actionViewItem instanceof BaseActionViewItem) {
            var context = (actionViewItem._context === null || actionViewItem._context === undefined) ? event : actionViewItem._context;
            this.run(actionViewItem._action, context);
        }
    };
    ActionBar.prototype.run = function (action, context) {
        return this._actionRunner.run(action, context);
    };
    ActionBar.prototype.dispose = function () {
        dispose(this.viewItems);
        this.viewItems = [];
        this._actionIds = [];
        this.getContainer().remove();
        _super.prototype.dispose.call(this);
    };
    return ActionBar;
}(Disposable));
export { ActionBar };
export function prepareActions(actions) {
    if (!actions.length) {
        return actions;
    }
    // Clean up leading separators
    var firstIndexOfAction = -1;
    for (var i = 0; i < actions.length; i++) {
        if (actions[i].id === Separator.ID) {
            continue;
        }
        firstIndexOfAction = i;
        break;
    }
    if (firstIndexOfAction === -1) {
        return [];
    }
    actions = actions.slice(firstIndexOfAction);
    // Clean up trailing separators
    for (var h = actions.length - 1; h >= 0; h--) {
        var isSeparator = actions[h].id === Separator.ID;
        if (isSeparator) {
            actions.splice(h, 1);
        }
        else {
            break;
        }
    }
    // Clean up separator duplicates
    var foundAction = false;
    for (var k = actions.length - 1; k >= 0; k--) {
        var isSeparator = actions[k].id === Separator.ID;
        if (isSeparator && !foundAction) {
            actions.splice(k, 1);
        }
        else if (!isSeparator) {
            foundAction = true;
        }
        else if (isSeparator) {
            foundAction = false;
        }
    }
    return actions;
}
