/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
var __extends = (this && this.__extends) || (function () {
    var extendStatics = function (d, b) {
        extendStatics = Object.setPrototypeOf ||
            ({ __proto__: [] } instanceof Array && function (d, b) { d.__proto__ = b; }) ||
            function (d, b) { for (var p in b) if (Object.prototype.hasOwnProperty.call(b, p)) d[p] = b[p]; };
        return extendStatics(d, b);
    };
    return function (d, b) {
        if (typeof b !== "function" && b !== null)
            throw new TypeError("Class extends value " + String(b) + " is not a constructor or null");
        extendStatics(d, b);
        function __() { this.constructor = d; }
        d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
    };
})();
var __read = (this && this.__read) || function (o, n) {
    var m = typeof Symbol === "function" && o[Symbol.iterator];
    if (!m) return o;
    var i = m.call(o), r, ar = [], e;
    try {
        while ((n === void 0 || n-- > 0) && !(r = i.next()).done) ar.push(r.value);
    }
    catch (error) { e = { error: error }; }
    finally {
        try {
            if (r && !r.done && (m = i["return"])) m.call(i);
        }
        finally { if (e) throw e.error; }
    }
    return ar;
};
var __spreadArray = (this && this.__spreadArray) || function (to, from) {
    for (var i = 0, il = from.length, j = to.length; i < il; i++, j++)
        to[j] = from[i];
    return to;
};
import './actionbar.css';
import * as platform from '../../../common/platform';
import * as nls from '../../../../nls';
import { Disposable } from '../../../common/lifecycle';
import { SelectBox } from '../selectBox/selectBox';
import { Action, ActionRunner, Separator } from '../../../common/actions';
import * as types from '../../../common/types';
import { EventType as TouchEventType, Gesture } from '../../touch';
import { DataTransfers } from '../../dnd';
import { isFirefox } from '../../browser';
import { $, addDisposableListener, append, EventHelper, EventType } from '../../dom';
var BaseActionViewItem = /** @class */ (function (_super) {
    __extends(BaseActionViewItem, _super);
    function BaseActionViewItem(context, action, options) {
        if (options === void 0) { options = {}; }
        var _this = _super.call(this) || this;
        _this.options = options;
        _this._context = context || _this;
        _this._action = action;
        if (action instanceof Action) {
            _this._register(action.onDidChange(function (event) {
                if (!_this.element) {
                    // we have not been rendered yet, so there
                    // is no point in updating the UI
                    return;
                }
                _this.handleActionChangeEvent(event);
            }));
        }
        return _this;
    }
    BaseActionViewItem.prototype.handleActionChangeEvent = function (event) {
        if (event.enabled !== undefined) {
            this.updateEnabled();
        }
        if (event.checked !== undefined) {
            this.updateChecked();
        }
        if (event.class !== undefined) {
            this.updateClass();
        }
        if (event.label !== undefined) {
            this.updateLabel();
            this.updateTooltip();
        }
        if (event.tooltip !== undefined) {
            this.updateTooltip();
        }
    };
    Object.defineProperty(BaseActionViewItem.prototype, "actionRunner", {
        get: function () {
            if (!this._actionRunner) {
                this._actionRunner = this._register(new ActionRunner());
            }
            return this._actionRunner;
        },
        set: function (actionRunner) {
            this._actionRunner = actionRunner;
        },
        enumerable: false,
        configurable: true
    });
    BaseActionViewItem.prototype.getAction = function () {
        return this._action;
    };
    BaseActionViewItem.prototype.isEnabled = function () {
        return this._action.enabled;
    };
    BaseActionViewItem.prototype.setActionContext = function (newContext) {
        this._context = newContext;
    };
    BaseActionViewItem.prototype.render = function (container) {
        var _this = this;
        var element = this.element = container;
        this._register(Gesture.addTarget(container));
        var enableDragging = this.options && this.options.draggable;
        if (enableDragging) {
            container.draggable = true;
            if (isFirefox) {
                // Firefox: requires to set a text data transfer to get going
                this._register(addDisposableListener(container, EventType.DRAG_START, function (e) { var _a; return (_a = e.dataTransfer) === null || _a === void 0 ? void 0 : _a.setData(DataTransfers.TEXT, _this._action.label); }));
            }
        }
        this._register(addDisposableListener(element, TouchEventType.Tap, function (e) { return _this.onClick(e); }));
        this._register(addDisposableListener(element, EventType.MOUSE_DOWN, function (e) {
            if (!enableDragging) {
                EventHelper.stop(e, true); // do not run when dragging is on because that would disable it
            }
            if (_this._action.enabled && e.button === 0) {
                element.classList.add('active');
            }
        }));
        if (platform.isMacintosh) {
            // macOS: allow to trigger the button when holding Ctrl+key and pressing the
            // main mouse button. This is for scenarios where e.g. some interaction forces
            // the Ctrl+key to be pressed and hold but the user still wants to interact
            // with the actions (for example quick access in quick navigation mode).
            this._register(addDisposableListener(element, EventType.CONTEXT_MENU, function (e) {
                if (e.button === 0 && e.ctrlKey === true) {
                    _this.onClick(e);
                }
            }));
        }
        this._register(addDisposableListener(element, EventType.CLICK, function (e) {
            EventHelper.stop(e, true);
            // menus do not use the click event
            if (!(_this.options && _this.options.isMenu)) {
                platform.setImmediate(function () { return _this.onClick(e); });
            }
        }));
        this._register(addDisposableListener(element, EventType.DBLCLICK, function (e) {
            EventHelper.stop(e, true);
        }));
        [EventType.MOUSE_UP, EventType.MOUSE_OUT].forEach(function (event) {
            _this._register(addDisposableListener(element, event, function (e) {
                EventHelper.stop(e);
                element.classList.remove('active');
            }));
        });
    };
    BaseActionViewItem.prototype.onClick = function (event) {
        var _a;
        EventHelper.stop(event, true);
        var context = types.isUndefinedOrNull(this._context) ? ((_a = this.options) === null || _a === void 0 ? void 0 : _a.useEventAsContext) ? event : undefined : this._context;
        this.actionRunner.run(this._action, context);
    };
    // Only set the tabIndex on the element once it is about to get focused
    // That way this element wont be a tab stop when it is not needed #106441
    BaseActionViewItem.prototype.focus = function () {
        if (this.element) {
            this.element.tabIndex = 0;
            this.element.focus();
            this.element.classList.add('focused');
        }
    };
    BaseActionViewItem.prototype.blur = function () {
        if (this.element) {
            this.element.blur();
            this.element.tabIndex = -1;
            this.element.classList.remove('focused');
        }
    };
    BaseActionViewItem.prototype.setFocusable = function (focusable) {
        if (this.element) {
            this.element.tabIndex = focusable ? 0 : -1;
        }
    };
    Object.defineProperty(BaseActionViewItem.prototype, "trapsArrowNavigation", {
        get: function () {
            return false;
        },
        enumerable: false,
        configurable: true
    });
    BaseActionViewItem.prototype.updateEnabled = function () {
        // implement in subclass
    };
    BaseActionViewItem.prototype.updateLabel = function () {
        // implement in subclass
    };
    BaseActionViewItem.prototype.updateTooltip = function () {
        // implement in subclass
    };
    BaseActionViewItem.prototype.updateClass = function () {
        // implement in subclass
    };
    BaseActionViewItem.prototype.updateChecked = function () {
        // implement in subclass
    };
    BaseActionViewItem.prototype.dispose = function () {
        if (this.element) {
            this.element.remove();
            this.element = undefined;
        }
        _super.prototype.dispose.call(this);
    };
    return BaseActionViewItem;
}(Disposable));
export { BaseActionViewItem };
var ActionViewItem = /** @class */ (function (_super) {
    __extends(ActionViewItem, _super);
    function ActionViewItem(context, action, options) {
        if (options === void 0) { options = {}; }
        var _this = _super.call(this, context, action, options) || this;
        _this.options = options;
        _this.options.icon = options.icon !== undefined ? options.icon : false;
        _this.options.label = options.label !== undefined ? options.label : true;
        _this.cssClass = '';
        return _this;
    }
    ActionViewItem.prototype.render = function (container) {
        _super.prototype.render.call(this, container);
        if (this.element) {
            this.label = append(this.element, $('a.action-label'));
        }
        if (this.label) {
            if (this._action.id === Separator.ID) {
                this.label.setAttribute('role', 'presentation'); // A separator is a presentation item
            }
            else {
                if (this.options.isMenu) {
                    this.label.setAttribute('role', 'menuitem');
                }
                else {
                    this.label.setAttribute('role', 'button');
                }
            }
        }
        if (this.options.label && this.options.keybinding && this.element) {
            append(this.element, $('span.keybinding')).textContent = this.options.keybinding;
        }
        this.updateClass();
        this.updateLabel();
        this.updateTooltip();
        this.updateEnabled();
        this.updateChecked();
    };
    // Only set the tabIndex on the element once it is about to get focused
    // That way this element wont be a tab stop when it is not needed #106441
    ActionViewItem.prototype.focus = function () {
        if (this.label) {
            this.label.tabIndex = 0;
            this.label.focus();
        }
    };
    ActionViewItem.prototype.blur = function () {
        if (this.label) {
            this.label.tabIndex = -1;
        }
    };
    ActionViewItem.prototype.setFocusable = function (focusable) {
        if (this.label) {
            this.label.tabIndex = focusable ? 0 : -1;
        }
    };
    ActionViewItem.prototype.updateLabel = function () {
        if (this.options.label && this.label) {
            this.label.textContent = this.getAction().label;
        }
    };
    ActionViewItem.prototype.updateTooltip = function () {
        var title = null;
        if (this.getAction().tooltip) {
            title = this.getAction().tooltip;
        }
        else if (!this.options.label && this.getAction().label && this.options.icon) {
            title = this.getAction().label;
            if (this.options.keybinding) {
                title = nls.localize({ key: 'titleLabel', comment: ['action title', 'action keybinding'] }, "{0} ({1})", title, this.options.keybinding);
            }
        }
        if (title && this.label) {
            this.label.title = title;
        }
    };
    ActionViewItem.prototype.updateClass = function () {
        var _a, _b;
        if (this.cssClass && this.label) {
            (_a = this.label.classList).remove.apply(_a, __spreadArray([], __read(this.cssClass.split(' '))));
        }
        if (this.options.icon) {
            this.cssClass = this.getAction().class;
            if (this.label) {
                this.label.classList.add('codicon');
                if (this.cssClass) {
                    (_b = this.label.classList).add.apply(_b, __spreadArray([], __read(this.cssClass.split(' '))));
                }
            }
            this.updateEnabled();
        }
        else {
            if (this.label) {
                this.label.classList.remove('codicon');
            }
        }
    };
    ActionViewItem.prototype.updateEnabled = function () {
        if (this.getAction().enabled) {
            if (this.label) {
                this.label.removeAttribute('aria-disabled');
                this.label.classList.remove('disabled');
            }
            if (this.element) {
                this.element.classList.remove('disabled');
            }
        }
        else {
            if (this.label) {
                this.label.setAttribute('aria-disabled', 'true');
                this.label.classList.add('disabled');
            }
            if (this.element) {
                this.element.classList.add('disabled');
            }
        }
    };
    ActionViewItem.prototype.updateChecked = function () {
        if (this.label) {
            if (this.getAction().checked) {
                this.label.classList.add('checked');
            }
            else {
                this.label.classList.remove('checked');
            }
        }
    };
    return ActionViewItem;
}(BaseActionViewItem));
export { ActionViewItem };
var SelectActionViewItem = /** @class */ (function (_super) {
    __extends(SelectActionViewItem, _super);
    function SelectActionViewItem(ctx, action, options, selected, contextViewProvider, selectBoxOptions) {
        var _this = _super.call(this, ctx, action) || this;
        _this.selectBox = new SelectBox(options, selected, contextViewProvider, undefined, selectBoxOptions);
        _this._register(_this.selectBox);
        _this.registerListeners();
        return _this;
    }
    SelectActionViewItem.prototype.setOptions = function (options, selected) {
        this.selectBox.setOptions(options, selected);
    };
    SelectActionViewItem.prototype.select = function (index) {
        this.selectBox.select(index);
    };
    SelectActionViewItem.prototype.registerListeners = function () {
        var _this = this;
        this._register(this.selectBox.onDidSelect(function (e) {
            _this.actionRunner.run(_this._action, _this.getActionContext(e.selected, e.index));
        }));
    };
    SelectActionViewItem.prototype.getActionContext = function (option, index) {
        return option;
    };
    SelectActionViewItem.prototype.focus = function () {
        if (this.selectBox) {
            this.selectBox.focus();
        }
    };
    SelectActionViewItem.prototype.blur = function () {
        if (this.selectBox) {
            this.selectBox.blur();
        }
    };
    SelectActionViewItem.prototype.render = function (container) {
        this.selectBox.render(container);
    };
    return SelectActionViewItem;
}(BaseActionViewItem));
export { SelectActionViewItem };
