"use strict";
/********************************************************************************
 * Copyright (C) 2020 Ericsson and others.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 ********************************************************************************/
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.ElectronMessagingContribution = void 0;
const electron_1 = require("@theia/electron/shared/electron");
const inversify_1 = require("inversify");
const connection_1 = require("vscode-ws-jsonrpc/lib/socket/connection");
const contribution_provider_1 = require("../../common/contribution-provider");
const web_socket_channel_1 = require("../../common/messaging/web-socket-channel");
const messaging_contribution_1 = require("../../node/messaging/messaging-contribution");
const logger_1 = require("../../node/messaging/logger");
const electron_connection_handler_1 = require("../../electron-common/messaging/electron-connection-handler");
const electron_messaging_service_1 = require("./electron-messaging-service");
/**
 * This component replicates the role filled by `MessagingContribution` but for Electron.
 * Unlike the WebSocket based implementation, we do not expect to receive
 * connection events. Instead, we'll create channels based on incoming `open`
 * events on the `ipcMain` channel.
 *
 * This component allows communication between renderer process (frontend) and electron main process.
 */
let ElectronMessagingContribution = class ElectronMessagingContribution {
    constructor() {
        this.channelHandlers = new messaging_contribution_1.MessagingContribution.ConnectionHandlers();
        this.windowChannels = new Map();
    }
    init() {
        electron_1.ipcMain.on(electron_connection_handler_1.THEIA_ELECTRON_IPC_CHANNEL_NAME, (event, data) => {
            this.handleIpcMessage(event, data);
        });
    }
    onStart() {
        for (const contribution of this.messagingContributions.getContributions()) {
            contribution.configure(this);
        }
        for (const connectionHandler of this.connectionHandlers.getContributions()) {
            this.channelHandlers.push(connectionHandler.path, (params, channel) => {
                const connection = connection_1.createWebSocketConnection(channel, new logger_1.ConsoleLogger());
                connectionHandler.onConnection(connection);
            });
        }
    }
    listen(spec, callback) {
        this.ipcChannel(spec, (params, channel) => {
            const connection = connection_1.createWebSocketConnection(channel, new logger_1.ConsoleLogger());
            callback(params, connection);
        });
    }
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    ipcChannel(spec, callback) {
        this.channelHandlers.push(spec, callback);
    }
    handleIpcMessage(event, data) {
        const sender = event.sender;
        try {
            // Get the channel map for a given window id
            let channels = this.windowChannels.get(sender.id);
            if (!channels) {
                this.windowChannels.set(sender.id, channels = new Map());
            }
            // Start parsing the message to extract the channel id and route
            const message = JSON.parse(data.toString());
            // Someone wants to open a logical channel
            if (message.kind === 'open') {
                const { id, path } = message;
                const channel = this.createChannel(id, sender);
                if (this.channelHandlers.route(path, channel)) {
                    channel.ready();
                    channels.set(id, channel);
                    channel.onClose(() => channels.delete(id));
                }
                else {
                    console.error('Cannot find a service for the path: ' + path);
                }
            }
            else {
                const { id } = message;
                const channel = channels.get(id);
                if (channel) {
                    channel.handleMessage(message);
                }
                else {
                    console.error('The ipc channel does not exist', id);
                }
            }
            const close = () => {
                for (const channel of Array.from(channels.values())) {
                    channel.close(undefined, 'webContent destroyed');
                }
                channels.clear();
            };
            sender.once('did-navigate', close); // When refreshing the browser window.
            sender.once('destroyed', close); // When closing the browser window.
        }
        catch (error) {
            console.error('IPC: Failed to handle message', { error, data });
        }
    }
    createChannel(id, sender) {
        return new web_socket_channel_1.WebSocketChannel(id, content => {
            if (!sender.isDestroyed()) {
                sender.send(electron_connection_handler_1.THEIA_ELECTRON_IPC_CHANNEL_NAME, content);
            }
        });
    }
};
__decorate([
    inversify_1.inject(contribution_provider_1.ContributionProvider),
    inversify_1.named(electron_messaging_service_1.ElectronMessagingService.Contribution),
    __metadata("design:type", Object)
], ElectronMessagingContribution.prototype, "messagingContributions", void 0);
__decorate([
    inversify_1.inject(contribution_provider_1.ContributionProvider),
    inversify_1.named(electron_connection_handler_1.ElectronConnectionHandler),
    __metadata("design:type", Object)
], ElectronMessagingContribution.prototype, "connectionHandlers", void 0);
__decorate([
    inversify_1.postConstruct(),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", []),
    __metadata("design:returntype", void 0)
], ElectronMessagingContribution.prototype, "init", null);
ElectronMessagingContribution = __decorate([
    inversify_1.injectable()
], ElectronMessagingContribution);
exports.ElectronMessagingContribution = ElectronMessagingContribution;
//# sourceMappingURL=electron-messaging-contribution.js.map