/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
import { Event, Emitter } from '../../../base/common/event';
import { DisposableStore } from '../../../base/common/lifecycle';
import { Dimension } from '../../../base/browser/dom';
import { OrthogonalEdge, Sash } from '../../../base/browser/ui/sash/sash';
var ResizableHTMLElement = /** @class */ (function () {
    function ResizableHTMLElement() {
        var _this = this;
        this._onDidWillResize = new Emitter();
        this.onDidWillResize = this._onDidWillResize.event;
        this._onDidResize = new Emitter();
        this.onDidResize = this._onDidResize.event;
        this._sashListener = new DisposableStore();
        this._size = new Dimension(0, 0);
        this._minSize = new Dimension(0, 0);
        this._maxSize = new Dimension(Number.MAX_SAFE_INTEGER, Number.MAX_SAFE_INTEGER);
        this.domNode = document.createElement('div');
        this._eastSash = new Sash(this.domNode, { getVerticalSashLeft: function () { return _this._size.width; } }, { orientation: 0 /* VERTICAL */ });
        this._westSash = new Sash(this.domNode, { getVerticalSashLeft: function () { return 0; } }, { orientation: 0 /* VERTICAL */ });
        this._northSash = new Sash(this.domNode, { getHorizontalSashTop: function () { return 0; } }, { orientation: 1 /* HORIZONTAL */, orthogonalEdge: OrthogonalEdge.North });
        this._southSash = new Sash(this.domNode, { getHorizontalSashTop: function () { return _this._size.height; } }, { orientation: 1 /* HORIZONTAL */, orthogonalEdge: OrthogonalEdge.South });
        this._northSash.orthogonalStartSash = this._westSash;
        this._northSash.orthogonalEndSash = this._eastSash;
        this._southSash.orthogonalStartSash = this._westSash;
        this._southSash.orthogonalEndSash = this._eastSash;
        var currentSize;
        var deltaY = 0;
        var deltaX = 0;
        this._sashListener.add(Event.any(this._northSash.onDidStart, this._eastSash.onDidStart, this._southSash.onDidStart, this._westSash.onDidStart)(function () {
            if (currentSize === undefined) {
                _this._onDidWillResize.fire();
                currentSize = _this._size;
                deltaY = 0;
                deltaX = 0;
            }
        }));
        this._sashListener.add(Event.any(this._northSash.onDidEnd, this._eastSash.onDidEnd, this._southSash.onDidEnd, this._westSash.onDidEnd)(function () {
            if (currentSize !== undefined) {
                currentSize = undefined;
                deltaY = 0;
                deltaX = 0;
                _this._onDidResize.fire({ dimension: _this._size, done: true });
            }
        }));
        this._sashListener.add(this._eastSash.onDidChange(function (e) {
            if (currentSize) {
                deltaX = e.currentX - e.startX;
                _this.layout(currentSize.height + deltaY, currentSize.width + deltaX);
                _this._onDidResize.fire({ dimension: _this._size, done: false, east: true });
            }
        }));
        this._sashListener.add(this._westSash.onDidChange(function (e) {
            if (currentSize) {
                deltaX = -(e.currentX - e.startX);
                _this.layout(currentSize.height + deltaY, currentSize.width + deltaX);
                _this._onDidResize.fire({ dimension: _this._size, done: false, west: true });
            }
        }));
        this._sashListener.add(this._northSash.onDidChange(function (e) {
            if (currentSize) {
                deltaY = -(e.currentY - e.startY);
                _this.layout(currentSize.height + deltaY, currentSize.width + deltaX);
                _this._onDidResize.fire({ dimension: _this._size, done: false, north: true });
            }
        }));
        this._sashListener.add(this._southSash.onDidChange(function (e) {
            if (currentSize) {
                deltaY = e.currentY - e.startY;
                _this.layout(currentSize.height + deltaY, currentSize.width + deltaX);
                _this._onDidResize.fire({ dimension: _this._size, done: false, south: true });
            }
        }));
        this._sashListener.add(Event.any(this._eastSash.onDidReset, this._westSash.onDidReset)(function (e) {
            if (_this._preferredSize) {
                _this.layout(_this._size.height, _this._preferredSize.width);
                _this._onDidResize.fire({ dimension: _this._size, done: true });
            }
        }));
        this._sashListener.add(Event.any(this._northSash.onDidReset, this._southSash.onDidReset)(function (e) {
            if (_this._preferredSize) {
                _this.layout(_this._preferredSize.height, _this._size.width);
                _this._onDidResize.fire({ dimension: _this._size, done: true });
            }
        }));
    }
    ResizableHTMLElement.prototype.dispose = function () {
        this._northSash.dispose();
        this._southSash.dispose();
        this._eastSash.dispose();
        this._westSash.dispose();
        this._sashListener.dispose();
        this.domNode.remove();
    };
    ResizableHTMLElement.prototype.enableSashes = function (north, east, south, west) {
        this._northSash.state = north ? 3 /* Enabled */ : 0 /* Disabled */;
        this._eastSash.state = east ? 3 /* Enabled */ : 0 /* Disabled */;
        this._southSash.state = south ? 3 /* Enabled */ : 0 /* Disabled */;
        this._westSash.state = west ? 3 /* Enabled */ : 0 /* Disabled */;
    };
    ResizableHTMLElement.prototype.layout = function (height, width) {
        if (height === void 0) { height = this.size.height; }
        if (width === void 0) { width = this.size.width; }
        var _a = this._minSize, minHeight = _a.height, minWidth = _a.width;
        var _b = this._maxSize, maxHeight = _b.height, maxWidth = _b.width;
        height = Math.max(minHeight, Math.min(maxHeight, height));
        width = Math.max(minWidth, Math.min(maxWidth, width));
        var newSize = new Dimension(width, height);
        if (!Dimension.equals(newSize, this._size)) {
            this.domNode.style.height = height + 'px';
            this.domNode.style.width = width + 'px';
            this._size = newSize;
            this._northSash.layout();
            this._eastSash.layout();
            this._southSash.layout();
            this._westSash.layout();
        }
    };
    Object.defineProperty(ResizableHTMLElement.prototype, "size", {
        get: function () {
            return this._size;
        },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(ResizableHTMLElement.prototype, "maxSize", {
        get: function () {
            return this._maxSize;
        },
        set: function (value) {
            this._maxSize = value;
        },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(ResizableHTMLElement.prototype, "minSize", {
        get: function () {
            return this._minSize;
        },
        set: function (value) {
            this._minSize = value;
        },
        enumerable: false,
        configurable: true
    });
    Object.defineProperty(ResizableHTMLElement.prototype, "preferredSize", {
        get: function () {
            return this._preferredSize;
        },
        set: function (value) {
            this._preferredSize = value;
        },
        enumerable: false,
        configurable: true
    });
    return ResizableHTMLElement;
}());
export { ResizableHTMLElement };
