/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
var __read = (this && this.__read) || function (o, n) {
    var m = typeof Symbol === "function" && o[Symbol.iterator];
    if (!m) return o;
    var i = m.call(o), r, ar = [], e;
    try {
        while ((n === void 0 || n-- > 0) && !(r = i.next()).done) ar.push(r.value);
    }
    catch (error) { e = { error: error }; }
    finally {
        try {
            if (r && !r.done && (m = i["return"])) m.call(i);
        }
        finally { if (e) throw e.error; }
    }
    return ar;
};
import { OverviewRulerLane } from '../../common/model';
import { themeColorFromId } from '../../../platform/theme/common/themeService';
import { overviewRulerRangeHighlight } from '../../common/view/editorColorRegistry';
import { DisposableStore, toDisposable, MutableDisposable } from '../../../base/common/lifecycle';
import { isDiffEditor, getCodeEditor } from '../../browser/editorBrowser';
import { withNullAsUndefined } from '../../../base/common/types';
import { once } from '../../../base/common/functional';
/**
 * A reusable quick access provider for the editor with support
 * for adding decorations for navigating in the currently active file
 * (for example "Go to line", "Go to symbol").
 */
var AbstractEditorNavigationQuickAccessProvider = /** @class */ (function () {
    function AbstractEditorNavigationQuickAccessProvider(options) {
        this.options = options;
        //#endregion
        //#region Decorations Utils
        this.rangeHighlightDecorationId = undefined;
    }
    //#region Provider methods
    AbstractEditorNavigationQuickAccessProvider.prototype.provide = function (picker, token) {
        var _this = this;
        var _a;
        var disposables = new DisposableStore();
        // Apply options if any
        picker.canAcceptInBackground = !!((_a = this.options) === null || _a === void 0 ? void 0 : _a.canAcceptInBackground);
        // Disable filtering & sorting, we control the results
        picker.matchOnLabel = picker.matchOnDescription = picker.matchOnDetail = picker.sortByLabel = false;
        // Provide based on current active editor
        var pickerDisposable = disposables.add(new MutableDisposable());
        pickerDisposable.value = this.doProvide(picker, token);
        // Re-create whenever the active editor changes
        disposables.add(this.onDidActiveTextEditorControlChange(function () {
            // Clear old
            pickerDisposable.value = undefined;
            // Add new
            pickerDisposable.value = _this.doProvide(picker, token);
        }));
        return disposables;
    };
    AbstractEditorNavigationQuickAccessProvider.prototype.doProvide = function (picker, token) {
        var _this = this;
        var disposables = new DisposableStore();
        // With text control
        var editor = this.activeTextEditorControl;
        if (editor && this.canProvideWithTextEditor(editor)) {
            var context_1 = { editor: editor };
            // Restore any view state if this picker was closed
            // without actually going to a line
            var codeEditor = getCodeEditor(editor);
            if (codeEditor) {
                // Remember view state and update it when the cursor position
                // changes even later because it could be that the user has
                // configured quick access to remain open when focus is lost and
                // we always want to restore the current location.
                var lastKnownEditorViewState_1 = withNullAsUndefined(editor.saveViewState());
                disposables.add(codeEditor.onDidChangeCursorPosition(function () {
                    lastKnownEditorViewState_1 = withNullAsUndefined(editor.saveViewState());
                }));
                context_1.restoreViewState = function () {
                    if (lastKnownEditorViewState_1 && editor === _this.activeTextEditorControl) {
                        editor.restoreViewState(lastKnownEditorViewState_1);
                    }
                };
                disposables.add(once(token.onCancellationRequested)(function () { var _a; return (_a = context_1.restoreViewState) === null || _a === void 0 ? void 0 : _a.call(context_1); }));
            }
            // Clean up decorations on dispose
            disposables.add(toDisposable(function () { return _this.clearDecorations(editor); }));
            // Ask subclass for entries
            disposables.add(this.provideWithTextEditor(context_1, picker, token));
        }
        // Without text control
        else {
            disposables.add(this.provideWithoutTextEditor(picker, token));
        }
        return disposables;
    };
    /**
     * Subclasses to implement if they can operate on the text editor.
     */
    AbstractEditorNavigationQuickAccessProvider.prototype.canProvideWithTextEditor = function (editor) {
        return true;
    };
    AbstractEditorNavigationQuickAccessProvider.prototype.gotoLocation = function (_a, options) {
        var editor = _a.editor;
        editor.setSelection(options.range);
        editor.revealRangeInCenter(options.range, 0 /* Smooth */);
        if (!options.preserveFocus) {
            editor.focus();
        }
    };
    AbstractEditorNavigationQuickAccessProvider.prototype.getModel = function (editor) {
        var _a;
        return isDiffEditor(editor) ?
            (_a = editor.getModel()) === null || _a === void 0 ? void 0 : _a.modified :
            editor.getModel();
    };
    AbstractEditorNavigationQuickAccessProvider.prototype.addDecorations = function (editor, range) {
        var _this = this;
        editor.changeDecorations(function (changeAccessor) {
            // Reset old decorations if any
            var deleteDecorations = [];
            if (_this.rangeHighlightDecorationId) {
                deleteDecorations.push(_this.rangeHighlightDecorationId.overviewRulerDecorationId);
                deleteDecorations.push(_this.rangeHighlightDecorationId.rangeHighlightId);
                _this.rangeHighlightDecorationId = undefined;
            }
            // Add new decorations for the range
            var newDecorations = [
                // highlight the entire line on the range
                {
                    range: range,
                    options: {
                        className: 'rangeHighlight',
                        isWholeLine: true
                    }
                },
                // also add overview ruler highlight
                {
                    range: range,
                    options: {
                        overviewRuler: {
                            color: themeColorFromId(overviewRulerRangeHighlight),
                            position: OverviewRulerLane.Full
                        }
                    }
                }
            ];
            var _a = __read(changeAccessor.deltaDecorations(deleteDecorations, newDecorations), 2), rangeHighlightId = _a[0], overviewRulerDecorationId = _a[1];
            _this.rangeHighlightDecorationId = { rangeHighlightId: rangeHighlightId, overviewRulerDecorationId: overviewRulerDecorationId };
        });
    };
    AbstractEditorNavigationQuickAccessProvider.prototype.clearDecorations = function (editor) {
        var rangeHighlightDecorationId = this.rangeHighlightDecorationId;
        if (rangeHighlightDecorationId) {
            editor.changeDecorations(function (changeAccessor) {
                changeAccessor.deltaDecorations([
                    rangeHighlightDecorationId.overviewRulerDecorationId,
                    rangeHighlightDecorationId.rangeHighlightId
                ], []);
            });
            this.rangeHighlightDecorationId = undefined;
        }
    };
    return AbstractEditorNavigationQuickAccessProvider;
}());
export { AbstractEditorNavigationQuickAccessProvider };
