/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
var __read = (this && this.__read) || function (o, n) {
    var m = typeof Symbol === "function" && o[Symbol.iterator];
    if (!m) return o;
    var i = m.call(o), r, ar = [], e;
    try {
        while ((n === void 0 || n-- > 0) && !(r = i.next()).done) ar.push(r.value);
    }
    catch (error) { e = { error: error }; }
    finally {
        try {
            if (r && !r.done && (m = i["return"])) m.call(i);
        }
        finally { if (e) throw e.error; }
    }
    return ar;
};
var __spreadArray = (this && this.__spreadArray) || function (to, from) {
    for (var i = 0, il = from.length, j = to.length; i < il; i++, j++)
        to[j] = from[i];
    return to;
};
var __values = (this && this.__values) || function(o) {
    var s = typeof Symbol === "function" && Symbol.iterator, m = s && o[s], i = 0;
    if (m) return m.call(o);
    if (o && typeof o.length === "number") return {
        next: function () {
            if (o && i >= o.length) o = void 0;
            return { value: o && o[i++], done: !o };
        }
    };
    throw new TypeError(s ? "Object is not iterable." : "Symbol.iterator is not defined.");
};
import { TreeError } from './tree';
import { tail2 } from '../../../common/arrays';
import { LcsDiff } from '../../../common/diff/diff';
import { Emitter, EventBufferer } from '../../../common/event';
import { Iterable } from '../../../common/iterator';
export function isFilterResult(obj) {
    return typeof obj === 'object' && 'visibility' in obj && 'data' in obj;
}
export function getVisibleState(visibility) {
    switch (visibility) {
        case true: return 1 /* Visible */;
        case false: return 0 /* Hidden */;
        default: return visibility;
    }
}
function isCollapsibleStateUpdate(update) {
    return typeof update.collapsible === 'boolean';
}
var IndexTreeModel = /** @class */ (function () {
    function IndexTreeModel(user, list, rootElement, options) {
        if (options === void 0) { options = {}; }
        this.user = user;
        this.list = list;
        this.rootRef = [];
        this.eventBufferer = new EventBufferer();
        this._onDidChangeCollapseState = new Emitter();
        this.onDidChangeCollapseState = this.eventBufferer.wrapEvent(this._onDidChangeCollapseState.event);
        this._onDidChangeRenderNodeCount = new Emitter();
        this.onDidChangeRenderNodeCount = this.eventBufferer.wrapEvent(this._onDidChangeRenderNodeCount.event);
        this._onDidSplice = new Emitter();
        this.onDidSplice = this._onDidSplice.event;
        this.collapseByDefault = typeof options.collapseByDefault === 'undefined' ? false : options.collapseByDefault;
        this.filter = options.filter;
        this.autoExpandSingleChildren = typeof options.autoExpandSingleChildren === 'undefined' ? false : options.autoExpandSingleChildren;
        this.root = {
            parent: undefined,
            element: rootElement,
            children: [],
            depth: 0,
            visibleChildrenCount: 0,
            visibleChildIndex: -1,
            collapsible: false,
            collapsed: false,
            renderNodeCount: 0,
            visibility: 1 /* Visible */,
            visible: true,
            filterData: undefined
        };
    }
    IndexTreeModel.prototype.splice = function (location, deleteCount, toInsert, options) {
        if (toInsert === void 0) { toInsert = Iterable.empty(); }
        if (options === void 0) { options = {}; }
        if (location.length === 0) {
            throw new TreeError(this.user, 'Invalid tree location');
        }
        if (options.diffIdentityProvider) {
            this.spliceSmart(options.diffIdentityProvider, location, deleteCount, toInsert, options);
        }
        else {
            this.spliceSimple(location, deleteCount, toInsert, options);
        }
    };
    IndexTreeModel.prototype.spliceSmart = function (identity, location, deleteCount, toInsertIterable, options, recurseLevels) {
        var e_1, _a;
        var _this = this;
        var _b;
        if (toInsertIterable === void 0) { toInsertIterable = Iterable.empty(); }
        if (recurseLevels === void 0) { recurseLevels = (_b = options.diffDepth) !== null && _b !== void 0 ? _b : 0; }
        var parentNode = this.getParentNodeWithListIndex(location).parentNode;
        var toInsert = __spreadArray([], __read(toInsertIterable));
        var index = location[location.length - 1];
        var diff = new LcsDiff({ getElements: function () { return parentNode.children.map(function (e) { return identity.getId(e.element).toString(); }); } }, {
            getElements: function () { return __spreadArray(__spreadArray(__spreadArray([], __read(parentNode.children.slice(0, index))), __read(toInsert)), __read(parentNode.children.slice(index + deleteCount))).map(function (e) { return identity.getId(e.element).toString(); }); }
        }).ComputeDiff(false);
        // if we were given a 'best effort' diff, use default behavior
        if (diff.quitEarly) {
            return this.spliceSimple(location, deleteCount, toInsert, options);
        }
        var locationPrefix = location.slice(0, -1);
        var recurseSplice = function (fromOriginal, fromModified, count) {
            if (recurseLevels > 0) {
                for (var i = 0; i < count; i++) {
                    fromOriginal--;
                    fromModified--;
                    _this.spliceSmart(identity, __spreadArray(__spreadArray([], __read(locationPrefix)), [fromOriginal, 0]), Number.MAX_SAFE_INTEGER, toInsert[fromModified].children, options, recurseLevels - 1);
                }
            }
        };
        var lastStartO = Math.min(parentNode.children.length, index + deleteCount);
        var lastStartM = toInsert.length;
        try {
            for (var _c = __values(diff.changes.sort(function (a, b) { return b.originalStart - a.originalStart; })), _d = _c.next(); !_d.done; _d = _c.next()) {
                var change = _d.value;
                recurseSplice(lastStartO, lastStartM, lastStartO - (change.originalStart + change.originalLength));
                lastStartO = change.originalStart;
                lastStartM = change.modifiedStart - index;
                this.spliceSimple(__spreadArray(__spreadArray([], __read(locationPrefix)), [lastStartO]), change.originalLength, Iterable.slice(toInsert, lastStartM, lastStartM + change.modifiedLength), options);
            }
        }
        catch (e_1_1) { e_1 = { error: e_1_1 }; }
        finally {
            try {
                if (_d && !_d.done && (_a = _c.return)) _a.call(_c);
            }
            finally { if (e_1) throw e_1.error; }
        }
        // at this point, startO === startM === count since any remaining prefix should match
        recurseSplice(lastStartO, lastStartM, lastStartO);
    };
    IndexTreeModel.prototype.spliceSimple = function (location, deleteCount, toInsert, _a) {
        var e_2, _b, _c, e_3, _d;
        var _this = this;
        if (toInsert === void 0) { toInsert = Iterable.empty(); }
        var onDidCreateNode = _a.onDidCreateNode, onDidDeleteNode = _a.onDidDeleteNode;
        var _e = this.getParentNodeWithListIndex(location), parentNode = _e.parentNode, listIndex = _e.listIndex, revealed = _e.revealed, visible = _e.visible;
        var treeListElementsToInsert = [];
        var nodesToInsertIterator = Iterable.map(toInsert, function (el) { return _this.createTreeNode(el, parentNode, parentNode.visible ? 1 /* Visible */ : 0 /* Hidden */, revealed, treeListElementsToInsert, onDidCreateNode); });
        var lastIndex = location[location.length - 1];
        var lastHadChildren = parentNode.children.length > 0;
        // figure out what's the visible child start index right before the
        // splice point
        var visibleChildStartIndex = 0;
        for (var i = lastIndex; i >= 0 && i < parentNode.children.length; i--) {
            var child = parentNode.children[i];
            if (child.visible) {
                visibleChildStartIndex = child.visibleChildIndex;
                break;
            }
        }
        var nodesToInsert = [];
        var insertedVisibleChildrenCount = 0;
        var renderNodeCount = 0;
        try {
            for (var nodesToInsertIterator_1 = __values(nodesToInsertIterator), nodesToInsertIterator_1_1 = nodesToInsertIterator_1.next(); !nodesToInsertIterator_1_1.done; nodesToInsertIterator_1_1 = nodesToInsertIterator_1.next()) {
                var child = nodesToInsertIterator_1_1.value;
                nodesToInsert.push(child);
                renderNodeCount += child.renderNodeCount;
                if (child.visible) {
                    child.visibleChildIndex = visibleChildStartIndex + insertedVisibleChildrenCount++;
                }
            }
        }
        catch (e_2_1) { e_2 = { error: e_2_1 }; }
        finally {
            try {
                if (nodesToInsertIterator_1_1 && !nodesToInsertIterator_1_1.done && (_b = nodesToInsertIterator_1.return)) _b.call(nodesToInsertIterator_1);
            }
            finally { if (e_2) throw e_2.error; }
        }
        var deletedNodes = (_c = parentNode.children).splice.apply(_c, __spreadArray([lastIndex, deleteCount], __read(nodesToInsert)));
        // figure out what is the count of deleted visible children
        var deletedVisibleChildrenCount = 0;
        try {
            for (var deletedNodes_1 = __values(deletedNodes), deletedNodes_1_1 = deletedNodes_1.next(); !deletedNodes_1_1.done; deletedNodes_1_1 = deletedNodes_1.next()) {
                var child = deletedNodes_1_1.value;
                if (child.visible) {
                    deletedVisibleChildrenCount++;
                }
            }
        }
        catch (e_3_1) { e_3 = { error: e_3_1 }; }
        finally {
            try {
                if (deletedNodes_1_1 && !deletedNodes_1_1.done && (_d = deletedNodes_1.return)) _d.call(deletedNodes_1);
            }
            finally { if (e_3) throw e_3.error; }
        }
        // and adjust for all visible children after the splice point
        if (deletedVisibleChildrenCount !== 0) {
            for (var i = lastIndex + nodesToInsert.length; i < parentNode.children.length; i++) {
                var child = parentNode.children[i];
                if (child.visible) {
                    child.visibleChildIndex -= deletedVisibleChildrenCount;
                }
            }
        }
        // update parent's visible children count
        parentNode.visibleChildrenCount += insertedVisibleChildrenCount - deletedVisibleChildrenCount;
        if (revealed && visible) {
            var visibleDeleteCount = deletedNodes.reduce(function (r, node) { return r + (node.visible ? node.renderNodeCount : 0); }, 0);
            this._updateAncestorsRenderNodeCount(parentNode, renderNodeCount - visibleDeleteCount);
            this.list.splice(listIndex, visibleDeleteCount, treeListElementsToInsert);
        }
        if (deletedNodes.length > 0 && onDidDeleteNode) {
            var visit_1 = function (node) {
                onDidDeleteNode(node);
                node.children.forEach(visit_1);
            };
            deletedNodes.forEach(visit_1);
        }
        var currentlyHasChildren = parentNode.children.length > 0;
        if (lastHadChildren !== currentlyHasChildren) {
            this.setCollapsible(location.slice(0, -1), currentlyHasChildren);
        }
        this._onDidSplice.fire({ insertedNodes: nodesToInsert, deletedNodes: deletedNodes });
        var node = parentNode;
        while (node) {
            if (node.visibility === 2 /* Recurse */) {
                this.refilter();
                break;
            }
            node = node.parent;
        }
    };
    IndexTreeModel.prototype.rerender = function (location) {
        if (location.length === 0) {
            throw new TreeError(this.user, 'Invalid tree location');
        }
        var _a = this.getTreeNodeWithListIndex(location), node = _a.node, listIndex = _a.listIndex, revealed = _a.revealed;
        if (node.visible && revealed) {
            this.list.splice(listIndex, 1, [node]);
        }
    };
    IndexTreeModel.prototype.updateElementHeight = function (location, height) {
        if (location.length === 0) {
            throw new TreeError(this.user, 'Invalid tree location');
        }
        var listIndex = this.getTreeNodeWithListIndex(location).listIndex;
        this.list.updateElementHeight(listIndex, height);
    };
    IndexTreeModel.prototype.has = function (location) {
        return this.hasTreeNode(location);
    };
    IndexTreeModel.prototype.getListIndex = function (location) {
        var _a = this.getTreeNodeWithListIndex(location), listIndex = _a.listIndex, visible = _a.visible, revealed = _a.revealed;
        return visible && revealed ? listIndex : -1;
    };
    IndexTreeModel.prototype.getListRenderCount = function (location) {
        return this.getTreeNode(location).renderNodeCount;
    };
    IndexTreeModel.prototype.isCollapsible = function (location) {
        return this.getTreeNode(location).collapsible;
    };
    IndexTreeModel.prototype.setCollapsible = function (location, collapsible) {
        var _this = this;
        var node = this.getTreeNode(location);
        if (typeof collapsible === 'undefined') {
            collapsible = !node.collapsible;
        }
        var update = { collapsible: collapsible };
        return this.eventBufferer.bufferEvents(function () { return _this._setCollapseState(location, update); });
    };
    IndexTreeModel.prototype.isCollapsed = function (location) {
        return this.getTreeNode(location).collapsed;
    };
    IndexTreeModel.prototype.setCollapsed = function (location, collapsed, recursive) {
        var _this = this;
        var node = this.getTreeNode(location);
        if (typeof collapsed === 'undefined') {
            collapsed = !node.collapsed;
        }
        var update = { collapsed: collapsed, recursive: recursive || false };
        return this.eventBufferer.bufferEvents(function () { return _this._setCollapseState(location, update); });
    };
    IndexTreeModel.prototype._setCollapseState = function (location, update) {
        var _a = this.getTreeNodeWithListIndex(location), node = _a.node, listIndex = _a.listIndex, revealed = _a.revealed;
        var result = this._setListNodeCollapseState(node, listIndex, revealed, update);
        if (node !== this.root && this.autoExpandSingleChildren && result && !isCollapsibleStateUpdate(update) && node.collapsible && !node.collapsed && !update.recursive) {
            var onlyVisibleChildIndex = -1;
            for (var i = 0; i < node.children.length; i++) {
                var child = node.children[i];
                if (child.visible) {
                    if (onlyVisibleChildIndex > -1) {
                        onlyVisibleChildIndex = -1;
                        break;
                    }
                    else {
                        onlyVisibleChildIndex = i;
                    }
                }
            }
            if (onlyVisibleChildIndex > -1) {
                this._setCollapseState(__spreadArray(__spreadArray([], __read(location)), [onlyVisibleChildIndex]), update);
            }
        }
        return result;
    };
    IndexTreeModel.prototype._setListNodeCollapseState = function (node, listIndex, revealed, update) {
        var result = this._setNodeCollapseState(node, update, false);
        if (!revealed || !node.visible || !result) {
            return result;
        }
        var previousRenderNodeCount = node.renderNodeCount;
        var toInsert = this.updateNodeAfterCollapseChange(node);
        var deleteCount = previousRenderNodeCount - (listIndex === -1 ? 0 : 1);
        this.list.splice(listIndex + 1, deleteCount, toInsert.slice(1));
        return result;
    };
    IndexTreeModel.prototype._setNodeCollapseState = function (node, update, deep) {
        var e_4, _a;
        var result;
        if (node === this.root) {
            result = false;
        }
        else {
            if (isCollapsibleStateUpdate(update)) {
                result = node.collapsible !== update.collapsible;
                node.collapsible = update.collapsible;
            }
            else if (!node.collapsible) {
                result = false;
            }
            else {
                result = node.collapsed !== update.collapsed;
                node.collapsed = update.collapsed;
            }
            if (result) {
                this._onDidChangeCollapseState.fire({ node: node, deep: deep });
            }
        }
        if (!isCollapsibleStateUpdate(update) && update.recursive) {
            try {
                for (var _b = __values(node.children), _c = _b.next(); !_c.done; _c = _b.next()) {
                    var child = _c.value;
                    result = this._setNodeCollapseState(child, update, true) || result;
                }
            }
            catch (e_4_1) { e_4 = { error: e_4_1 }; }
            finally {
                try {
                    if (_c && !_c.done && (_a = _b.return)) _a.call(_b);
                }
                finally { if (e_4) throw e_4.error; }
            }
        }
        return result;
    };
    IndexTreeModel.prototype.expandTo = function (location) {
        var _this = this;
        this.eventBufferer.bufferEvents(function () {
            var node = _this.getTreeNode(location);
            while (node.parent) {
                node = node.parent;
                location = location.slice(0, location.length - 1);
                if (node.collapsed) {
                    _this._setCollapseState(location, { collapsed: false, recursive: false });
                }
            }
        });
    };
    IndexTreeModel.prototype.refilter = function () {
        var previousRenderNodeCount = this.root.renderNodeCount;
        var toInsert = this.updateNodeAfterFilterChange(this.root);
        this.list.splice(0, previousRenderNodeCount, toInsert);
    };
    IndexTreeModel.prototype.createTreeNode = function (treeElement, parent, parentVisibility, revealed, treeListElements, onDidCreateNode) {
        var e_5, _a;
        var _this = this;
        var node = {
            parent: parent,
            element: treeElement.element,
            children: [],
            depth: parent.depth + 1,
            visibleChildrenCount: 0,
            visibleChildIndex: -1,
            collapsible: typeof treeElement.collapsible === 'boolean' ? treeElement.collapsible : (typeof treeElement.collapsed !== 'undefined'),
            collapsed: typeof treeElement.collapsed === 'undefined' ? this.collapseByDefault : treeElement.collapsed,
            renderNodeCount: 1,
            visibility: 1 /* Visible */,
            visible: true,
            filterData: undefined
        };
        var visibility = this._filterNode(node, parentVisibility);
        node.visibility = visibility;
        if (revealed) {
            treeListElements.push(node);
        }
        var childElements = treeElement.children || Iterable.empty();
        var childRevealed = revealed && visibility !== 0 /* Hidden */ && !node.collapsed;
        var childNodes = Iterable.map(childElements, function (el) { return _this.createTreeNode(el, node, visibility, childRevealed, treeListElements, onDidCreateNode); });
        var visibleChildrenCount = 0;
        var renderNodeCount = 1;
        try {
            for (var childNodes_1 = __values(childNodes), childNodes_1_1 = childNodes_1.next(); !childNodes_1_1.done; childNodes_1_1 = childNodes_1.next()) {
                var child = childNodes_1_1.value;
                node.children.push(child);
                renderNodeCount += child.renderNodeCount;
                if (child.visible) {
                    child.visibleChildIndex = visibleChildrenCount++;
                }
            }
        }
        catch (e_5_1) { e_5 = { error: e_5_1 }; }
        finally {
            try {
                if (childNodes_1_1 && !childNodes_1_1.done && (_a = childNodes_1.return)) _a.call(childNodes_1);
            }
            finally { if (e_5) throw e_5.error; }
        }
        node.collapsible = node.collapsible || node.children.length > 0;
        node.visibleChildrenCount = visibleChildrenCount;
        node.visible = visibility === 2 /* Recurse */ ? visibleChildrenCount > 0 : (visibility === 1 /* Visible */);
        if (!node.visible) {
            node.renderNodeCount = 0;
            if (revealed) {
                treeListElements.pop();
            }
        }
        else if (!node.collapsed) {
            node.renderNodeCount = renderNodeCount;
        }
        if (onDidCreateNode) {
            onDidCreateNode(node);
        }
        return node;
    };
    IndexTreeModel.prototype.updateNodeAfterCollapseChange = function (node) {
        var previousRenderNodeCount = node.renderNodeCount;
        var result = [];
        this._updateNodeAfterCollapseChange(node, result);
        this._updateAncestorsRenderNodeCount(node.parent, result.length - previousRenderNodeCount);
        return result;
    };
    IndexTreeModel.prototype._updateNodeAfterCollapseChange = function (node, result) {
        var e_6, _a;
        if (node.visible === false) {
            return 0;
        }
        result.push(node);
        node.renderNodeCount = 1;
        if (!node.collapsed) {
            try {
                for (var _b = __values(node.children), _c = _b.next(); !_c.done; _c = _b.next()) {
                    var child = _c.value;
                    node.renderNodeCount += this._updateNodeAfterCollapseChange(child, result);
                }
            }
            catch (e_6_1) { e_6 = { error: e_6_1 }; }
            finally {
                try {
                    if (_c && !_c.done && (_a = _b.return)) _a.call(_b);
                }
                finally { if (e_6) throw e_6.error; }
            }
        }
        this._onDidChangeRenderNodeCount.fire(node);
        return node.renderNodeCount;
    };
    IndexTreeModel.prototype.updateNodeAfterFilterChange = function (node) {
        var previousRenderNodeCount = node.renderNodeCount;
        var result = [];
        this._updateNodeAfterFilterChange(node, node.visible ? 1 /* Visible */ : 0 /* Hidden */, result);
        this._updateAncestorsRenderNodeCount(node.parent, result.length - previousRenderNodeCount);
        return result;
    };
    IndexTreeModel.prototype._updateNodeAfterFilterChange = function (node, parentVisibility, result, revealed) {
        var e_7, _a;
        if (revealed === void 0) { revealed = true; }
        var visibility;
        if (node !== this.root) {
            visibility = this._filterNode(node, parentVisibility);
            if (visibility === 0 /* Hidden */) {
                node.visible = false;
                node.renderNodeCount = 0;
                return false;
            }
            if (revealed) {
                result.push(node);
            }
        }
        var resultStartLength = result.length;
        node.renderNodeCount = node === this.root ? 0 : 1;
        var hasVisibleDescendants = false;
        if (!node.collapsed || visibility !== 0 /* Hidden */) {
            var visibleChildIndex = 0;
            try {
                for (var _b = __values(node.children), _c = _b.next(); !_c.done; _c = _b.next()) {
                    var child = _c.value;
                    hasVisibleDescendants = this._updateNodeAfterFilterChange(child, visibility, result, revealed && !node.collapsed) || hasVisibleDescendants;
                    if (child.visible) {
                        child.visibleChildIndex = visibleChildIndex++;
                    }
                }
            }
            catch (e_7_1) { e_7 = { error: e_7_1 }; }
            finally {
                try {
                    if (_c && !_c.done && (_a = _b.return)) _a.call(_b);
                }
                finally { if (e_7) throw e_7.error; }
            }
            node.visibleChildrenCount = visibleChildIndex;
        }
        else {
            node.visibleChildrenCount = 0;
        }
        if (node !== this.root) {
            node.visible = visibility === 2 /* Recurse */ ? hasVisibleDescendants : (visibility === 1 /* Visible */);
        }
        if (!node.visible) {
            node.renderNodeCount = 0;
            if (revealed) {
                result.pop();
            }
        }
        else if (!node.collapsed) {
            node.renderNodeCount += result.length - resultStartLength;
        }
        this._onDidChangeRenderNodeCount.fire(node);
        return node.visible;
    };
    IndexTreeModel.prototype._updateAncestorsRenderNodeCount = function (node, diff) {
        if (diff === 0) {
            return;
        }
        while (node) {
            node.renderNodeCount += diff;
            this._onDidChangeRenderNodeCount.fire(node);
            node = node.parent;
        }
    };
    IndexTreeModel.prototype._filterNode = function (node, parentVisibility) {
        var result = this.filter ? this.filter.filter(node.element, parentVisibility) : 1 /* Visible */;
        if (typeof result === 'boolean') {
            node.filterData = undefined;
            return result ? 1 /* Visible */ : 0 /* Hidden */;
        }
        else if (isFilterResult(result)) {
            node.filterData = result.data;
            return getVisibleState(result.visibility);
        }
        else {
            node.filterData = undefined;
            return getVisibleState(result);
        }
    };
    // cheap
    IndexTreeModel.prototype.hasTreeNode = function (location, node) {
        if (node === void 0) { node = this.root; }
        if (!location || location.length === 0) {
            return true;
        }
        var _a = __read(location), index = _a[0], rest = _a.slice(1);
        if (index < 0 || index > node.children.length) {
            return false;
        }
        return this.hasTreeNode(rest, node.children[index]);
    };
    // cheap
    IndexTreeModel.prototype.getTreeNode = function (location, node) {
        if (node === void 0) { node = this.root; }
        if (!location || location.length === 0) {
            return node;
        }
        var _a = __read(location), index = _a[0], rest = _a.slice(1);
        if (index < 0 || index > node.children.length) {
            throw new TreeError(this.user, 'Invalid tree location');
        }
        return this.getTreeNode(rest, node.children[index]);
    };
    // expensive
    IndexTreeModel.prototype.getTreeNodeWithListIndex = function (location) {
        if (location.length === 0) {
            return { node: this.root, listIndex: -1, revealed: true, visible: false };
        }
        var _a = this.getParentNodeWithListIndex(location), parentNode = _a.parentNode, listIndex = _a.listIndex, revealed = _a.revealed, visible = _a.visible;
        var index = location[location.length - 1];
        if (index < 0 || index > parentNode.children.length) {
            throw new TreeError(this.user, 'Invalid tree location');
        }
        var node = parentNode.children[index];
        return { node: node, listIndex: listIndex, revealed: revealed, visible: visible && node.visible };
    };
    IndexTreeModel.prototype.getParentNodeWithListIndex = function (location, node, listIndex, revealed, visible) {
        if (node === void 0) { node = this.root; }
        if (listIndex === void 0) { listIndex = 0; }
        if (revealed === void 0) { revealed = true; }
        if (visible === void 0) { visible = true; }
        var _a = __read(location), index = _a[0], rest = _a.slice(1);
        if (index < 0 || index > node.children.length) {
            throw new TreeError(this.user, 'Invalid tree location');
        }
        // TODO@joao perf!
        for (var i = 0; i < index; i++) {
            listIndex += node.children[i].renderNodeCount;
        }
        revealed = revealed && !node.collapsed;
        visible = visible && node.visible;
        if (rest.length === 0) {
            return { parentNode: node, listIndex: listIndex, revealed: revealed, visible: visible };
        }
        return this.getParentNodeWithListIndex(rest, node.children[index], listIndex + 1, revealed, visible);
    };
    IndexTreeModel.prototype.getNode = function (location) {
        if (location === void 0) { location = []; }
        return this.getTreeNode(location);
    };
    // TODO@joao perf!
    IndexTreeModel.prototype.getNodeLocation = function (node) {
        var location = [];
        var indexTreeNode = node; // typing woes
        while (indexTreeNode.parent) {
            location.push(indexTreeNode.parent.children.indexOf(indexTreeNode));
            indexTreeNode = indexTreeNode.parent;
        }
        return location.reverse();
    };
    IndexTreeModel.prototype.getParentNodeLocation = function (location) {
        if (location.length === 0) {
            return undefined;
        }
        else if (location.length === 1) {
            return [];
        }
        else {
            return tail2(location)[0];
        }
    };
    IndexTreeModel.prototype.getFirstElementChild = function (location) {
        var node = this.getTreeNode(location);
        if (node.children.length === 0) {
            return undefined;
        }
        return node.children[0].element;
    };
    IndexTreeModel.prototype.getLastElementAncestor = function (location) {
        if (location === void 0) { location = []; }
        var node = this.getTreeNode(location);
        if (node.children.length === 0) {
            return undefined;
        }
        return this._getLastElementAncestor(node);
    };
    IndexTreeModel.prototype._getLastElementAncestor = function (node) {
        if (node.children.length === 0) {
            return node.element;
        }
        return this._getLastElementAncestor(node.children[node.children.length - 1]);
    };
    return IndexTreeModel;
}());
export { IndexTreeModel };
