/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
var __values = (this && this.__values) || function(o) {
    var s = typeof Symbol === "function" && Symbol.iterator, m = s && o[s], i = 0;
    if (m) return m.call(o);
    if (o && typeof o.length === "number") return {
        next: function () {
            if (o && i >= o.length) o = void 0;
            return { value: o && o[i++], done: !o };
        }
    };
    throw new TypeError(s ? "Object is not iterable." : "Symbol.iterator is not defined.");
};
var _a;
import * as DOM from './dom';
import { createElement } from './formattedTextRenderer';
import { onUnexpectedError } from '../common/errors';
import { parseHrefAndDimensions, removeMarkdownEscapes } from '../common/htmlContent';
import { defaultGenerator } from '../common/idGenerator';
import * as marked from '../common/marked/marked';
import { insane } from '../common/insane/insane';
import { parse } from '../common/marshalling';
import { cloneAndChange } from '../common/objects';
import { escape } from '../common/strings';
import { URI } from '../common/uri';
import { FileAccess, Schemas } from '../common/network';
import { markdownEscapeEscapedIcons } from '../common/iconLabels';
import { resolvePath } from '../common/resources';
import { StandardMouseEvent } from './mouseEvent';
import { renderLabelWithIcons } from './ui/iconLabel/iconLabels';
import { Event } from '../common/event';
import { domEvent } from './event';
var _ttpInsane = (_a = window.trustedTypes) === null || _a === void 0 ? void 0 : _a.createPolicy('insane', {
    createHTML: function (value, options) {
        return insane(value, options);
    }
});
/**
 * Low-level way create a html element from a markdown string.
 *
 * **Note** that for most cases you should be using [`MarkdownRenderer`](./src/vs/editor/browser/core/markdownRenderer.ts)
 * which comes with support for pretty code block rendering and which uses the default way of handling links.
 */
export function renderMarkdown(markdown, options, markedOptions) {
    var e_1, _a;
    var _b;
    if (options === void 0) { options = {}; }
    if (markedOptions === void 0) { markedOptions = {}; }
    var element = createElement(options);
    var _uriMassage = function (part) {
        var data;
        try {
            data = parse(decodeURIComponent(part));
        }
        catch (e) {
            // ignore
        }
        if (!data) {
            return part;
        }
        data = cloneAndChange(data, function (value) {
            if (markdown.uris && markdown.uris[value]) {
                return URI.revive(markdown.uris[value]);
            }
            else {
                return undefined;
            }
        });
        return encodeURIComponent(JSON.stringify(data));
    };
    var _href = function (href, isDomUri) {
        var data = markdown.uris && markdown.uris[href];
        if (!data) {
            return href; // no uri exists
        }
        var uri = URI.revive(data);
        if (URI.parse(href).toString() === uri.toString()) {
            return href; // no tranformation performed
        }
        if (isDomUri) {
            // this URI will end up as "src"-attribute of a dom node
            // and because of that special rewriting needs to be done
            // so that the URI uses a protocol that's understood by
            // browsers (like http or https)
            return FileAccess.asBrowserUri(uri).toString(true);
        }
        if (uri.query) {
            uri = uri.with({ query: _uriMassage(uri.query) });
        }
        return uri.toString();
    };
    // signal to code-block render that the
    // element has been created
    var signalInnerHTML;
    var withInnerHTML = new Promise(function (c) { return signalInnerHTML = c; });
    var renderer = new marked.Renderer();
    renderer.image = function (href, title, text) {
        var _a;
        var dimensions = [];
        var attributes = [];
        if (href) {
            (_a = parseHrefAndDimensions(href), href = _a.href, dimensions = _a.dimensions);
            href = _href(href, true);
            try {
                var hrefAsUri = URI.parse(href);
                if (options.baseUrl && hrefAsUri.scheme === Schemas.file) { // absolute or relative local path, or file: uri
                    href = resolvePath(options.baseUrl, href).toString();
                }
            }
            catch (err) { }
            attributes.push("src=\"" + href + "\"");
        }
        if (text) {
            attributes.push("alt=\"" + text + "\"");
        }
        if (title) {
            attributes.push("title=\"" + title + "\"");
        }
        if (dimensions.length) {
            attributes = attributes.concat(dimensions);
        }
        return '<img ' + attributes.join(' ') + '>';
    };
    renderer.link = function (href, title, text) {
        // Remove markdown escapes. Workaround for https://github.com/chjj/marked/issues/829
        if (href === text) { // raw link case
            text = removeMarkdownEscapes(text);
        }
        href = _href(href, false);
        if (options.baseUrl) {
            var hasScheme = /^\w[\w\d+.-]*:/.test(href);
            if (!hasScheme) {
                href = resolvePath(options.baseUrl, href).toString();
            }
        }
        title = removeMarkdownEscapes(title);
        href = removeMarkdownEscapes(href);
        if (!href
            || href.match(/^data:|javascript:/i)
            || (href.match(/^command:/i) && !markdown.isTrusted)
            || href.match(/^command:(\/\/\/)?_workbench\.downloadResource/i)) {
            // drop the link
            return text;
        }
        else {
            // HTML Encode href
            href = href.replace(/&/g, '&amp;')
                .replace(/</g, '&lt;')
                .replace(/>/g, '&gt;')
                .replace(/"/g, '&quot;')
                .replace(/'/g, '&#39;');
            return "<a href=\"#\" data-href=\"" + href + "\" title=\"" + (title || href) + "\">" + text + "</a>";
        }
    };
    renderer.paragraph = function (text) {
        if (markdown.supportThemeIcons) {
            var elements = renderLabelWithIcons(text);
            text = elements.map(function (e) { return typeof e === 'string' ? e : e.outerHTML; }).join('');
        }
        return "<p>" + text + "</p>";
    };
    if (options.codeBlockRenderer) {
        renderer.code = function (code, lang) {
            var value = options.codeBlockRenderer(lang, code);
            // when code-block rendering is async we return sync
            // but update the node with the real result later.
            var id = defaultGenerator.nextId();
            var promise = Promise.all([value, withInnerHTML]).then(function (values) {
                var span = element.querySelector("div[data-code=\"" + id + "\"]");
                if (span) {
                    DOM.reset(span, values[0]);
                }
            }).catch(function (_err) {
                // ignore
            });
            if (options.asyncRenderCallback) {
                promise.then(options.asyncRenderCallback);
            }
            return "<div class=\"code\" data-code=\"" + id + "\">" + escape(code) + "</div>";
        };
    }
    if (options.actionHandler) {
        options.actionHandler.disposeables.add(Event.any(domEvent(element, 'click'), domEvent(element, 'auxclick'))(function (e) {
            var mouseEvent = new StandardMouseEvent(e);
            if (!mouseEvent.leftButton && !mouseEvent.middleButton) {
                return;
            }
            var target = mouseEvent.target;
            if (target.tagName !== 'A') {
                target = target.parentElement;
                if (!target || target.tagName !== 'A') {
                    return;
                }
            }
            try {
                var href = target.dataset['href'];
                if (href) {
                    options.actionHandler.callback(href, mouseEvent);
                }
            }
            catch (err) {
                onUnexpectedError(err);
            }
            finally {
                mouseEvent.preventDefault();
            }
        }));
    }
    // Use our own sanitizer so that we can let through only spans.
    // Otherwise, we'd be letting all html be rendered.
    // If we want to allow markdown permitted tags, then we can delete sanitizer and sanitize.
    // We always pass the output through insane after this so that we don't rely on
    // marked for sanitization.
    markedOptions.sanitizer = function (html) {
        var match = markdown.isTrusted ? html.match(/^(<span[^>]+>)|(<\/\s*span>)$/) : undefined;
        return match ? html : '';
    };
    markedOptions.sanitize = true;
    markedOptions.silent = true;
    markedOptions.renderer = renderer;
    // values that are too long will freeze the UI
    var value = (_b = markdown.value) !== null && _b !== void 0 ? _b : '';
    if (value.length > 100000) {
        value = value.substr(0, 100000) + "\u2026";
    }
    // escape theme icons
    if (markdown.supportThemeIcons) {
        value = markdownEscapeEscapedIcons(value);
    }
    var renderedMarkdown = marked.parse(value, markedOptions);
    // sanitize with insane
    element.innerHTML = sanitizeRenderedMarkdown(markdown, renderedMarkdown);
    // signal that async code blocks can be now be inserted
    signalInnerHTML();
    // signal size changes for image tags
    if (options.asyncRenderCallback) {
        var _loop_1 = function (img) {
            var listener = DOM.addDisposableListener(img, 'load', function () {
                listener.dispose();
                options.asyncRenderCallback();
            });
        };
        try {
            for (var _c = __values(element.getElementsByTagName('img')), _d = _c.next(); !_d.done; _d = _c.next()) {
                var img = _d.value;
                _loop_1(img);
            }
        }
        catch (e_1_1) { e_1 = { error: e_1_1 }; }
        finally {
            try {
                if (_d && !_d.done && (_a = _c.return)) _a.call(_c);
            }
            finally { if (e_1) throw e_1.error; }
        }
    }
    return element;
}
function sanitizeRenderedMarkdown(options, renderedMarkdown) {
    var _a;
    var insaneOptions = getInsaneOptions(options);
    return (_a = _ttpInsane === null || _ttpInsane === void 0 ? void 0 : _ttpInsane.createHTML(renderedMarkdown, insaneOptions)) !== null && _a !== void 0 ? _a : insane(renderedMarkdown, insaneOptions);
}
function getInsaneOptions(options) {
    var allowedSchemes = [
        Schemas.http,
        Schemas.https,
        Schemas.mailto,
        Schemas.data,
        Schemas.file,
        Schemas.vscodeRemote,
        Schemas.vscodeRemoteResource,
    ];
    if (options.isTrusted) {
        allowedSchemes.push(Schemas.command);
    }
    return {
        allowedSchemes: allowedSchemes,
        // allowedTags should included everything that markdown renders to.
        // Since we have our own sanitize function for marked, it's possible we missed some tag so let insane make sure.
        // HTML tags that can result from markdown are from reading https://spec.commonmark.org/0.29/
        // HTML table tags that can result from markdown are from https://github.github.com/gfm/#tables-extension-
        allowedTags: ['ul', 'li', 'p', 'code', 'blockquote', 'ol', 'h1', 'h2', 'h3', 'h4', 'h5', 'h6', 'hr', 'em', 'pre', 'table', 'thead', 'tbody', 'tr', 'th', 'td', 'div', 'del', 'a', 'strong', 'br', 'img', 'span'],
        allowedAttributes: {
            'a': ['href', 'name', 'target', 'data-href'],
            'img': ['src', 'title', 'alt', 'width', 'height'],
            'div': ['class', 'data-code'],
            'span': ['class', 'style'],
            // https://github.com/microsoft/vscode/issues/95937
            'th': ['align'],
            'td': ['align']
        },
        filter: function (token) {
            if (token.tag === 'span' && options.isTrusted) {
                if (token.attrs['style'] && (Object.keys(token.attrs).length === 1)) {
                    return !!token.attrs['style'].match(/^(color\:#[0-9a-fA-F]+;)?(background-color\:#[0-9a-fA-F]+;)?$/);
                }
                else if (token.attrs['class']) {
                    // The class should match codicon rendering in src\vs\base\common\codicons.ts
                    return !!token.attrs['class'].match(/^codicon codicon-[a-z\-]+( codicon-modifier-[a-z\-]+)?$/);
                }
                return false;
            }
            return true;
        }
    };
}
/**
 * Strips all markdown from `markdown`. For example `# Header` would be output as `Header`.
 */
export function renderMarkdownAsPlaintext(markdown) {
    var _a;
    var renderer = new marked.Renderer();
    renderer.code = function (code) {
        return code;
    };
    renderer.blockquote = function (quote) {
        return quote;
    };
    renderer.html = function (_html) {
        return '';
    };
    renderer.heading = function (text, _level, _raw) {
        return text + '\n';
    };
    renderer.hr = function () {
        return '';
    };
    renderer.list = function (body, _ordered) {
        return body;
    };
    renderer.listitem = function (text) {
        return text + '\n';
    };
    renderer.paragraph = function (text) {
        return text + '\n';
    };
    renderer.table = function (header, body) {
        return header + body + '\n';
    };
    renderer.tablerow = function (content) {
        return content;
    };
    renderer.tablecell = function (content, _flags) {
        return content + ' ';
    };
    renderer.strong = function (text) {
        return text;
    };
    renderer.em = function (text) {
        return text;
    };
    renderer.codespan = function (code) {
        return code;
    };
    renderer.br = function () {
        return '\n';
    };
    renderer.del = function (text) {
        return text;
    };
    renderer.image = function (_href, _title, _text) {
        return '';
    };
    renderer.text = function (text) {
        return text;
    };
    renderer.link = function (_href, _title, text) {
        return text;
    };
    // values that are too long will freeze the UI
    var value = (_a = markdown.value) !== null && _a !== void 0 ? _a : '';
    if (value.length > 100000) {
        value = value.substr(0, 100000) + "\u2026";
    }
    var unescapeInfo = new Map([
        ['&quot;', '"'],
        ['&amp;', '&'],
        ['&#39;', '\''],
        ['&lt;', '<'],
        ['&gt;', '>'],
    ]);
    var html = marked.parse(value, { renderer: renderer }).replace(/&(#\d+|[a-zA-Z]+);/g, function (m) { var _a; return (_a = unescapeInfo.get(m)) !== null && _a !== void 0 ? _a : m; });
    return sanitizeRenderedMarkdown({ isTrusted: false }, html).toString();
}
