"use strict";
/********************************************************************************
 * Copyright (C) 2021 Ericsson and others.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 ********************************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
exports.EditorPreviewWidget = void 0;
const browser_1 = require("@theia/core/lib/browser");
const browser_2 = require("@theia/editor/lib/browser");
const common_1 = require("@theia/core/lib/common");
const algorithm_1 = require("@theia/core/shared/@phosphor/algorithm");
const PREVIEW_TITLE_CLASS = 'theia-editor-preview-title-unpinned';
class EditorPreviewWidget extends browser_2.EditorWidget {
    constructor(editor, selectionService) {
        super(editor, selectionService);
        this.editor = editor;
        this.selectionService = selectionService;
        this._isPreview = false;
        this.onDidChangePreviewStateEmitter = new common_1.Emitter();
        this.onDidChangePreviewState = this.onDidChangePreviewStateEmitter.event;
        this.toDisposeOnLocationChange = new common_1.DisposableCollection();
        this.toDispose.push(this.onDidChangePreviewStateEmitter);
        this.toDispose.push(this.toDisposeOnLocationChange);
    }
    get isPreview() {
        return this._isPreview;
    }
    initializePreview() {
        this._isPreview = true;
        this.title.className += ` ${PREVIEW_TITLE_CLASS}`;
        const oneTimeDirtyChangeListener = this.saveable.onDirtyChanged(() => {
            this.convertToNonPreview();
            oneTimeDirtyChangeListener.dispose();
        });
        this.toDispose.push(oneTimeDirtyChangeListener);
    }
    convertToNonPreview() {
        if (this._isPreview) {
            this._isPreview = false;
            this.toDisposeOnLocationChange.dispose();
            this.lastTabbar = undefined;
            this.title.className = this.title.className.replace(PREVIEW_TITLE_CLASS, '');
            this.onDidChangePreviewStateEmitter.fire();
            this.onDidChangePreviewStateEmitter.dispose();
        }
    }
    onAfterAttach(msg) {
        super.onAfterAttach(msg);
        if (this._isPreview) {
            this.checkForTabbarChange();
        }
    }
    checkForTabbarChange() {
        const { parent } = this;
        if (parent instanceof browser_1.DockPanel) {
            this.toDisposeOnLocationChange.dispose();
            const newTabbar = algorithm_1.find(parent.tabBars(), tabbar => !!tabbar.titles.find(title => title === this.title));
            if (this.lastTabbar && this.lastTabbar !== newTabbar) {
                this.convertToNonPreview();
            }
            else {
                this.lastTabbar = newTabbar;
                const listener = () => this.checkForTabbarChange();
                parent.layoutModified.connect(listener);
                this.toDisposeOnLocationChange.push(common_1.Disposable.create(() => parent.layoutModified.disconnect(listener)));
            }
        }
    }
    storeState() {
        const { _isPreview: isPreview } = this;
        return { isPreview, editorState: this.editor.storeViewState() };
    }
    restoreState(oldState) {
        if (!oldState.isPreview) {
            this.convertToNonPreview();
        }
        this.editor.restoreViewState(oldState.editorState);
    }
}
exports.EditorPreviewWidget = EditorPreviewWidget;
//# sourceMappingURL=editor-preview-widget.js.map