"use strict";
/********************************************************************************
 * Copyright (C) 2017 TypeFox and others.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 ********************************************************************************/
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.TerminalFrontendContribution = exports.TerminalCommands = exports.TerminalMenus = void 0;
const inversify_1 = require("@theia/core/shared/inversify");
const common_1 = require("@theia/core/lib/common");
const browser_1 = require("@theia/core/lib/browser");
const terminal_widget_impl_1 = require("./terminal-widget-impl");
const terminal_keybinding_contexts_1 = require("./terminal-keybinding-contexts");
const terminal_widget_1 = require("./base/terminal-widget");
const uri_command_handler_1 = require("@theia/core/lib/common/uri-command-handler");
const shell_terminal_protocol_1 = require("../common/shell-terminal-protocol");
const core_1 = require("@theia/core");
const browser_2 = require("@theia/workspace/lib/browser");
const context_key_service_1 = require("@theia/core/lib/browser/context-key-service");
const terminal_theme_service_1 = require("./terminal-theme-service");
const file_service_1 = require("@theia/filesystem/lib/browser/file-service");
const terminal_watcher_1 = require("../common/terminal-watcher");
const base_terminal_protocol_1 = require("../common/base-terminal-protocol");
const nls_1 = require("@theia/core/lib/common/nls");
const terminal_preferences_1 = require("./terminal-preferences");
var TerminalMenus;
(function (TerminalMenus) {
    TerminalMenus.TERMINAL = [...core_1.MAIN_MENU_BAR, '7_terminal'];
    TerminalMenus.TERMINAL_NEW = [...TerminalMenus.TERMINAL, '1_terminal'];
    TerminalMenus.TERMINAL_TASKS = [...TerminalMenus.TERMINAL, '2_terminal'];
    TerminalMenus.TERMINAL_TASKS_INFO = [...TerminalMenus.TERMINAL_TASKS, '3_terminal'];
    TerminalMenus.TERMINAL_TASKS_CONFIG = [...TerminalMenus.TERMINAL_TASKS, '4_terminal'];
    TerminalMenus.TERMINAL_NAVIGATOR_CONTEXT_MENU = ['navigator-context-menu', 'navigation'];
    TerminalMenus.TERMINAL_OPEN_EDITORS_CONTEXT_MENU = ['open-editors-context-menu', 'navigation'];
})(TerminalMenus = exports.TerminalMenus || (exports.TerminalMenus = {}));
var TerminalCommands;
(function (TerminalCommands) {
    const TERMINAL_CATEGORY = 'Terminal';
    TerminalCommands.NEW = common_1.Command.toDefaultLocalizedCommand({
        id: 'terminal:new',
        category: TERMINAL_CATEGORY,
        label: 'Create New Integrated Terminal'
    });
    TerminalCommands.NEW_ACTIVE_WORKSPACE = common_1.Command.toDefaultLocalizedCommand({
        id: 'terminal:new:active:workspace',
        category: TERMINAL_CATEGORY,
        label: 'Create New Integrated Terminal (In Active Workspace)'
    });
    TerminalCommands.TERMINAL_CLEAR = common_1.Command.toDefaultLocalizedCommand({
        id: 'terminal:clear',
        category: TERMINAL_CATEGORY,
        label: 'Clear'
    });
    TerminalCommands.TERMINAL_CONTEXT = common_1.Command.toDefaultLocalizedCommand({
        id: 'terminal:context',
        category: TERMINAL_CATEGORY,
        label: 'Open in Terminal'
    });
    TerminalCommands.SPLIT = common_1.Command.toDefaultLocalizedCommand({
        id: 'terminal:split',
        category: TERMINAL_CATEGORY,
        label: 'Split Terminal'
    });
    TerminalCommands.TERMINAL_FIND_TEXT = common_1.Command.toDefaultLocalizedCommand({
        id: 'terminal:find',
        category: TERMINAL_CATEGORY,
        label: 'Find'
    });
    TerminalCommands.TERMINAL_FIND_TEXT_CANCEL = common_1.Command.toDefaultLocalizedCommand({
        id: 'terminal:find:cancel',
        category: TERMINAL_CATEGORY,
        label: 'Hide Find'
    });
    TerminalCommands.SCROLL_LINE_UP = common_1.Command.toDefaultLocalizedCommand({
        id: 'terminal:scroll:line:up',
        category: TERMINAL_CATEGORY,
        label: 'Scroll Up (Line)'
    });
    TerminalCommands.SCROLL_LINE_DOWN = common_1.Command.toDefaultLocalizedCommand({
        id: 'terminal:scroll:line:down',
        category: TERMINAL_CATEGORY,
        label: 'Scroll Down (Line)'
    });
    TerminalCommands.SCROLL_TO_TOP = common_1.Command.toDefaultLocalizedCommand({
        id: 'terminal:scroll:top',
        category: TERMINAL_CATEGORY,
        label: 'Scroll to Top'
    });
    TerminalCommands.SCROLL_PAGE_UP = common_1.Command.toDefaultLocalizedCommand({
        id: 'terminal:scroll:page:up',
        category: TERMINAL_CATEGORY,
        label: 'Scroll Up (Page)'
    });
    TerminalCommands.SCROLL_PAGE_DOWN = common_1.Command.toDefaultLocalizedCommand({
        id: 'terminal:scroll:page:down',
        category: TERMINAL_CATEGORY,
        label: 'Scroll Down (Page)'
    });
    /**
     * Command that displays all terminals that are currently opened
     */
    TerminalCommands.SHOW_ALL_OPENED_TERMINALS = common_1.Command.toDefaultLocalizedCommand({
        id: 'workbench.action.showAllTerminals',
        category: browser_1.CommonCommands.VIEW_CATEGORY,
        label: 'Show All Opened Terminals'
    });
})(TerminalCommands = exports.TerminalCommands || (exports.TerminalCommands = {}));
let TerminalFrontendContribution = class TerminalFrontendContribution {
    constructor() {
        this.onDidCreateTerminalEmitter = new common_1.Emitter();
        this.onDidCreateTerminal = this.onDidCreateTerminalEmitter.event;
        this.onDidChangeCurrentTerminalEmitter = new common_1.Emitter();
        this.onDidChangeCurrentTerminal = this.onDidChangeCurrentTerminalEmitter.event;
        // IDs of the most recently used terminals
        this.mostRecentlyUsedTerminalEntries = [];
    }
    init() {
        this.shell.currentChanged.connect(() => this.updateCurrentTerminal());
        this.widgetManager.onDidCreateWidget(({ widget }) => {
            if (widget instanceof terminal_widget_1.TerminalWidget) {
                this.updateCurrentTerminal();
                this.onDidCreateTerminalEmitter.fire(widget);
                this.setLastUsedTerminal(widget);
            }
        });
        const terminalFocusKey = this.contextKeyService.createKey('terminalFocus', false);
        const updateFocusKey = () => terminalFocusKey.set(this.shell.activeWidget instanceof terminal_widget_1.TerminalWidget);
        updateFocusKey();
        this.shell.activeChanged.connect(updateFocusKey);
        this.terminalWatcher.onStoreTerminalEnvVariablesRequested(data => {
            this.storageService.setData(base_terminal_protocol_1.ENVIRONMENT_VARIABLE_COLLECTIONS_KEY, data);
        });
        this.terminalWatcher.onUpdateTerminalEnvVariablesRequested(() => {
            this.storageService.getData(base_terminal_protocol_1.ENVIRONMENT_VARIABLE_COLLECTIONS_KEY).then(data => {
                if (data) {
                    const collectionsJson = JSON.parse(data);
                    collectionsJson.forEach(c => this.shellTerminalServer.setCollection(c.extensionIdentifier, true, c.collection));
                }
            });
        });
    }
    onWillStop() {
        return (this.terminalPreferences['terminal.integrated.confirmOnExit'] !== 'never' &&
            this.widgetManager.getWidgets(terminal_widget_impl_1.TERMINAL_WIDGET_FACTORY_ID).length > 0);
    }
    get currentTerminal() {
        return this._currentTerminal;
    }
    setCurrentTerminal(current) {
        if (this._currentTerminal !== current) {
            this._currentTerminal = current;
            this.onDidChangeCurrentTerminalEmitter.fire(this._currentTerminal);
        }
    }
    updateCurrentTerminal() {
        const widget = this.shell.currentWidget;
        if (widget instanceof terminal_widget_1.TerminalWidget) {
            this.setCurrentTerminal(widget);
        }
        else if (!this._currentTerminal || !this._currentTerminal.isVisible) {
            this.setCurrentTerminal(undefined);
        }
    }
    getLastUsedTerminalId() {
        const mostRecent = this.mostRecentlyUsedTerminalEntries[this.mostRecentlyUsedTerminalEntries.length - 1];
        if (mostRecent) {
            return mostRecent.id;
        }
    }
    get lastUsedTerminal() {
        const id = this.getLastUsedTerminalId();
        if (id) {
            return this.getById(id);
        }
    }
    setLastUsedTerminal(lastUsedTerminal) {
        const lastUsedTerminalId = lastUsedTerminal.id;
        const entryIndex = this.mostRecentlyUsedTerminalEntries.findIndex(entry => entry.id === lastUsedTerminalId);
        let toDispose;
        if (entryIndex >= 0) {
            toDispose = this.mostRecentlyUsedTerminalEntries[entryIndex].disposables;
            this.mostRecentlyUsedTerminalEntries.splice(entryIndex, 1);
        }
        else {
            toDispose = new common_1.DisposableCollection();
            toDispose.push(lastUsedTerminal.onDidChangeVisibility((isVisible) => {
                if (isVisible) {
                    this.setLastUsedTerminal(lastUsedTerminal);
                }
            }));
            toDispose.push(lastUsedTerminal.onDidDispose(() => {
                const index = this.mostRecentlyUsedTerminalEntries.findIndex(entry => entry.id === lastUsedTerminalId);
                if (index >= 0) {
                    this.mostRecentlyUsedTerminalEntries[index].disposables.dispose();
                    this.mostRecentlyUsedTerminalEntries.splice(index, 1);
                }
            }));
        }
        const newEntry = { id: lastUsedTerminalId, disposables: toDispose };
        if (lastUsedTerminal.isVisible) {
            this.mostRecentlyUsedTerminalEntries.push(newEntry);
        }
        else {
            this.mostRecentlyUsedTerminalEntries = [newEntry, ...this.mostRecentlyUsedTerminalEntries];
        }
    }
    get all() {
        return this.widgetManager.getWidgets(terminal_widget_impl_1.TERMINAL_WIDGET_FACTORY_ID);
    }
    getById(id) {
        return this.all.find(terminal => terminal.id === id);
    }
    getByTerminalId(terminalId) {
        return this.all.find(terminal => terminal.terminalId === terminalId);
    }
    getDefaultShell() {
        return this.shellTerminalServer.getDefaultShell();
    }
    registerCommands(commands) {
        commands.registerCommand(TerminalCommands.NEW, {
            execute: () => this.openTerminal()
        });
        commands.registerCommand(TerminalCommands.NEW_ACTIVE_WORKSPACE, {
            execute: () => this.openActiveWorkspaceTerminal()
        });
        commands.registerCommand(TerminalCommands.SPLIT, {
            execute: widget => this.splitTerminal(widget),
            isEnabled: widget => !!this.getTerminalRef(widget),
            isVisible: widget => !!this.getTerminalRef(widget)
        });
        commands.registerCommand(TerminalCommands.TERMINAL_CLEAR);
        commands.registerHandler(TerminalCommands.TERMINAL_CLEAR.id, {
            isEnabled: () => this.shell.activeWidget instanceof terminal_widget_1.TerminalWidget,
            execute: () => this.shell.activeWidget.clearOutput()
        });
        commands.registerCommand(TerminalCommands.TERMINAL_CONTEXT, uri_command_handler_1.UriAwareCommandHandler.MonoSelect(this.selectionService, {
            execute: uri => this.openInTerminal(uri)
        }));
        commands.registerCommand(TerminalCommands.TERMINAL_FIND_TEXT);
        commands.registerHandler(TerminalCommands.TERMINAL_FIND_TEXT.id, {
            isEnabled: () => {
                if (this.shell.activeWidget instanceof terminal_widget_1.TerminalWidget) {
                    return !this.shell.activeWidget.getSearchBox().isVisible;
                }
                return false;
            },
            execute: () => {
                const termWidget = this.shell.activeWidget;
                const terminalSearchBox = termWidget.getSearchBox();
                terminalSearchBox.show();
            }
        });
        commands.registerCommand(TerminalCommands.TERMINAL_FIND_TEXT_CANCEL);
        commands.registerHandler(TerminalCommands.TERMINAL_FIND_TEXT_CANCEL.id, {
            isEnabled: () => {
                if (this.shell.activeWidget instanceof terminal_widget_1.TerminalWidget) {
                    return this.shell.activeWidget.getSearchBox().isVisible;
                }
                return false;
            },
            execute: () => {
                const termWidget = this.shell.activeWidget;
                const terminalSearchBox = termWidget.getSearchBox();
                terminalSearchBox.hide();
            }
        });
        commands.registerCommand(TerminalCommands.SCROLL_LINE_UP, {
            isEnabled: () => this.shell.activeWidget instanceof terminal_widget_1.TerminalWidget,
            isVisible: () => false,
            execute: () => {
                this.shell.activeWidget.scrollLineUp();
            }
        });
        commands.registerCommand(TerminalCommands.SCROLL_LINE_DOWN, {
            isEnabled: () => this.shell.activeWidget instanceof terminal_widget_1.TerminalWidget,
            isVisible: () => false,
            execute: () => {
                this.shell.activeWidget.scrollLineDown();
            }
        });
        commands.registerCommand(TerminalCommands.SCROLL_TO_TOP, {
            isEnabled: () => this.shell.activeWidget instanceof terminal_widget_1.TerminalWidget,
            isVisible: () => false,
            execute: () => {
                this.shell.activeWidget.scrollToTop();
            }
        });
        commands.registerCommand(TerminalCommands.SCROLL_PAGE_UP, {
            isEnabled: () => this.shell.activeWidget instanceof terminal_widget_1.TerminalWidget,
            isVisible: () => false,
            execute: () => {
                this.shell.activeWidget.scrollPageUp();
            }
        });
        commands.registerCommand(TerminalCommands.SCROLL_PAGE_DOWN, {
            isEnabled: () => this.shell.activeWidget instanceof terminal_widget_1.TerminalWidget,
            isVisible: () => false,
            execute: () => {
                this.shell.activeWidget.scrollPageDown();
            }
        });
    }
    async openInTerminal(uri) {
        // Determine folder path of URI
        let stat;
        try {
            stat = await this.fileService.resolve(uri);
        }
        catch (_a) {
            return;
        }
        // Use folder if a file was selected
        const cwd = (stat.isDirectory) ? uri.toString() : uri.parent.toString();
        // Open terminal
        const termWidget = await this.newTerminal({ cwd });
        termWidget.start();
        this.activateTerminal(termWidget);
    }
    registerMenus(menus) {
        menus.registerSubmenu(TerminalMenus.TERMINAL, terminal_widget_impl_1.TerminalWidgetImpl.LABEL);
        menus.registerMenuAction(TerminalMenus.TERMINAL_NEW, {
            commandId: TerminalCommands.NEW.id,
            label: nls_1.nls.localizeByDefault('New Terminal'),
            order: '0'
        });
        menus.registerMenuAction(TerminalMenus.TERMINAL_NEW, {
            commandId: TerminalCommands.SPLIT.id,
            order: '1'
        });
        menus.registerMenuAction(TerminalMenus.TERMINAL_NAVIGATOR_CONTEXT_MENU, {
            commandId: TerminalCommands.TERMINAL_CONTEXT.id,
            order: 'z'
        });
        menus.registerMenuAction(TerminalMenus.TERMINAL_OPEN_EDITORS_CONTEXT_MENU, {
            commandId: TerminalCommands.TERMINAL_CONTEXT.id,
            order: 'z'
        });
    }
    registerToolbarItems(toolbar) {
        toolbar.registerItem({
            id: TerminalCommands.SPLIT.id,
            command: TerminalCommands.SPLIT.id,
            icon: browser_1.codicon('split-horizontal'),
            tooltip: TerminalCommands.SPLIT.label
        });
    }
    registerKeybindings(keybindings) {
        /* Register passthrough keybindings for combinations recognized by
           xterm.js and converted to control characters.
             See: https://github.com/xtermjs/xterm.js/blob/v3/src/Terminal.ts#L1684 */
        /* Register ctrl + k (the passed Key) as a passthrough command in the
           context of the terminal.  */
        const regCtrl = (k) => {
            keybindings.registerKeybinding({
                command: browser_1.KeybindingRegistry.PASSTHROUGH_PSEUDO_COMMAND,
                keybinding: browser_1.KeyCode.createKeyCode({ key: k, ctrl: true }).toString(),
                context: terminal_keybinding_contexts_1.TerminalKeybindingContexts.terminalActive,
            });
        };
        /* Register alt + k (the passed Key) as a passthrough command in the
           context of the terminal.  */
        const regAlt = (k) => {
            keybindings.registerKeybinding({
                command: browser_1.KeybindingRegistry.PASSTHROUGH_PSEUDO_COMMAND,
                keybinding: browser_1.KeyCode.createKeyCode({ key: k, alt: true }).toString(),
                context: terminal_keybinding_contexts_1.TerminalKeybindingContexts.terminalActive
            });
        };
        /* ctrl-space (000 - NUL).  */
        regCtrl(browser_1.Key.SPACE);
        /* ctrl-A (001/1/0x1) through ctrl-Z (032/26/0x1A).  */
        for (let i = 0; i < 26; i++) {
            regCtrl({
                keyCode: browser_1.Key.KEY_A.keyCode + i,
                code: 'Key' + String.fromCharCode('A'.charCodeAt(0) + i)
            });
        }
        /* ctrl-[ or ctrl-3 (033/27/0x1B - ESC).  */
        regCtrl(browser_1.Key.BRACKET_LEFT);
        regCtrl(browser_1.Key.DIGIT3);
        /* ctrl-\ or ctrl-4 (034/28/0x1C - FS).  */
        regCtrl(browser_1.Key.BACKSLASH);
        regCtrl(browser_1.Key.DIGIT4);
        /* ctrl-] or ctrl-5 (035/29/0x1D - GS).  */
        regCtrl(browser_1.Key.BRACKET_RIGHT);
        regCtrl(browser_1.Key.DIGIT5);
        /* ctrl-6 (036/30/0x1E - RS).  */
        regCtrl(browser_1.Key.DIGIT6);
        /* ctrl-7 (037/31/0x1F - US).  */
        regCtrl(browser_1.Key.DIGIT7);
        /* ctrl-8 (177/127/0x7F - DEL).  */
        regCtrl(browser_1.Key.DIGIT8);
        /* alt-A (0x1B 0x62) through alt-Z (0x1B 0x7A).  */
        for (let i = 0; i < 26; i++) {
            regAlt({
                keyCode: browser_1.Key.KEY_A.keyCode + i,
                code: 'Key' + String.fromCharCode('A'.charCodeAt(0) + i)
            });
        }
        /* alt-` (0x1B 0x60).  */
        regAlt(browser_1.Key.BACKQUOTE);
        /* alt-0 (0x1B 0x30) through alt-9 (0x1B 0x39).  */
        for (let i = 0; i < 10; i++) {
            regAlt({
                keyCode: browser_1.Key.DIGIT0.keyCode + i,
                code: 'Digit' + String.fromCharCode('0'.charCodeAt(0) + i)
            });
        }
        if (common_1.isOSX) {
            // selectAll on OSX
            keybindings.registerKeybinding({
                command: browser_1.KeybindingRegistry.PASSTHROUGH_PSEUDO_COMMAND,
                keybinding: 'ctrlcmd+a',
                context: terminal_keybinding_contexts_1.TerminalKeybindingContexts.terminalActive
            });
        }
        keybindings.registerKeybinding({
            command: TerminalCommands.NEW.id,
            keybinding: 'ctrl+shift+`'
        });
        keybindings.registerKeybinding({
            command: TerminalCommands.NEW_ACTIVE_WORKSPACE.id,
            keybinding: 'ctrl+`'
        });
        keybindings.registerKeybinding({
            command: TerminalCommands.TERMINAL_CLEAR.id,
            keybinding: 'ctrlcmd+k',
            context: terminal_keybinding_contexts_1.TerminalKeybindingContexts.terminalActive
        });
        keybindings.registerKeybinding({
            command: TerminalCommands.TERMINAL_FIND_TEXT.id,
            keybinding: 'ctrlcmd+f',
            context: terminal_keybinding_contexts_1.TerminalKeybindingContexts.terminalActive
        });
        keybindings.registerKeybinding({
            command: TerminalCommands.TERMINAL_FIND_TEXT_CANCEL.id,
            keybinding: 'esc',
            context: terminal_keybinding_contexts_1.TerminalKeybindingContexts.terminalHideSearch
        });
        keybindings.registerKeybinding({
            command: TerminalCommands.SCROLL_LINE_UP.id,
            keybinding: 'ctrl+shift+up',
            context: terminal_keybinding_contexts_1.TerminalKeybindingContexts.terminalActive
        });
        keybindings.registerKeybinding({
            command: TerminalCommands.SCROLL_LINE_DOWN.id,
            keybinding: 'ctrl+shift+down',
            context: terminal_keybinding_contexts_1.TerminalKeybindingContexts.terminalActive
        });
        keybindings.registerKeybinding({
            command: TerminalCommands.SCROLL_TO_TOP.id,
            keybinding: 'shift-home',
            context: terminal_keybinding_contexts_1.TerminalKeybindingContexts.terminalActive
        });
        keybindings.registerKeybinding({
            command: TerminalCommands.SCROLL_PAGE_UP.id,
            keybinding: 'shift-pageUp',
            context: terminal_keybinding_contexts_1.TerminalKeybindingContexts.terminalActive
        });
        keybindings.registerKeybinding({
            command: TerminalCommands.SCROLL_PAGE_DOWN.id,
            keybinding: 'shift-pageDown',
            context: terminal_keybinding_contexts_1.TerminalKeybindingContexts.terminalActive
        });
    }
    async newTerminal(options) {
        const widget = await this.widgetManager.getOrCreateWidget(terminal_widget_impl_1.TERMINAL_WIDGET_FACTORY_ID, Object.assign({ created: new Date().toString() }, options));
        return widget;
    }
    activateTerminal(widget, widgetOptions) {
        this.open(widget, { widgetOptions });
    }
    // TODO: reuse WidgetOpenHandler.open
    open(widget, options) {
        const op = Object.assign(Object.assign({ mode: 'activate' }, options), { widgetOptions: Object.assign({ area: 'bottom' }, (options && options.widgetOptions)) });
        if (!widget.isAttached) {
            this.shell.addWidget(widget, op.widgetOptions);
        }
        if (op.mode === 'activate') {
            this.shell.activateWidget(widget.id);
        }
        else if (op.mode === 'reveal') {
            this.shell.revealWidget(widget.id);
        }
    }
    async selectTerminalCwd() {
        return new Promise(async (resolve) => {
            var _a, _b;
            const roots = this.workspaceService.tryGetRoots();
            if (roots.length === 0) {
                resolve(undefined);
            }
            else if (roots.length === 1) {
                resolve(roots[0].resource.toString());
            }
            else {
                const items = roots.map(({ resource }) => ({
                    label: this.labelProvider.getName(resource),
                    description: this.labelProvider.getLongName(resource),
                    resource
                }));
                const selectedItem = await ((_a = this.quickInputService) === null || _a === void 0 ? void 0 : _a.showQuickPick(items, {
                    placeholder: nls_1.nls.localizeByDefault('Select current working directory for new terminal')
                }));
                resolve((_b = selectedItem === null || selectedItem === void 0 ? void 0 : selectedItem.resource) === null || _b === void 0 ? void 0 : _b.toString());
            }
        });
    }
    async splitTerminal(widget) {
        const ref = this.getTerminalRef(widget);
        if (ref) {
            await this.openTerminal({ ref, mode: 'split-right' });
        }
    }
    getTerminalRef(widget) {
        const ref = widget ? widget : this.shell.currentWidget;
        return ref instanceof terminal_widget_1.TerminalWidget ? ref : undefined;
    }
    async openTerminal(options) {
        const cwd = await this.selectTerminalCwd();
        const termWidget = await this.newTerminal({ cwd });
        termWidget.start();
        this.open(termWidget, { widgetOptions: options });
    }
    async openActiveWorkspaceTerminal(options) {
        const termWidget = await this.newTerminal({});
        termWidget.start();
        this.open(termWidget, { widgetOptions: options });
    }
    /**
     * It should be aligned with https://code.visualstudio.com/api/references/theme-color#integrated-terminal-colors
     */
    registerColors(colors) {
        colors.register({
            id: 'terminal.background',
            defaults: {
                dark: 'panel.background',
                light: 'panel.background',
                hc: 'panel.background'
            },
            description: 'The background color of the terminal, this allows coloring the terminal differently to the panel.'
        });
        colors.register({
            id: 'terminal.foreground',
            defaults: {
                light: '#333333',
                dark: '#CCCCCC',
                hc: '#FFFFFF'
            },
            description: 'The foreground color of the terminal.'
        });
        colors.register({
            id: 'terminalCursor.foreground',
            description: 'The foreground color of the terminal cursor.'
        });
        colors.register({
            id: 'terminalCursor.background',
            description: 'The background color of the terminal cursor. Allows customizing the color of a character overlapped by a block cursor.'
        });
        colors.register({
            id: 'terminal.selectionBackground',
            defaults: {
                light: '#00000040',
                dark: '#FFFFFF40',
                hc: '#FFFFFF80'
            },
            description: 'The selection background color of the terminal.'
        });
        colors.register({
            id: 'terminal.border',
            defaults: {
                light: 'panel.border',
                dark: 'panel.border',
                hc: 'panel.border'
            },
            description: 'The color of the border that separates split panes within the terminal. This defaults to panel.border.'
        });
        // eslint-disable-next-line guard-for-in
        for (const id in terminal_theme_service_1.terminalAnsiColorMap) {
            const entry = terminal_theme_service_1.terminalAnsiColorMap[id];
            const colorName = id.substring(13);
            colors.register({
                id,
                defaults: entry.defaults,
                description: `'${colorName}'  ANSI color in the terminal.`
            });
        }
    }
};
__decorate([
    inversify_1.inject(browser_1.ApplicationShell),
    __metadata("design:type", browser_1.ApplicationShell)
], TerminalFrontendContribution.prototype, "shell", void 0);
__decorate([
    inversify_1.inject(shell_terminal_protocol_1.ShellTerminalServerProxy),
    __metadata("design:type", Object)
], TerminalFrontendContribution.prototype, "shellTerminalServer", void 0);
__decorate([
    inversify_1.inject(browser_1.WidgetManager),
    __metadata("design:type", browser_1.WidgetManager)
], TerminalFrontendContribution.prototype, "widgetManager", void 0);
__decorate([
    inversify_1.inject(file_service_1.FileService),
    __metadata("design:type", file_service_1.FileService)
], TerminalFrontendContribution.prototype, "fileService", void 0);
__decorate([
    inversify_1.inject(common_1.SelectionService),
    __metadata("design:type", common_1.SelectionService)
], TerminalFrontendContribution.prototype, "selectionService", void 0);
__decorate([
    inversify_1.inject(browser_1.LabelProvider),
    __metadata("design:type", browser_1.LabelProvider)
], TerminalFrontendContribution.prototype, "labelProvider", void 0);
__decorate([
    inversify_1.inject(browser_1.QuickInputService),
    inversify_1.optional(),
    __metadata("design:type", Object)
], TerminalFrontendContribution.prototype, "quickInputService", void 0);
__decorate([
    inversify_1.inject(browser_2.WorkspaceService),
    __metadata("design:type", browser_2.WorkspaceService)
], TerminalFrontendContribution.prototype, "workspaceService", void 0);
__decorate([
    inversify_1.inject(terminal_watcher_1.TerminalWatcher),
    __metadata("design:type", terminal_watcher_1.TerminalWatcher)
], TerminalFrontendContribution.prototype, "terminalWatcher", void 0);
__decorate([
    inversify_1.inject(browser_1.StorageService),
    __metadata("design:type", Object)
], TerminalFrontendContribution.prototype, "storageService", void 0);
__decorate([
    inversify_1.inject(terminal_preferences_1.TerminalPreferences),
    __metadata("design:type", Object)
], TerminalFrontendContribution.prototype, "terminalPreferences", void 0);
__decorate([
    inversify_1.inject(context_key_service_1.ContextKeyService),
    __metadata("design:type", context_key_service_1.ContextKeyService)
], TerminalFrontendContribution.prototype, "contextKeyService", void 0);
__decorate([
    inversify_1.postConstruct(),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", []),
    __metadata("design:returntype", void 0)
], TerminalFrontendContribution.prototype, "init", null);
TerminalFrontendContribution = __decorate([
    inversify_1.injectable()
], TerminalFrontendContribution);
exports.TerminalFrontendContribution = TerminalFrontendContribution;
//# sourceMappingURL=terminal-frontend-contribution.js.map