"use strict";
/********************************************************************************
 * Copyright (C) 2020 Red Hat, Inc. and others.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 ********************************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
exports.AuthenticationProviderImpl = exports.AuthenticationMainImpl = void 0;
const plugin_api_rpc_1 = require("../../common/plugin-api-rpc");
const message_service_1 = require("@theia/core/lib/common/message-service");
const browser_1 = require("@theia/core/lib/browser");
const authentication_service_1 = require("@theia/core/lib/browser/authentication-service");
const quick_pick_service_1 = require("@theia/core/lib/common/quick-pick-service");
class AuthenticationMainImpl {
    constructor(rpc, container) {
        this.proxy = rpc.getProxy(plugin_api_rpc_1.MAIN_RPC_CONTEXT.AUTHENTICATION_EXT);
        this.messageService = container.get(message_service_1.MessageService);
        this.storageService = container.get(browser_1.StorageService);
        this.authenticationService = container.get(authentication_service_1.AuthenticationService);
        this.quickPickService = container.get(quick_pick_service_1.QuickPickService);
        this.authenticationService.onDidChangeSessions(e => {
            this.proxy.$onDidChangeAuthenticationSessions(e.providerId, e.label, e.event);
        });
        this.authenticationService.onDidRegisterAuthenticationProvider(info => {
            this.proxy.$onDidChangeAuthenticationProviders([info], []);
        });
        this.authenticationService.onDidUnregisterAuthenticationProvider(providerId => {
            this.proxy.$onDidChangeAuthenticationProviders([], [providerId]);
        });
    }
    $getProviderIds() {
        return Promise.resolve(this.authenticationService.getProviderIds());
    }
    async $registerAuthenticationProvider(id, label, supportsMultipleAccounts) {
        const provider = new AuthenticationProviderImpl(this.proxy, id, label, supportsMultipleAccounts, this.storageService, this.messageService);
        this.authenticationService.registerAuthenticationProvider(id, provider);
    }
    async $unregisterAuthenticationProvider(id) {
        this.authenticationService.unregisterAuthenticationProvider(id);
    }
    async $updateSessions(id, event) {
        this.authenticationService.updateSessions(id, event);
    }
    $logout(providerId, sessionId) {
        return this.authenticationService.logout(providerId, sessionId);
    }
    async requestNewSession(providerId, scopes, extensionId, extensionName) {
        return this.authenticationService.requestNewSession(providerId, scopes, extensionId, extensionName);
    }
    async $getSession(providerId, scopes, extensionId, extensionName, options) {
        const orderedScopes = scopes.sort().join(' ');
        const sessions = (await this.authenticationService.getSessions(providerId)).filter(session => session.scopes.slice().sort().join(' ') === orderedScopes);
        const label = this.authenticationService.getLabel(providerId);
        if (sessions.length) {
            if (!this.authenticationService.supportsMultipleAccounts(providerId)) {
                const session = sessions[0];
                const allowed = await this.getSessionsPrompt(providerId, session.account.label, label, extensionId, extensionName);
                if (allowed) {
                    return session;
                }
                else {
                    throw new Error('User did not consent to login.');
                }
            }
            // On renderer side, confirm consent, ask user to choose between accounts if multiple sessions are valid
            const selected = await this.selectSession(providerId, label, extensionId, extensionName, sessions, scopes, !!options.clearSessionPreference);
            return sessions.find(session => session.id === selected.id);
        }
        else {
            if (options.createIfNone) {
                const isAllowed = await this.loginPrompt(label, extensionName);
                if (!isAllowed) {
                    throw new Error('User did not consent to login.');
                }
                const session = await this.authenticationService.login(providerId, scopes);
                await this.setTrustedExtensionAndAccountPreference(providerId, session.account.label, extensionId, extensionName, session.id);
                return session;
            }
            else {
                await this.requestNewSession(providerId, scopes, extensionId, extensionName);
                return undefined;
            }
        }
    }
    async selectSession(providerId, providerName, extensionId, extensionName, potentialSessions, scopes, clearSessionPreference) {
        if (!potentialSessions.length) {
            throw new Error('No potential sessions found');
        }
        if (clearSessionPreference) {
            await this.storageService.setData(`authentication-session-${extensionName}-${providerId}`, undefined);
        }
        else {
            const existingSessionPreference = await this.storageService.getData(`authentication-session-${extensionName}-${providerId}`);
            if (existingSessionPreference) {
                const matchingSession = potentialSessions.find(session => session.id === existingSessionPreference);
                if (matchingSession) {
                    const allowed = await this.getSessionsPrompt(providerId, matchingSession.account.label, providerName, extensionId, extensionName);
                    if (allowed) {
                        return matchingSession;
                    }
                }
            }
        }
        return new Promise(async (resolve, reject) => {
            var _a, _b;
            const items = potentialSessions.map(session => ({
                label: session.account.label,
                value: { session }
            }));
            items.push({
                label: 'Sign in to another account',
                value: { session: undefined }
            });
            const selected = await this.quickPickService.show(items, {
                title: `The extension '${extensionName}' wants to access a ${providerName} account`,
                ignoreFocusOut: true
            });
            if (selected) {
                const session = (_b = (_a = selected.value) === null || _a === void 0 ? void 0 : _a.session) !== null && _b !== void 0 ? _b : await this.authenticationService.login(providerId, scopes);
                const accountName = session.account.label;
                const allowList = await authentication_service_1.readAllowedExtensions(this.storageService, providerId, accountName);
                if (!allowList.find(allowed => allowed.id === extensionId)) {
                    allowList.push({ id: extensionId, name: extensionName });
                    this.storageService.setData(`authentication-trusted-extensions-${providerId}-${accountName}`, JSON.stringify(allowList));
                }
                this.storageService.setData(`authentication-session-${extensionName}-${providerId}`, session.id);
                resolve(session);
            }
            else {
                reject('User did not consent to account access');
            }
        });
    }
    async getSessionsPrompt(providerId, accountName, providerName, extensionId, extensionName) {
        const allowList = await authentication_service_1.readAllowedExtensions(this.storageService, providerId, accountName);
        const extensionData = allowList.find(extension => extension.id === extensionId);
        if (extensionData) {
            addAccountUsage(this.storageService, providerId, accountName, extensionId, extensionName);
            return true;
        }
        const choice = await this.messageService.info(`The extension '${extensionName}' wants to access the ${providerName} account '${accountName}'.`, 'Allow', 'Cancel');
        const allow = choice === 'Allow';
        if (allow) {
            await addAccountUsage(this.storageService, providerId, accountName, extensionId, extensionName);
            allowList.push({ id: extensionId, name: extensionName });
            this.storageService.setData(`authentication-trusted-extensions-${providerId}-${accountName}`, JSON.stringify(allowList));
        }
        return allow;
    }
    async loginPrompt(providerName, extensionName) {
        const choice = await this.messageService.info(`The extension '${extensionName}' wants to sign in using ${providerName}.`, 'Allow', 'Cancel');
        return choice === 'Allow';
    }
    async setTrustedExtensionAndAccountPreference(providerId, accountName, extensionId, extensionName, sessionId) {
        const allowList = await authentication_service_1.readAllowedExtensions(this.storageService, providerId, accountName);
        if (!allowList.find(allowed => allowed.id === extensionId)) {
            allowList.push({ id: extensionId, name: extensionName });
            this.storageService.setData(`authentication-trusted-extensions-${providerId}-${accountName}`, JSON.stringify(allowList));
        }
        this.storageService.setData(`authentication-session-${extensionName}-${providerId}`, sessionId);
    }
}
exports.AuthenticationMainImpl = AuthenticationMainImpl;
async function addAccountUsage(storageService, providerId, accountName, extensionId, extensionName) {
    const accountKey = `authentication-${providerId}-${accountName}-usages`;
    const usages = await readAccountUsages(storageService, providerId, accountName);
    const existingUsageIndex = usages.findIndex(usage => usage.extensionId === extensionId);
    if (existingUsageIndex > -1) {
        usages.splice(existingUsageIndex, 1, {
            extensionId,
            extensionName,
            lastUsed: Date.now()
        });
    }
    else {
        usages.push({
            extensionId,
            extensionName,
            lastUsed: Date.now()
        });
    }
    await storageService.setData(accountKey, JSON.stringify(usages));
}
class AuthenticationProviderImpl {
    constructor(proxy, id, label, supportsMultipleAccounts, storageService, messageService) {
        this.proxy = proxy;
        this.id = id;
        this.label = label;
        this.supportsMultipleAccounts = supportsMultipleAccounts;
        this.storageService = storageService;
        this.messageService = messageService;
        this.accounts = new Map(); // Map account name to session ids
        this.sessions = new Map(); // Map account id to name
    }
    hasSessions() {
        return !!this.sessions.size;
    }
    registerSession(session) {
        this.sessions.set(session.id, session.account.label);
        const existingSessionsForAccount = this.accounts.get(session.account.label);
        if (existingSessionsForAccount) {
            this.accounts.set(session.account.label, existingSessionsForAccount.concat(session.id));
            return;
        }
        else {
            this.accounts.set(session.account.label, [session.id]);
        }
    }
    async signOut(accountName) {
        const accountUsages = await readAccountUsages(this.storageService, this.id, accountName);
        const sessionsForAccount = this.accounts.get(accountName);
        const result = await this.messageService.info(accountUsages.length ? `The account ${accountName} has been used by:
        ${accountUsages.map(usage => usage.extensionName).join(', ')}. Sign out of these features?` : `Sign out of ${accountName}?`, 'Yes');
        if (result && result === 'Yes' && sessionsForAccount) {
            sessionsForAccount.forEach(sessionId => this.logout(sessionId));
            removeAccountUsage(this.storageService, this.id, accountName);
        }
    }
    async getSessions() {
        return this.proxy.$getSessions(this.id);
    }
    async updateSessionItems(event) {
        const { added, removed } = event;
        const session = await this.proxy.$getSessions(this.id);
        const addedSessions = session.filter(s => added.some(id => id === s.id));
        removed.forEach(sessionId => {
            const accountName = this.sessions.get(sessionId);
            if (accountName) {
                this.sessions.delete(sessionId);
                const sessionsForAccount = this.accounts.get(accountName) || [];
                const sessionIndex = sessionsForAccount.indexOf(sessionId);
                sessionsForAccount.splice(sessionIndex);
                if (!sessionsForAccount.length) {
                    this.accounts.delete(accountName);
                }
            }
        });
        addedSessions.forEach(s => this.registerSession(s));
    }
    login(scopes) {
        return this.proxy.$login(this.id, scopes);
    }
    async logout(sessionId) {
        await this.proxy.$logout(this.id, sessionId);
        this.messageService.info('Successfully signed out.');
    }
}
exports.AuthenticationProviderImpl = AuthenticationProviderImpl;
async function readAccountUsages(storageService, providerId, accountName) {
    const accountKey = `authentication-${providerId}-${accountName}-usages`;
    const storedUsages = await storageService.getData(accountKey);
    let usages = [];
    if (storedUsages) {
        try {
            usages = JSON.parse(storedUsages);
        }
        catch (e) {
            console.log(e);
        }
    }
    return usages;
}
function removeAccountUsage(storageService, providerId, accountName) {
    const accountKey = `authentication-${providerId}-${accountName}-usages`;
    storageService.setData(accountKey, undefined);
}
//# sourceMappingURL=authentication-main.js.map