/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
var __assign = (this && this.__assign) || function () {
    __assign = Object.assign || function(t) {
        for (var s, i = 1, n = arguments.length; i < n; i++) {
            s = arguments[i];
            for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p))
                t[p] = s[p];
        }
        return t;
    };
    return __assign.apply(this, arguments);
};
var __read = (this && this.__read) || function (o, n) {
    var m = typeof Symbol === "function" && o[Symbol.iterator];
    if (!m) return o;
    var i = m.call(o), r, ar = [], e;
    try {
        while ((n === void 0 || n-- > 0) && !(r = i.next()).done) ar.push(r.value);
    }
    catch (error) { e = { error: error }; }
    finally {
        try {
            if (r && !r.done && (m = i["return"])) m.call(i);
        }
        finally { if (e) throw e.error; }
    }
    return ar;
};
var __spreadArray = (this && this.__spreadArray) || function (to, from) {
    for (var i = 0, il = from.length, j = to.length; i < il; i++, j++)
        to[j] = from[i];
    return to;
};
import { Iterable } from '../../../common/iterator';
import { IndexTreeModel } from './indexTreeModel';
import { TreeError } from './tree';
import { mergeSort } from '../../../common/arrays';
var ObjectTreeModel = /** @class */ (function () {
    function ObjectTreeModel(user, list, options) {
        if (options === void 0) { options = {}; }
        this.user = user;
        this.rootRef = null;
        this.nodes = new Map();
        this.nodesByIdentity = new Map();
        this.model = new IndexTreeModel(user, list, null, options);
        this.onDidSplice = this.model.onDidSplice;
        this.onDidChangeCollapseState = this.model.onDidChangeCollapseState;
        this.onDidChangeRenderNodeCount = this.model.onDidChangeRenderNodeCount;
        if (options.sorter) {
            this.sorter = {
                compare: function (a, b) {
                    return options.sorter.compare(a.element, b.element);
                }
            };
        }
        this.identityProvider = options.identityProvider;
    }
    Object.defineProperty(ObjectTreeModel.prototype, "size", {
        get: function () { return this.nodes.size; },
        enumerable: false,
        configurable: true
    });
    ObjectTreeModel.prototype.setChildren = function (element, children, options) {
        if (children === void 0) { children = Iterable.empty(); }
        if (options === void 0) { options = {}; }
        var location = this.getElementLocation(element);
        this._setChildren(location, this.preserveCollapseState(children), options);
    };
    ObjectTreeModel.prototype._setChildren = function (location, children, options) {
        var _this = this;
        if (children === void 0) { children = Iterable.empty(); }
        var insertedElements = new Set();
        var insertedElementIds = new Set();
        var onDidCreateNode = function (node) {
            var _a;
            if (node.element === null) {
                return;
            }
            var tnode = node;
            insertedElements.add(tnode.element);
            _this.nodes.set(tnode.element, tnode);
            if (_this.identityProvider) {
                var id = _this.identityProvider.getId(tnode.element).toString();
                insertedElementIds.add(id);
                _this.nodesByIdentity.set(id, tnode);
            }
            (_a = options.onDidCreateNode) === null || _a === void 0 ? void 0 : _a.call(options, tnode);
        };
        var onDidDeleteNode = function (node) {
            var _a;
            if (node.element === null) {
                return;
            }
            var tnode = node;
            if (!insertedElements.has(tnode.element)) {
                _this.nodes.delete(tnode.element);
            }
            if (_this.identityProvider) {
                var id = _this.identityProvider.getId(tnode.element).toString();
                if (!insertedElementIds.has(id)) {
                    _this.nodesByIdentity.delete(id);
                }
            }
            (_a = options.onDidDeleteNode) === null || _a === void 0 ? void 0 : _a.call(options, tnode);
        };
        this.model.splice(__spreadArray(__spreadArray([], __read(location)), [0]), Number.MAX_VALUE, children, __assign(__assign({}, options), { onDidCreateNode: onDidCreateNode, onDidDeleteNode: onDidDeleteNode }));
    };
    ObjectTreeModel.prototype.preserveCollapseState = function (elements) {
        var _this = this;
        if (elements === void 0) { elements = Iterable.empty(); }
        if (this.sorter) {
            elements = mergeSort(__spreadArray([], __read(elements)), this.sorter.compare.bind(this.sorter));
        }
        return Iterable.map(elements, function (treeElement) {
            var node = _this.nodes.get(treeElement.element);
            if (!node && _this.identityProvider) {
                var id = _this.identityProvider.getId(treeElement.element).toString();
                node = _this.nodesByIdentity.get(id);
            }
            if (!node) {
                return __assign(__assign({}, treeElement), { children: _this.preserveCollapseState(treeElement.children) });
            }
            var collapsible = typeof treeElement.collapsible === 'boolean' ? treeElement.collapsible : node.collapsible;
            var collapsed = typeof treeElement.collapsed !== 'undefined' ? treeElement.collapsed : node.collapsed;
            return __assign(__assign({}, treeElement), { collapsible: collapsible,
                collapsed: collapsed, children: _this.preserveCollapseState(treeElement.children) });
        });
    };
    ObjectTreeModel.prototype.rerender = function (element) {
        var location = this.getElementLocation(element);
        this.model.rerender(location);
    };
    ObjectTreeModel.prototype.updateElementHeight = function (element, height) {
        var location = this.getElementLocation(element);
        this.model.updateElementHeight(location, height);
    };
    ObjectTreeModel.prototype.resort = function (element, recursive) {
        if (element === void 0) { element = null; }
        if (recursive === void 0) { recursive = true; }
        if (!this.sorter) {
            return;
        }
        var location = this.getElementLocation(element);
        var node = this.model.getNode(location);
        this._setChildren(location, this.resortChildren(node, recursive), {});
    };
    ObjectTreeModel.prototype.resortChildren = function (node, recursive, first) {
        var _this = this;
        if (first === void 0) { first = true; }
        var childrenNodes = __spreadArray([], __read(node.children));
        if (recursive || first) {
            childrenNodes = mergeSort(childrenNodes, this.sorter.compare.bind(this.sorter));
        }
        return Iterable.map(childrenNodes, function (node) { return ({
            element: node.element,
            collapsible: node.collapsible,
            collapsed: node.collapsed,
            children: _this.resortChildren(node, recursive, false)
        }); });
    };
    ObjectTreeModel.prototype.getFirstElementChild = function (ref) {
        if (ref === void 0) { ref = null; }
        var location = this.getElementLocation(ref);
        return this.model.getFirstElementChild(location);
    };
    ObjectTreeModel.prototype.getLastElementAncestor = function (ref) {
        if (ref === void 0) { ref = null; }
        var location = this.getElementLocation(ref);
        return this.model.getLastElementAncestor(location);
    };
    ObjectTreeModel.prototype.has = function (element) {
        return this.nodes.has(element);
    };
    ObjectTreeModel.prototype.getListIndex = function (element) {
        var location = this.getElementLocation(element);
        return this.model.getListIndex(location);
    };
    ObjectTreeModel.prototype.getListRenderCount = function (element) {
        var location = this.getElementLocation(element);
        return this.model.getListRenderCount(location);
    };
    ObjectTreeModel.prototype.isCollapsible = function (element) {
        var location = this.getElementLocation(element);
        return this.model.isCollapsible(location);
    };
    ObjectTreeModel.prototype.setCollapsible = function (element, collapsible) {
        var location = this.getElementLocation(element);
        return this.model.setCollapsible(location, collapsible);
    };
    ObjectTreeModel.prototype.isCollapsed = function (element) {
        var location = this.getElementLocation(element);
        return this.model.isCollapsed(location);
    };
    ObjectTreeModel.prototype.setCollapsed = function (element, collapsed, recursive) {
        var location = this.getElementLocation(element);
        return this.model.setCollapsed(location, collapsed, recursive);
    };
    ObjectTreeModel.prototype.expandTo = function (element) {
        var location = this.getElementLocation(element);
        this.model.expandTo(location);
    };
    ObjectTreeModel.prototype.refilter = function () {
        this.model.refilter();
    };
    ObjectTreeModel.prototype.getNode = function (element) {
        if (element === void 0) { element = null; }
        if (element === null) {
            return this.model.getNode(this.model.rootRef);
        }
        var node = this.nodes.get(element);
        if (!node) {
            throw new TreeError(this.user, "Tree element not found: " + element);
        }
        return node;
    };
    ObjectTreeModel.prototype.getNodeLocation = function (node) {
        return node.element;
    };
    ObjectTreeModel.prototype.getParentNodeLocation = function (element) {
        if (element === null) {
            throw new TreeError(this.user, "Invalid getParentNodeLocation call");
        }
        var node = this.nodes.get(element);
        if (!node) {
            throw new TreeError(this.user, "Tree element not found: " + element);
        }
        var location = this.model.getNodeLocation(node);
        var parentLocation = this.model.getParentNodeLocation(location);
        var parent = this.model.getNode(parentLocation);
        return parent.element;
    };
    ObjectTreeModel.prototype.getElementLocation = function (element) {
        if (element === null) {
            return [];
        }
        var node = this.nodes.get(element);
        if (!node) {
            throw new TreeError(this.user, "Tree element not found: " + element);
        }
        return this.model.getNodeLocation(node);
    };
    return ObjectTreeModel;
}());
export { ObjectTreeModel };
