"use strict";
/********************************************************************************
 * Copyright (C) 2018 Red Hat, Inc. and others.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 ********************************************************************************/
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.DebugSessionManager = void 0;
/* eslint-disable @typescript-eslint/no-explicit-any */
const core_1 = require("@theia/core");
const browser_1 = require("@theia/core/lib/browser");
const context_key_service_1 = require("@theia/core/lib/browser/context-key-service");
const uri_1 = require("@theia/core/lib/common/uri");
const browser_2 = require("@theia/editor/lib/browser");
const quick_open_task_1 = require("@theia/task/lib/browser/quick-open-task");
const task_service_1 = require("@theia/task/lib/browser/task-service");
const browser_3 = require("@theia/variable-resolver/lib/browser");
const inversify_1 = require("@theia/core/shared/inversify");
const debug_service_1 = require("../common/debug-service");
const breakpoint_manager_1 = require("./breakpoint/breakpoint-manager");
const debug_configuration_manager_1 = require("./debug-configuration-manager");
const debug_session_1 = require("./debug-session");
const debug_session_contribution_1 = require("./debug-session-contribution");
const debug_session_options_1 = require("./debug-session-options");
const debug_source_breakpoint_1 = require("./model/debug-source-breakpoint");
const debug_function_breakpoint_1 = require("./model/debug-function-breakpoint");
let DebugSessionManager = class DebugSessionManager {
    constructor() {
        this._sessions = new Map();
        this.onWillStartDebugSessionEmitter = new core_1.Emitter();
        this.onWillStartDebugSession = this.onWillStartDebugSessionEmitter.event;
        this.onWillResolveDebugConfigurationEmitter = new core_1.Emitter();
        this.onWillResolveDebugConfiguration = this.onWillResolveDebugConfigurationEmitter.event;
        this.onDidCreateDebugSessionEmitter = new core_1.Emitter();
        this.onDidCreateDebugSession = this.onDidCreateDebugSessionEmitter.event;
        this.onDidStartDebugSessionEmitter = new core_1.Emitter();
        this.onDidStartDebugSession = this.onDidStartDebugSessionEmitter.event;
        this.onDidStopDebugSessionEmitter = new core_1.Emitter();
        this.onDidStopDebugSession = this.onDidStopDebugSessionEmitter.event;
        this.onDidChangeActiveDebugSessionEmitter = new core_1.Emitter();
        this.onDidChangeActiveDebugSession = this.onDidChangeActiveDebugSessionEmitter.event;
        this.onDidDestroyDebugSessionEmitter = new core_1.Emitter();
        this.onDidDestroyDebugSession = this.onDidDestroyDebugSessionEmitter.event;
        this.onDidReceiveDebugSessionCustomEventEmitter = new core_1.Emitter();
        this.onDidReceiveDebugSessionCustomEvent = this.onDidReceiveDebugSessionCustomEventEmitter.event;
        this.onDidChangeBreakpointsEmitter = new core_1.Emitter();
        this.onDidChangeBreakpoints = this.onDidChangeBreakpointsEmitter.event;
        this.onDidChangeEmitter = new core_1.Emitter();
        this.onDidChange = this.onDidChangeEmitter.event;
        this.configurationIds = new Map();
        this.toDisposeOnCurrentSession = new core_1.DisposableCollection();
    }
    fireDidChangeBreakpoints(event) {
        this.onDidChangeBreakpointsEmitter.fire(event);
    }
    fireDidChange(current) {
        this.inDebugModeKey.set(this.inDebugMode);
        this.onDidChangeEmitter.fire(current);
    }
    init() {
        this.debugTypeKey = this.contextKeyService.createKey('debugType', undefined);
        this.inDebugModeKey = this.contextKeyService.createKey('inDebugMode', this.inDebugMode);
        this.breakpoints.onDidChangeMarkers(uri => this.fireDidChangeBreakpoints({ uri }));
        this.labelProvider.onDidChange(event => {
            for (const uriString of this.breakpoints.getUris()) {
                const uri = new uri_1.default(uriString);
                if (event.affects(uri)) {
                    this.fireDidChangeBreakpoints({ uri });
                }
            }
        });
    }
    get inDebugMode() {
        return this.state > debug_session_1.DebugState.Inactive;
    }
    isCurrentEditorFrame(uri) {
        var _a, _b;
        return ((_b = (_a = this.currentFrame) === null || _a === void 0 ? void 0 : _a.source) === null || _b === void 0 ? void 0 : _b.uri.toString()) === (uri instanceof uri_1.default ? uri : new uri_1.default(uri)).toString();
    }
    async saveAll() {
        if (!this.shell.canSaveAll()) {
            return true; // Nothing to save.
        }
        try {
            await this.shell.saveAll();
            return true;
        }
        catch (error) {
            console.error('saveAll failed:', error);
            return false;
        }
    }
    async start(options) {
        return this.progressService.withProgress('Start...', 'debug', async () => {
            try {
                if (!await this.saveAll()) {
                    return undefined;
                }
                await this.fireWillStartDebugSession();
                const resolved = await this.resolveConfiguration(options);
                // preLaunchTask isn't run in case of auto restart as well as postDebugTask
                if (!options.configuration.__restart) {
                    const taskRun = await this.runTask(options.workspaceFolderUri, resolved.configuration.preLaunchTask, true);
                    if (!taskRun) {
                        return undefined;
                    }
                }
                const sessionId = await this.debug.createDebugSession(resolved.configuration);
                return this.doStart(sessionId, resolved);
            }
            catch (e) {
                if (debug_service_1.DebugError.NotFound.is(e)) {
                    this.messageService.error(`The debug session type "${e.data.type}" is not supported.`);
                    return undefined;
                }
                this.messageService.error('There was an error starting the debug session, check the logs for more details.');
                console.error('Error starting the debug session', e);
                throw e;
            }
        });
    }
    async fireWillStartDebugSession() {
        await core_1.WaitUntilEvent.fire(this.onWillStartDebugSessionEmitter, {});
    }
    async resolveConfiguration(options) {
        if (debug_session_options_1.InternalDebugSessionOptions.is(options)) {
            return options;
        }
        const { workspaceFolderUri } = options;
        let configuration = await this.resolveDebugConfiguration(options.configuration, workspaceFolderUri);
        configuration = await this.variableResolver.resolve(configuration, {
            context: options.workspaceFolderUri ? new uri_1.default(options.workspaceFolderUri) : undefined,
            configurationSection: 'launch'
        });
        configuration = await this.resolveDebugConfigurationWithSubstitutedVariables(configuration, workspaceFolderUri);
        const key = configuration.name + workspaceFolderUri;
        const id = this.configurationIds.has(key) ? this.configurationIds.get(key) + 1 : 0;
        this.configurationIds.set(key, id);
        return {
            id,
            configuration,
            workspaceFolderUri
        };
    }
    async resolveDebugConfiguration(configuration, workspaceFolderUri) {
        await this.fireWillResolveDebugConfiguration(configuration.type);
        return this.debug.resolveDebugConfiguration(configuration, workspaceFolderUri);
    }
    async fireWillResolveDebugConfiguration(debugType) {
        await core_1.WaitUntilEvent.fire(this.onWillResolveDebugConfigurationEmitter, { debugType });
    }
    async resolveDebugConfigurationWithSubstitutedVariables(configuration, workspaceFolderUri) {
        return this.debug.resolveDebugConfigurationWithSubstitutedVariables(configuration, workspaceFolderUri);
    }
    async doStart(sessionId, options) {
        const parentSession = options.configuration.parentSession && this._sessions.get(options.configuration.parentSession.id);
        const contrib = this.sessionContributionRegistry.get(options.configuration.type);
        const sessionFactory = contrib ? contrib.debugSessionFactory() : this.debugSessionFactory;
        const session = sessionFactory.get(sessionId, options, parentSession);
        this._sessions.set(sessionId, session);
        this.debugTypeKey.set(session.configuration.type);
        this.onDidCreateDebugSessionEmitter.fire(session);
        let state = debug_session_1.DebugState.Inactive;
        session.onDidChange(() => {
            if (state !== session.state) {
                state = session.state;
                if (state === debug_session_1.DebugState.Stopped) {
                    this.onDidStopDebugSessionEmitter.fire(session);
                }
            }
            this.updateCurrentSession(session);
        });
        session.onDidChangeBreakpoints(uri => this.fireDidChangeBreakpoints({ session, uri }));
        session.on('terminated', async (event) => {
            const restart = event.body && event.body.restart;
            if (restart) {
                // postDebugTask isn't run in case of auto restart as well as preLaunchTask
                this.doRestart(session, restart);
            }
            else {
                session.terminate();
                await this.runTask(session.options.workspaceFolderUri, session.configuration.postDebugTask);
            }
        });
        session.on('exited', () => this.destroy(session.id));
        session.start().then(() => this.onDidStartDebugSessionEmitter.fire(session));
        session.onDidCustomEvent(({ event, body }) => this.onDidReceiveDebugSessionCustomEventEmitter.fire({ event, body, session }));
        return session;
    }
    async restart(session = this.currentSession) {
        return session && this.doRestart(session);
    }
    async doRestart(session, restart) {
        if (await session.restart()) {
            return session;
        }
        await session.terminate(true);
        const { options, configuration } = session;
        configuration.__restart = restart;
        return this.start(options);
    }
    async terminateSessions() {
        var _a;
        this.updateCurrentSession(undefined);
        (_a = this.currentSession) === null || _a === void 0 ? void 0 : _a.terminate();
    }
    async restartSessions() {
        var _a;
        this.updateCurrentSession(undefined);
        (_a = this.currentSession) === null || _a === void 0 ? void 0 : _a.restart();
    }
    remove(sessionId) {
        this._sessions.delete(sessionId);
        const { currentSession } = this;
        if (currentSession && currentSession.id === sessionId) {
            this.updateCurrentSession(undefined);
        }
    }
    getSession(sessionId) {
        return this._sessions.get(sessionId);
    }
    get sessions() {
        return Array.from(this._sessions.values()).filter(session => session.state > debug_session_1.DebugState.Inactive);
    }
    get currentSession() {
        return this._currentSession;
    }
    set currentSession(current) {
        if (this._currentSession === current) {
            return;
        }
        this.toDisposeOnCurrentSession.dispose();
        const previous = this.currentSession;
        this._currentSession = current;
        this.onDidChangeActiveDebugSessionEmitter.fire({ previous, current });
        if (current) {
            this.toDisposeOnCurrentSession.push(current.onDidChange(() => {
                if (this.currentFrame === this.topFrame) {
                    this.open();
                }
                this.fireDidChange(current);
            }));
        }
        this.updateBreakpoints(previous, current);
        this.open();
        this.fireDidChange(current);
    }
    open() {
        const { currentFrame } = this;
        if (currentFrame) {
            currentFrame.open();
        }
    }
    updateBreakpoints(previous, current) {
        const affectedUri = new Set();
        for (const session of [previous, current]) {
            if (session) {
                for (const uriString of session.breakpointUris) {
                    if (!affectedUri.has(uriString)) {
                        affectedUri.add(uriString);
                        this.fireDidChangeBreakpoints({
                            session: current,
                            uri: new uri_1.default(uriString)
                        });
                    }
                }
            }
        }
    }
    updateCurrentSession(session) {
        this.currentSession = session || this.sessions[0];
    }
    get currentThread() {
        const session = this.currentSession;
        return session && session.currentThread;
    }
    get state() {
        const session = this.currentSession;
        return session ? session.state : debug_session_1.DebugState.Inactive;
    }
    get currentFrame() {
        const { currentThread } = this;
        return currentThread && currentThread.currentFrame;
    }
    get topFrame() {
        const { currentThread } = this;
        return currentThread && currentThread.topFrame;
    }
    /**
     * Destroy the debug session. If session identifier isn't provided then
     * all active debug session will be destroyed.
     * @param sessionId The session identifier
     */
    destroy(sessionId) {
        if (sessionId) {
            const session = this._sessions.get(sessionId);
            if (session) {
                this.doDestroy(session);
            }
        }
        else {
            this._sessions.forEach(session => this.doDestroy(session));
        }
    }
    doDestroy(session) {
        this.debug.terminateDebugSession(session.id);
        session.dispose();
        this.remove(session.id);
        this.onDidDestroyDebugSessionEmitter.fire(session);
    }
    getFunctionBreakpoints(session = this.currentSession) {
        if (session && session.state > debug_session_1.DebugState.Initializing) {
            return session.getFunctionBreakpoints();
        }
        const { labelProvider, breakpoints, editorManager } = this;
        return this.breakpoints.getFunctionBreakpoints().map(origin => new debug_function_breakpoint_1.DebugFunctionBreakpoint(origin, { labelProvider, breakpoints, editorManager }));
    }
    getBreakpoints(arg, arg2) {
        const uri = arg instanceof uri_1.default ? arg : undefined;
        const session = arg instanceof debug_session_1.DebugSession ? arg : arg2 instanceof debug_session_1.DebugSession ? arg2 : this.currentSession;
        if (session && session.state > debug_session_1.DebugState.Initializing) {
            return session.getSourceBreakpoints(uri);
        }
        const { labelProvider, breakpoints, editorManager } = this;
        return this.breakpoints.findMarkers({ uri }).map(({ data }) => new debug_source_breakpoint_1.DebugSourceBreakpoint(data, { labelProvider, breakpoints, editorManager }));
    }
    getLineBreakpoints(uri, line) {
        const session = this.currentSession;
        if (session && session.state > debug_session_1.DebugState.Initializing) {
            return session.getSourceBreakpoints(uri).filter(breakpoint => breakpoint.line === line);
        }
        const { labelProvider, breakpoints, editorManager } = this;
        return this.breakpoints.getLineBreakpoints(uri, line).map(origin => new debug_source_breakpoint_1.DebugSourceBreakpoint(origin, { labelProvider, breakpoints, editorManager }));
    }
    getInlineBreakpoint(uri, line, column) {
        const session = this.currentSession;
        if (session && session.state > debug_session_1.DebugState.Initializing) {
            return session.getSourceBreakpoints(uri).filter(breakpoint => breakpoint.line === line && breakpoint.column === column)[0];
        }
        const origin = this.breakpoints.getInlineBreakpoint(uri, line, column);
        const { labelProvider, breakpoints, editorManager } = this;
        return origin && new debug_source_breakpoint_1.DebugSourceBreakpoint(origin, { labelProvider, breakpoints, editorManager });
    }
    /**
     * Runs the given tasks.
     * @param taskName the task name to run, see [TaskNameResolver](#TaskNameResolver)
     * @return true if it allowed to continue debugging otherwise it returns false
     */
    async runTask(workspaceFolderUri, taskName, checkErrors) {
        if (!taskName) {
            return true;
        }
        const taskInfo = await this.taskService.runWorkspaceTask(this.taskService.startUserAction(), workspaceFolderUri, taskName);
        if (!checkErrors) {
            return true;
        }
        if (!taskInfo) {
            return this.doPostTaskAction(`Could not run the task '${taskName}'.`);
        }
        const getExitCodePromise = this.taskService.getExitCode(taskInfo.taskId).then(result => ({ taskEndedType: task_service_1.TaskEndedTypes.TaskExited, value: result }));
        const isBackgroundTaskEndedPromise = this.taskService.isBackgroundTaskEnded(taskInfo.taskId).then(result => ({ taskEndedType: task_service_1.TaskEndedTypes.BackgroundTaskEnded, value: result }));
        // After start running the task, we wait for the task process to exit and if it is a background task, we also wait for a feedback
        // that a background task is active, as soon as one of the promises fulfills, we can continue and analyze the results.
        const taskEndedInfo = await Promise.race([getExitCodePromise, isBackgroundTaskEndedPromise]);
        if (taskEndedInfo.taskEndedType === task_service_1.TaskEndedTypes.BackgroundTaskEnded && taskEndedInfo.value) {
            return true;
        }
        if (taskEndedInfo.taskEndedType === task_service_1.TaskEndedTypes.TaskExited && taskEndedInfo.value === 0) {
            return true;
        }
        else if (taskEndedInfo.taskEndedType === task_service_1.TaskEndedTypes.TaskExited && taskEndedInfo.value !== undefined) {
            return this.doPostTaskAction(`Task '${taskName}' terminated with exit code ${taskEndedInfo.value}.`);
        }
        else {
            const signal = await this.taskService.getTerminateSignal(taskInfo.taskId);
            if (signal !== undefined) {
                return this.doPostTaskAction(`Task '${taskName}' terminated by signal ${signal}.`);
            }
            else {
                return this.doPostTaskAction(`Task '${taskName}' terminated for unknown reason.`);
            }
        }
    }
    async doPostTaskAction(errorMessage) {
        const actions = ['Open launch.json', 'Cancel', 'Configure Task', 'Debug Anyway'];
        const result = await this.messageService.error(errorMessage, ...actions);
        switch (result) {
            case actions[0]: // open launch.json
                this.debugConfigurationManager.openConfiguration();
                return false;
            case actions[1]: // cancel
                return false;
            case actions[2]: // configure tasks
                this.quickOpenTask.configure();
                return false;
            default: // continue debugging
                return true;
        }
    }
};
__decorate([
    inversify_1.inject(debug_session_contribution_1.DebugSessionFactory),
    __metadata("design:type", Object)
], DebugSessionManager.prototype, "debugSessionFactory", void 0);
__decorate([
    inversify_1.inject(debug_service_1.DebugService),
    __metadata("design:type", Object)
], DebugSessionManager.prototype, "debug", void 0);
__decorate([
    inversify_1.inject(browser_1.LabelProvider),
    __metadata("design:type", browser_1.LabelProvider)
], DebugSessionManager.prototype, "labelProvider", void 0);
__decorate([
    inversify_1.inject(browser_2.EditorManager),
    __metadata("design:type", browser_2.EditorManager)
], DebugSessionManager.prototype, "editorManager", void 0);
__decorate([
    inversify_1.inject(breakpoint_manager_1.BreakpointManager),
    __metadata("design:type", breakpoint_manager_1.BreakpointManager)
], DebugSessionManager.prototype, "breakpoints", void 0);
__decorate([
    inversify_1.inject(browser_3.VariableResolverService),
    __metadata("design:type", browser_3.VariableResolverService)
], DebugSessionManager.prototype, "variableResolver", void 0);
__decorate([
    inversify_1.inject(debug_session_contribution_1.DebugSessionContributionRegistry),
    __metadata("design:type", Object)
], DebugSessionManager.prototype, "sessionContributionRegistry", void 0);
__decorate([
    inversify_1.inject(core_1.MessageService),
    __metadata("design:type", core_1.MessageService)
], DebugSessionManager.prototype, "messageService", void 0);
__decorate([
    inversify_1.inject(core_1.ProgressService),
    __metadata("design:type", core_1.ProgressService)
], DebugSessionManager.prototype, "progressService", void 0);
__decorate([
    inversify_1.inject(context_key_service_1.ContextKeyService),
    __metadata("design:type", context_key_service_1.ContextKeyService)
], DebugSessionManager.prototype, "contextKeyService", void 0);
__decorate([
    inversify_1.inject(task_service_1.TaskService),
    __metadata("design:type", task_service_1.TaskService)
], DebugSessionManager.prototype, "taskService", void 0);
__decorate([
    inversify_1.inject(debug_configuration_manager_1.DebugConfigurationManager),
    __metadata("design:type", debug_configuration_manager_1.DebugConfigurationManager)
], DebugSessionManager.prototype, "debugConfigurationManager", void 0);
__decorate([
    inversify_1.inject(quick_open_task_1.QuickOpenTask),
    __metadata("design:type", quick_open_task_1.QuickOpenTask)
], DebugSessionManager.prototype, "quickOpenTask", void 0);
__decorate([
    inversify_1.inject(browser_1.ApplicationShell),
    __metadata("design:type", browser_1.ApplicationShell)
], DebugSessionManager.prototype, "shell", void 0);
__decorate([
    inversify_1.postConstruct(),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", []),
    __metadata("design:returntype", void 0)
], DebugSessionManager.prototype, "init", null);
DebugSessionManager = __decorate([
    inversify_1.injectable()
], DebugSessionManager);
exports.DebugSessionManager = DebugSessionManager;
//# sourceMappingURL=debug-session-manager.js.map