"use strict";
/********************************************************************************
 * Copyright (C) 2017 TypeFox and others.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 ********************************************************************************/
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.TerminalWidgetImpl = exports.TERMINAL_WIDGET_FACTORY_ID = void 0;
const xterm_1 = require("xterm");
const xterm_addon_fit_1 = require("xterm-addon-fit");
const inversify_1 = require("@theia/core/shared/inversify");
const core_1 = require("@theia/core");
const browser_1 = require("@theia/core/lib/browser");
const common_1 = require("@theia/core/lib/common");
const browser_2 = require("@theia/workspace/lib/browser");
const shell_terminal_protocol_1 = require("../common/shell-terminal-protocol");
const terminal_protocol_1 = require("../common/terminal-protocol");
const base_terminal_protocol_1 = require("../common/base-terminal-protocol");
const terminal_watcher_1 = require("../common/terminal-watcher");
const terminal_widget_1 = require("./base/terminal-widget");
const promise_util_1 = require("@theia/core/lib/common/promise-util");
const terminal_preferences_1 = require("./terminal-preferences");
const terminal_contribution_1 = require("./terminal-contribution");
const uri_1 = require("@theia/core/lib/common/uri");
const terminal_service_1 = require("./base/terminal-service");
const terminal_search_widget_1 = require("./search/terminal-search-widget");
const terminal_copy_on_selection_handler_1 = require("./terminal-copy-on-selection-handler");
const terminal_theme_service_1 = require("./terminal-theme-service");
const shell_command_builder_1 = require("@theia/process/lib/common/shell-command-builder");
const keys_1 = require("@theia/core/lib/browser/keys");
exports.TERMINAL_WIDGET_FACTORY_ID = 'terminal';
let TerminalWidgetImpl = class TerminalWidgetImpl extends terminal_widget_1.TerminalWidget {
    constructor() {
        super(...arguments);
        this.TERMINAL = 'Terminal';
        this.terminalKind = 'user';
        this._terminalId = -1;
        this.onTermDidClose = new core_1.Emitter();
        this.restored = false;
        this.closeOnDispose = true;
        this.isAttachedCloseListener = false;
        this.lastCwd = new uri_1.default();
        this.onDidOpenEmitter = new core_1.Emitter();
        this.onDidOpen = this.onDidOpenEmitter.event;
        this.onDidOpenFailureEmitter = new core_1.Emitter();
        this.onDidOpenFailure = this.onDidOpenFailureEmitter.event;
        this.onSizeChangedEmitter = new core_1.Emitter();
        this.onSizeChanged = this.onSizeChangedEmitter.event;
        this.onDataEmitter = new core_1.Emitter();
        this.onData = this.onDataEmitter.event;
        this.toDisposeOnConnect = new core_1.DisposableCollection();
        this.needsResize = true;
        // Device status code emitted by Xterm.js
        // Check: https://github.com/xtermjs/xterm.js/blob/release/3.14/src/InputHandler.ts#L1055-L1082
        this.deviceStatusCodes = new Set(['\u001B[>0;276;0c', '\u001B[>85;95;0c', '\u001B[>83;40003;0c', '\u001B[?1;2c', '\u001B[?6c']);
        this.termOpened = false;
        this.initialData = '';
    }
    init() {
        this.setTitle(this.options.title || this.TERMINAL);
        this.title.iconClass = 'fa fa-terminal';
        if (this.options.kind) {
            this.terminalKind = this.options.kind;
        }
        if (this.options.destroyTermOnClose === true) {
            this.toDispose.push(core_1.Disposable.create(() => this.term.dispose()));
        }
        this.title.closable = true;
        this.addClass('terminal-container');
        this.term = new xterm_1.Terminal({
            cursorBlink: this.preferences['terminal.integrated.cursorBlinking'],
            cursorStyle: this.getCursorStyle(),
            cursorWidth: this.preferences['terminal.integrated.cursorWidth'],
            fontFamily: this.preferences['terminal.integrated.fontFamily'],
            fontSize: this.preferences['terminal.integrated.fontSize'],
            fontWeight: this.preferences['terminal.integrated.fontWeight'],
            fontWeightBold: this.preferences['terminal.integrated.fontWeightBold'],
            drawBoldTextInBrightColors: this.preferences['terminal.integrated.drawBoldTextInBrightColors'],
            letterSpacing: this.preferences['terminal.integrated.letterSpacing'],
            lineHeight: this.preferences['terminal.integrated.lineHeight'],
            scrollback: this.preferences['terminal.integrated.scrollback'],
            fastScrollSensitivity: this.preferences['terminal.integrated.fastScrollSensitivity'],
            rendererType: this.getTerminalRendererType(this.preferences['terminal.integrated.rendererType']),
            theme: this.themeService.theme
        });
        this.fitAddon = new xterm_addon_fit_1.FitAddon();
        this.term.loadAddon(this.fitAddon);
        this.hoverMessage = document.createElement('div');
        this.hoverMessage.textContent = 'Cmd + click to follow link';
        this.hoverMessage.style.position = 'fixed';
        // TODO use `var(--theia-editorHoverWidget-foreground) with a newer Monaco version
        this.hoverMessage.style.color = 'var(--theia-editorWidget-foreground)';
        this.hoverMessage.style.backgroundColor = 'var(--theia-editorHoverWidget-background)';
        this.hoverMessage.style.borderColor = 'var(--theia-editorHoverWidget-border)';
        this.hoverMessage.style.borderWidth = '0.5px';
        this.hoverMessage.style.borderStyle = 'solid';
        this.hoverMessage.style.padding = '5px';
        // Above the xterm.js canvas layers:
        // https://github.com/xtermjs/xterm.js/blob/ff790236c1b205469f17a21246141f512d844295/src/renderer/Renderer.ts#L41-L46
        this.hoverMessage.style.zIndex = '10';
        // Initially invisible:
        this.hoverMessage.style.display = 'none';
        this.node.appendChild(this.hoverMessage);
        this.toDispose.push(this.preferences.onPreferenceChanged(change => {
            const lastSeparator = change.preferenceName.lastIndexOf('.');
            if (lastSeparator > 0) {
                let preferenceName = change.preferenceName.substr(lastSeparator + 1);
                let preferenceValue = change.newValue;
                if (preferenceName === 'rendererType') {
                    const newRendererType = preferenceValue;
                    if (newRendererType !== this.getTerminalRendererType(newRendererType)) {
                        // Given terminal renderer type is not supported or invalid
                        preferenceValue = terminal_preferences_1.DEFAULT_TERMINAL_RENDERER_TYPE;
                    }
                }
                else if (preferenceName === 'cursorBlinking') {
                    // Convert the terminal preference into a valid `xterm` option
                    preferenceName = 'cursorBlink';
                }
                else if (preferenceName === 'cursorStyle') {
                    preferenceValue = this.getCursorStyle();
                }
                this.term.setOption(preferenceName, preferenceValue);
                this.needsResize = true;
                this.update();
            }
        }));
        this.toDispose.push(this.themeService.onDidChange(() => this.term.setOption('theme', this.themeService.theme)));
        this.attachCustomKeyEventHandler();
        const titleChangeListenerDispose = this.term.onTitleChange((title) => {
            if (this.options.useServerTitle) {
                this.title.label = title;
            }
        });
        this.toDispose.push(titleChangeListenerDispose);
        this.toDispose.push(this.terminalWatcher.onTerminalError(({ terminalId, error }) => {
            if (terminalId === this.terminalId) {
                this.dispose();
                this.onTermDidClose.fire(this);
                this.onTermDidClose.dispose();
                this.logger.error(`The terminal process terminated. Cause: ${error}`);
            }
        }));
        this.toDispose.push(this.terminalWatcher.onTerminalExit(({ terminalId }) => {
            if (terminalId === this.terminalId) {
                this.dispose();
                this.onTermDidClose.fire(this);
                this.onTermDidClose.dispose();
            }
        }));
        this.toDispose.push(this.toDisposeOnConnect);
        this.toDispose.push(this.shellTerminalServer.onDidCloseConnection(() => {
            const disposable = this.shellTerminalServer.onDidOpenConnection(() => {
                disposable.dispose();
                this.reconnectTerminalProcess();
            });
            this.toDispose.push(disposable);
        }));
        this.toDispose.push(this.onTermDidClose);
        this.toDispose.push(this.onDidOpenEmitter);
        this.toDispose.push(this.onDidOpenFailureEmitter);
        this.toDispose.push(this.onSizeChangedEmitter);
        this.toDispose.push(this.onDataEmitter);
        const touchEndListener = (event) => {
            if (this.node.contains(event.target)) {
                this.lastTouchEnd = event;
            }
        };
        document.addEventListener('touchend', touchEndListener, { passive: true });
        this.onDispose(() => {
            document.removeEventListener('touchend', touchEndListener);
        });
        this.toDispose.push(this.term.onSelectionChange(() => {
            if (this.copyOnSelection) {
                this.copyOnSelectionHandler.copy(this.term.getSelection());
            }
        }));
        this.toDispose.push(this.term.onResize(data => {
            this.onSizeChangedEmitter.fire(data);
        }));
        this.toDispose.push(this.term.onData(data => {
            this.onDataEmitter.fire(data);
        }));
        for (const contribution of this.terminalContributionProvider.getContributions()) {
            contribution.onCreate(this);
        }
        this.searchBox = this.terminalSearchBoxFactory(this.term);
        this.toDispose.push(this.searchBox);
    }
    get kind() {
        return this.terminalKind;
    }
    /**
     * Get the cursor style compatible with `xterm`.
     * @returns CursorStyle
     */
    getCursorStyle() {
        const value = this.preferences['terminal.integrated.cursorStyle'];
        return value === 'line' ? 'bar' : value;
    }
    /**
     * Returns given renderer type if it is valid and supported or default renderer otherwise.
     *
     * @param terminalRendererType desired terminal renderer type
     */
    getTerminalRendererType(terminalRendererType) {
        if (terminalRendererType && terminal_preferences_1.isTerminalRendererType(terminalRendererType)) {
            return terminalRendererType;
        }
        return terminal_preferences_1.DEFAULT_TERMINAL_RENDERER_TYPE;
    }
    showHoverMessage(x, y, message) {
        this.hoverMessage.innerText = message;
        this.hoverMessage.style.display = 'inline';
        this.hoverMessage.style.top = `${y - 30}px`;
        this.hoverMessage.style.left = `${x - 60}px`;
    }
    hideHover() {
        this.hoverMessage.style.display = 'none';
    }
    getTerminal() {
        return this.term;
    }
    getSearchBox() {
        return this.searchBox;
    }
    get dimensions() {
        return {
            cols: this.term.cols,
            rows: this.term.rows,
        };
    }
    get cwd() {
        if (!base_terminal_protocol_1.IBaseTerminalServer.validateId(this.terminalId)) {
            return Promise.reject(new Error('terminal is not started'));
        }
        if (this.terminalService.getById(this.id)) {
            return this.shellTerminalServer.getCwdURI(this.terminalId)
                .then(cwdUrl => {
                this.lastCwd = new uri_1.default(cwdUrl);
                return this.lastCwd;
            }).catch(() => this.lastCwd);
        }
        return Promise.resolve(new uri_1.default());
    }
    get processId() {
        if (!base_terminal_protocol_1.IBaseTerminalServer.validateId(this.terminalId)) {
            return Promise.reject(new Error('terminal is not started'));
        }
        return this.shellTerminalServer.getProcessId(this.terminalId);
    }
    get processInfo() {
        if (!base_terminal_protocol_1.IBaseTerminalServer.validateId(this.terminalId)) {
            return Promise.reject(new Error('terminal is not started'));
        }
        return this.shellTerminalServer.getProcessInfo(this.terminalId);
    }
    get terminalId() {
        return this._terminalId;
    }
    get lastTouchEndEvent() {
        return this.lastTouchEnd;
    }
    onDispose(onDispose) {
        this.toDispose.push(core_1.Disposable.create(onDispose));
    }
    clearOutput() {
        this.term.clear();
    }
    async hasChildProcesses() {
        return this.shellTerminalServer.hasChildProcesses(await this.processId);
    }
    storeState() {
        this.closeOnDispose = false;
        if (this.options.isPseudoTerminal) {
            return {};
        }
        return { terminalId: this.terminalId, titleLabel: this.title.label };
    }
    restoreState(oldState) {
        // pseudo terminal can not restore
        if (this.options.isPseudoTerminal) {
            this.dispose();
            return;
        }
        if (this.restored === false) {
            const state = oldState;
            /* This is a workaround to issue #879 */
            this.restored = true;
            this.title.label = state.titleLabel;
            this.start(state.terminalId);
        }
    }
    /**
     * Create a new shell terminal in the back-end and attach it to a
     * new terminal widget.
     * If id is provided attach to the terminal for this id.
     */
    async start(id) {
        this._terminalId = typeof id !== 'number' ? await this.createTerminal() : await this.attachTerminal(id);
        this.resizeTerminalProcess();
        this.connectTerminalProcess();
        if (base_terminal_protocol_1.IBaseTerminalServer.validateId(this.terminalId)) {
            this.onDidOpenEmitter.fire(undefined);
            await this.shellTerminalServer.onAttachAttempted(this._terminalId);
            return this.terminalId;
        }
        this.onDidOpenFailureEmitter.fire(undefined);
        throw new Error('Failed to start terminal' + (id ? ` for id: ${id}.` : '.'));
    }
    async attachTerminal(id) {
        const terminalId = await this.shellTerminalServer.attach(id);
        if (base_terminal_protocol_1.IBaseTerminalServer.validateId(terminalId)) {
            return terminalId;
        }
        this.logger.warn(`Failed attaching to terminal id ${id}, the terminal is most likely gone. Starting up a new terminal instead.`);
        if (this.kind === 'user') {
            return this.createTerminal();
        }
        else {
            return -1;
        }
    }
    async createTerminal() {
        var _a, _b;
        let rootURI = (_a = this.options.cwd) === null || _a === void 0 ? void 0 : _a.toString();
        if (!rootURI) {
            const root = (await this.workspaceService.roots)[0];
            rootURI = (_b = root === null || root === void 0 ? void 0 : root.resource) === null || _b === void 0 ? void 0 : _b.toString();
        }
        const { cols, rows } = this.term;
        const terminalId = await this.shellTerminalServer.create({
            shellPreferences: this.shellPreferences,
            shell: this.options.shellPath,
            args: this.options.shellArgs,
            env: this.options.env,
            isPseudo: this.options.isPseudoTerminal,
            rootURI,
            cols,
            rows
        });
        if (base_terminal_protocol_1.IBaseTerminalServer.validateId(terminalId)) {
            return terminalId;
        }
        throw new Error('Error creating terminal widget, see the backend error log for more information.');
    }
    processMessage(msg) {
        super.processMessage(msg);
        switch (msg.type) {
            case 'fit-request':
                this.onFitRequest(msg);
                break;
            default:
                break;
        }
    }
    onFitRequest(msg) {
        super.onFitRequest(msg);
        browser_1.MessageLoop.sendMessage(this, browser_1.Widget.ResizeMessage.UnknownSize);
    }
    onActivateRequest(msg) {
        super.onActivateRequest(msg);
        this.term.focus();
    }
    onAfterShow(msg) {
        super.onAfterShow(msg);
        this.update();
    }
    onAfterAttach(msg) {
        browser_1.Widget.attach(this.searchBox, this.node);
        super.onAfterAttach(msg);
        this.update();
    }
    onBeforeDetach(msg) {
        browser_1.Widget.detach(this.searchBox);
        super.onBeforeDetach(msg);
    }
    onResize(msg) {
        super.onResize(msg);
        this.needsResize = true;
        this.update();
    }
    onUpdateRequest(msg) {
        super.onUpdateRequest(msg);
        if (!this.isVisible || !this.isAttached) {
            return;
        }
        this.open();
        if (this.needsResize) {
            this.resizeTerminal();
            this.needsResize = false;
            this.resizeTerminalProcess();
        }
    }
    connectTerminalProcess() {
        if (typeof this.terminalId !== 'number') {
            return;
        }
        if (this.options.isPseudoTerminal) {
            return;
        }
        this.toDisposeOnConnect.dispose();
        this.toDispose.push(this.toDisposeOnConnect);
        const waitForConnection = this.waitForConnection = new promise_util_1.Deferred();
        this.webSocketConnectionProvider.listen({
            path: `${terminal_protocol_1.terminalsPath}/${this.terminalId}`,
            onConnection: connection => {
                connection.onNotification('onData', (data) => this.write(data));
                // Excludes the device status code emitted by Xterm.js
                const sendData = (data) => {
                    if (data && !this.deviceStatusCodes.has(data) && !this.disableEnterWhenAttachCloseListener()) {
                        return connection.sendRequest('write', data);
                    }
                };
                const disposable = this.term.onData(sendData);
                connection.onDispose(() => disposable.dispose());
                this.toDisposeOnConnect.push(connection);
                connection.listen();
                if (waitForConnection) {
                    waitForConnection.resolve(connection);
                }
            }
        }, { reconnecting: false });
    }
    async reconnectTerminalProcess() {
        if (this.options.isPseudoTerminal) {
            return;
        }
        if (typeof this.terminalId === 'number') {
            await this.start(this.terminalId);
        }
    }
    open() {
        if (this.termOpened) {
            return;
        }
        this.term.open(this.node);
        if (this.initialData) {
            this.term.write(this.initialData);
        }
        this.termOpened = true;
        this.initialData = '';
        if (browser_1.isFirefox) {
            // The software scrollbars don't work with xterm.js, so we disable the scrollbar if we are on firefox.
            if (this.term.element) {
                this.term.element.children.item(0).style.overflow = 'hidden';
            }
        }
    }
    write(data) {
        if (this.termOpened) {
            this.term.write(data);
        }
        else {
            this.initialData += data;
        }
    }
    resize(cols, rows) {
        this.term.resize(cols, rows);
    }
    sendText(text) {
        if (this.waitForConnection) {
            this.waitForConnection.promise.then(connection => connection.sendRequest('write', text));
        }
    }
    async executeCommand(commandOptions) {
        this.sendText(this.shellCommandBuilder.buildCommand(await this.processInfo, commandOptions) + '\n');
    }
    scrollLineUp() {
        this.term.scrollLines(-1);
    }
    scrollLineDown() {
        this.term.scrollLines(1);
    }
    scrollToTop() {
        this.term.scrollToTop();
    }
    scrollToBottom() {
        this.term.scrollToBottom();
    }
    scrollPageUp() {
        this.term.scrollPages(-1);
    }
    scrollPageDown() {
        this.term.scrollPages(1);
    }
    resetTerminal() {
        this.term.reset();
    }
    writeLine(text) {
        this.term.writeln(text);
    }
    get onTerminalDidClose() {
        return this.onTermDidClose.event;
    }
    dispose() {
        /* Close the backend terminal only when explicitly closing the terminal
         * a refresh for example won't close it.  */
        if (this.closeOnDispose === true && typeof this.terminalId === 'number') {
            this.shellTerminalServer.close(this.terminalId);
            this.onTermDidClose.fire(this);
        }
        super.dispose();
    }
    resizeTerminal() {
        const geo = this.fitAddon.proposeDimensions();
        const cols = geo.cols;
        const rows = geo.rows - 1; // subtract one row for margin
        this.term.resize(cols, rows);
    }
    resizeTerminalProcess() {
        if (this.options.isPseudoTerminal) {
            return;
        }
        if (!base_terminal_protocol_1.IBaseTerminalServer.validateId(this.terminalId)
            || !this.terminalService.getById(this.id)) {
            return;
        }
        const { cols, rows } = this.term;
        this.shellTerminalServer.resize(this.terminalId, cols, rows);
    }
    get enableCopy() {
        return this.preferences['terminal.enableCopy'];
    }
    get enablePaste() {
        return this.preferences['terminal.enablePaste'];
    }
    get shellPreferences() {
        var _a, _b, _c;
        return {
            shell: {
                Windows: (_a = this.preferences['terminal.integrated.shell.windows']) !== null && _a !== void 0 ? _a : undefined,
                Linux: (_b = this.preferences['terminal.integrated.shell.linux']) !== null && _b !== void 0 ? _b : undefined,
                OSX: (_c = this.preferences['terminal.integrated.shell.osx']) !== null && _c !== void 0 ? _c : undefined,
            },
            shellArgs: {
                Windows: this.preferences['terminal.integrated.shellArgs.windows'],
                Linux: this.preferences['terminal.integrated.shellArgs.linux'],
                OSX: this.preferences['terminal.integrated.shellArgs.osx'],
            }
        };
    }
    customKeyHandler(event) {
        const keyBindings = browser_1.KeyCode.createKeyCode(event).toString();
        const ctrlCmdCopy = (common_1.isOSX && keyBindings === 'meta+c') || (!common_1.isOSX && keyBindings === 'ctrl+c');
        const ctrlCmdPaste = (common_1.isOSX && keyBindings === 'meta+v') || (!common_1.isOSX && keyBindings === 'ctrl+v');
        if (ctrlCmdCopy && this.enableCopy && this.term.hasSelection()) {
            return false;
        }
        if (ctrlCmdPaste && this.enablePaste) {
            return false;
        }
        return true;
    }
    get copyOnSelection() {
        return this.preferences['terminal.integrated.copyOnSelection'];
    }
    attachCustomKeyEventHandler() {
        this.term.attachCustomKeyEventHandler(e => this.customKeyHandler(e));
    }
    setTitle(title) {
        this.title.caption = title;
        this.title.label = title;
    }
    waitOnExit(waitOnExit) {
        if (waitOnExit) {
            if (typeof waitOnExit === 'string') {
                let message = waitOnExit;
                // Bold the message and add an extra new line to make it stand out from the rest of the output
                message = `\r\n\x1b[1m${message}\x1b[0m`;
                this.write(message);
            }
            if (this.closeOnDispose === true && typeof this.terminalId === 'number') {
                this.shellTerminalServer.close(this.terminalId);
                this.onTermDidClose.fire(this);
            }
            this.attachPressEnterKeyToCloseListener(this.term);
            return;
        }
        this.dispose();
    }
    attachPressEnterKeyToCloseListener(term) {
        if (term.textarea) {
            this.isAttachedCloseListener = true;
            this.addKeyListener(term.textarea, keys_1.Key.ENTER, (event) => {
                this.dispose();
                this.isAttachedCloseListener = false;
            });
        }
    }
    disableEnterWhenAttachCloseListener() {
        return this.isAttachedCloseListener;
    }
};
__decorate([
    inversify_1.inject(browser_2.WorkspaceService),
    __metadata("design:type", browser_2.WorkspaceService)
], TerminalWidgetImpl.prototype, "workspaceService", void 0);
__decorate([
    inversify_1.inject(browser_1.WebSocketConnectionProvider),
    __metadata("design:type", browser_1.WebSocketConnectionProvider)
], TerminalWidgetImpl.prototype, "webSocketConnectionProvider", void 0);
__decorate([
    inversify_1.inject(terminal_widget_1.TerminalWidgetOptions),
    __metadata("design:type", Object)
], TerminalWidgetImpl.prototype, "options", void 0);
__decorate([
    inversify_1.inject(shell_terminal_protocol_1.ShellTerminalServerProxy),
    __metadata("design:type", Object)
], TerminalWidgetImpl.prototype, "shellTerminalServer", void 0);
__decorate([
    inversify_1.inject(terminal_watcher_1.TerminalWatcher),
    __metadata("design:type", terminal_watcher_1.TerminalWatcher)
], TerminalWidgetImpl.prototype, "terminalWatcher", void 0);
__decorate([
    inversify_1.inject(core_1.ILogger),
    inversify_1.named('terminal'),
    __metadata("design:type", Object)
], TerminalWidgetImpl.prototype, "logger", void 0);
__decorate([
    inversify_1.inject('terminal-dom-id'),
    __metadata("design:type", String)
], TerminalWidgetImpl.prototype, "id", void 0);
__decorate([
    inversify_1.inject(terminal_preferences_1.TerminalPreferences),
    __metadata("design:type", Object)
], TerminalWidgetImpl.prototype, "preferences", void 0);
__decorate([
    inversify_1.inject(core_1.ContributionProvider),
    inversify_1.named(terminal_contribution_1.TerminalContribution),
    __metadata("design:type", Object)
], TerminalWidgetImpl.prototype, "terminalContributionProvider", void 0);
__decorate([
    inversify_1.inject(terminal_service_1.TerminalService),
    __metadata("design:type", Object)
], TerminalWidgetImpl.prototype, "terminalService", void 0);
__decorate([
    inversify_1.inject(terminal_search_widget_1.TerminalSearchWidgetFactory),
    __metadata("design:type", Function)
], TerminalWidgetImpl.prototype, "terminalSearchBoxFactory", void 0);
__decorate([
    inversify_1.inject(terminal_copy_on_selection_handler_1.TerminalCopyOnSelectionHandler),
    __metadata("design:type", terminal_copy_on_selection_handler_1.TerminalCopyOnSelectionHandler)
], TerminalWidgetImpl.prototype, "copyOnSelectionHandler", void 0);
__decorate([
    inversify_1.inject(terminal_theme_service_1.TerminalThemeService),
    __metadata("design:type", terminal_theme_service_1.TerminalThemeService)
], TerminalWidgetImpl.prototype, "themeService", void 0);
__decorate([
    inversify_1.inject(shell_command_builder_1.ShellCommandBuilder),
    __metadata("design:type", shell_command_builder_1.ShellCommandBuilder)
], TerminalWidgetImpl.prototype, "shellCommandBuilder", void 0);
__decorate([
    inversify_1.postConstruct(),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", []),
    __metadata("design:returntype", void 0)
], TerminalWidgetImpl.prototype, "init", null);
TerminalWidgetImpl = __decorate([
    inversify_1.injectable()
], TerminalWidgetImpl);
exports.TerminalWidgetImpl = TerminalWidgetImpl;
//# sourceMappingURL=terminal-widget-impl.js.map