"use strict";
/********************************************************************************
 * Copyright (C) 2019 Ericsson and others.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 ********************************************************************************/
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.ProvidedTaskConfigurations = void 0;
const inversify_1 = require("@theia/core/shared/inversify");
const task_contribution_1 = require("./task-contribution");
const task_definition_registry_1 = require("./task-definition-registry");
const common_1 = require("../common");
let ProvidedTaskConfigurations = class ProvidedTaskConfigurations {
    constructor() {
        /**
         * Map of source (name of extension, or path of root folder that the task config comes from) and `task config map`.
         * For the second level of inner map, the key is task label.
         * For the third level of inner map, the key is the task scope and value TaskConfiguration.
         */
        this.tasksMap = new Map();
        this.currentToken = 0;
        this.nextToken = 1;
    }
    startUserAction() {
        return this.nextToken++;
    }
    /** returns a list of provided tasks */
    async getTasks(token) {
        await this.refreshTasks(token);
        const tasks = [];
        for (const taskLabelMap of this.tasksMap.values()) {
            for (const taskScopeMap of taskLabelMap.values()) {
                for (const task of taskScopeMap.values()) {
                    tasks.push(task);
                }
            }
        }
        return tasks;
    }
    async refreshTasks(token) {
        if (token !== this.currentToken) {
            this.currentToken = token;
            const providers = await this.taskProviderRegistry.getProviders();
            const providedTasks = (await Promise.all(providers.map(p => p.provideTasks())))
                .reduce((acc, taskArray) => acc.concat(taskArray), [])
                // Global/User tasks from providers are not supported.
                .filter(task => task.scope !== common_1.TaskScope.Global)
                .map(providedTask => {
                const originalPresentation = providedTask.presentation || {};
                return Object.assign(Object.assign({}, providedTask), { presentation: Object.assign(Object.assign({}, common_1.TaskOutputPresentation.getDefault()), originalPresentation) });
            });
            this.cacheTasks(providedTasks);
        }
    }
    /** returns the task configuration for a given source and label or undefined if none */
    async getTask(token, source, taskLabel, scope) {
        await this.refreshTasks(token);
        return this.getCachedTask(source, taskLabel, scope);
    }
    /**
     * Finds the detected task for the given task customization.
     * The detected task is considered as a "match" to the task customization if it has all the `required` properties.
     * In case that more than one customization is found, return the one that has the biggest number of matched properties.
     *
     * @param customization the task customization
     * @return the detected task for the given task customization. If the task customization is not found, `undefined` is returned.
     */
    async getTaskToCustomize(token, customization, scope) {
        const definition = this.taskDefinitionRegistry.getDefinition(customization);
        if (!definition) {
            return undefined;
        }
        const matchedTasks = [];
        let highest = -1;
        const tasks = await this.getTasks(token);
        for (const task of tasks) { // find detected tasks that match the `definition`
            let score = 0;
            if (!definition.properties.required.every(requiredProp => customization[requiredProp] !== undefined)) {
                continue;
            }
            score += definition.properties.required.length; // number of required properties
            const requiredProps = new Set(definition.properties.required);
            // number of optional properties
            score += definition.properties.all.filter(p => !requiredProps.has(p) && customization[p] !== undefined).length;
            if (score >= highest) {
                if (score > highest) {
                    highest = score;
                    matchedTasks.length = 0;
                }
                matchedTasks.push(task);
            }
        }
        // Tasks with scope set to 'Workspace' can be customized in a workspace root, and will not match
        // providers scope 'TaskScope.Workspace' unless specifically included as below.
        const scopes = [scope, common_1.TaskScope.Workspace];
        // find the task that matches the `customization`.
        // The scenario where more than one match is found should not happen unless users manually enter multiple customizations for one type of task
        // If this does happen, return the first match
        const matchedTask = matchedTasks.find(t => scopes.some(scp => scp === t._scope) && definition.properties.all.every(p => t[p] === customization[p]));
        return matchedTask;
    }
    getCachedTask(source, taskLabel, scope) {
        const labelConfigMap = this.tasksMap.get(source);
        if (labelConfigMap) {
            const scopeConfigMap = labelConfigMap.get(taskLabel);
            if (scopeConfigMap) {
                if (scope) {
                    return scopeConfigMap.get(scope.toString());
                }
                return Array.from(scopeConfigMap.values())[0];
            }
        }
    }
    cacheTasks(tasks) {
        this.tasksMap.clear();
        for (const task of tasks) {
            const label = task.label;
            const source = task._source;
            const scope = task._scope;
            if (this.tasksMap.has(source)) {
                const labelConfigMap = this.tasksMap.get(source);
                if (labelConfigMap.has(label)) {
                    labelConfigMap.get(label).set(scope.toString(), task);
                }
                else {
                    const newScopeConfigMap = new Map();
                    newScopeConfigMap.set(scope.toString(), task);
                    labelConfigMap.set(label, newScopeConfigMap);
                }
            }
            else {
                const newLabelConfigMap = new Map();
                const newScopeConfigMap = new Map();
                newScopeConfigMap.set(scope.toString(), task);
                newLabelConfigMap.set(label, newScopeConfigMap);
                this.tasksMap.set(source, newLabelConfigMap);
            }
        }
    }
};
__decorate([
    inversify_1.inject(task_contribution_1.TaskProviderRegistry),
    __metadata("design:type", task_contribution_1.TaskProviderRegistry)
], ProvidedTaskConfigurations.prototype, "taskProviderRegistry", void 0);
__decorate([
    inversify_1.inject(task_definition_registry_1.TaskDefinitionRegistry),
    __metadata("design:type", task_definition_registry_1.TaskDefinitionRegistry)
], ProvidedTaskConfigurations.prototype, "taskDefinitionRegistry", void 0);
ProvidedTaskConfigurations = __decorate([
    inversify_1.injectable()
], ProvidedTaskConfigurations);
exports.ProvidedTaskConfigurations = ProvidedTaskConfigurations;
//# sourceMappingURL=provided-task-configurations.js.map