"use strict";
/********************************************************************************
 * Copyright (C) 2018 Red Hat, Inc. and others.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 ********************************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConnectionMainImpl = void 0;
const disposable_1 = require("@theia/core/lib/common/disposable");
const plugin_api_rpc_1 = require("../../common/plugin-api-rpc");
const connection_1 = require("../../common/connection");
const plugin_message_reader_1 = require("../../common/plugin-message-reader");
const plugin_message_writer_1 = require("../../common/plugin-message-writer");
/**
 * Implementation of connection system of the plugin API.
 * Creates holds the connections to the plugins. Allows to send a message to the plugin by getting already created connection via id.
 */
class ConnectionMainImpl {
    constructor(rpc) {
        this.connections = new Map();
        this.toDispose = new disposable_1.DisposableCollection();
        this.proxy = rpc.getProxy(plugin_api_rpc_1.MAIN_RPC_CONTEXT.CONNECTION_EXT);
    }
    dispose() {
        this.toDispose.dispose();
    }
    /**
     * Gets the connection between plugin by id and sends string message to it.
     *
     * @param id connection's id
     * @param message incoming message
     */
    async $sendMessage(id, message) {
        if (this.connections.has(id)) {
            this.connections.get(id).reader.readMessage(message);
        }
        else {
            console.warn('It is not possible to read message. Connection missed.');
        }
    }
    /**
     * Instantiates a new connection by the given id.
     * @param id the connection id
     */
    async $createConnection(id) {
        await this.doEnsureConnection(id);
    }
    /**
     * Deletes a connection.
     * @param id the connection id
     */
    async $deleteConnection(id) {
        this.connections.delete(id);
    }
    /**
     * Returns existed connection or creates a new one.
     * @param id the connection id
     */
    async ensureConnection(id) {
        const connection = await this.doEnsureConnection(id);
        await this.proxy.$createConnection(id);
        return connection;
    }
    /**
     * Returns existed connection or creates a new one.
     * @param id the connection id
     */
    async doEnsureConnection(id) {
        const connection = this.connections.get(id) || await this.doCreateConnection(id);
        this.connections.set(id, connection);
        return connection;
    }
    async doCreateConnection(id) {
        const reader = new plugin_message_reader_1.PluginMessageReader();
        const writer = new plugin_message_writer_1.PluginMessageWriter(id, this.proxy);
        const connection = new connection_1.PluginConnection(reader, writer, () => {
            this.connections.delete(id);
            if (!toClose.disposed) {
                this.proxy.$deleteConnection(id);
            }
        });
        const toClose = new disposable_1.DisposableCollection(disposable_1.Disposable.create(() => reader.fireClose()));
        this.toDispose.push(toClose);
        return connection;
    }
}
exports.ConnectionMainImpl = ConnectionMainImpl;
//# sourceMappingURL=connection-main.js.map