"use strict";
/********************************************************************************
 * Copyright (C) 2018 Red Hat, Inc. and others.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 ********************************************************************************/
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
var HostedPluginController_1;
Object.defineProperty(exports, "__esModule", { value: true });
exports.HostedPluginController = void 0;
const inversify_1 = require("@theia/core/shared/inversify");
const status_bar_1 = require("@theia/core/lib/browser/status-bar/status-bar");
const browser_1 = require("@theia/core/lib/browser");
const common_1 = require("@theia/core/lib/common");
const commands_1 = require("@theia/core/shared/@phosphor/commands");
const widgets_1 = require("@theia/core/shared/@phosphor/widgets");
const frontend_application_state_1 = require("@theia/core/lib/browser/frontend-application-state");
const connection_status_service_1 = require("@theia/core/lib/browser/connection-status-service");
const plugin_dev_protocol_1 = require("../common/plugin-dev-protocol");
const hosted_plugin_manager_client_1 = require("./hosted-plugin-manager-client");
const hosted_plugin_log_viewer_1 = require("./hosted-plugin-log-viewer");
const hosted_plugin_preferences_1 = require("./hosted-plugin-preferences");
/**
 * Adds a status bar element displaying the state of secondary Theia instance with hosted plugin and
 * allows controlling the instance by simple clicking on the status bar element.
 */
let HostedPluginController = HostedPluginController_1 = class HostedPluginController {
    constructor() {
        this.pluginState = hosted_plugin_manager_client_1.HostedInstanceState.STOPPED;
    }
    initialize() {
        this.hostedPluginServer.getHostedPlugin().then(pluginMetadata => {
            if (!pluginMetadata) {
                this.frontendApplicationStateService.reachedState('ready').then(() => {
                    // handles status bar item
                    this.hostedPluginManagerClient.onStateChanged(e => {
                        if (e.state === hosted_plugin_manager_client_1.HostedInstanceState.STARTING) {
                            this.onHostedPluginStarting();
                        }
                        else if (e.state === hosted_plugin_manager_client_1.HostedInstanceState.RUNNING) {
                            this.onHostedPluginRunning();
                        }
                        else if (e.state === hosted_plugin_manager_client_1.HostedInstanceState.STOPPED) {
                            this.onHostedPluginStopped();
                        }
                        else if (e.state === hosted_plugin_manager_client_1.HostedInstanceState.FAILED) {
                            this.onHostedPluginFailed();
                        }
                    });
                    // handles watch compilation
                    this.hostedPluginManagerClient.onStateChanged(e => this.handleWatchers(e));
                    // updates status bar if page is loading when hosted instance is already running
                    this.hostedPluginServer.isHostedPluginInstanceRunning().then(running => {
                        if (running) {
                            this.onHostedPluginRunning();
                        }
                    });
                });
                this.connectionStatusService.onStatusChange(() => this.onConnectionStatusChanged());
                this.preferenceService.onPreferenceChanged(preference => this.onPreferencesChanged(preference));
            }
            else {
                console.error(`Need to load plugin ${pluginMetadata.model.id}`);
            }
        });
    }
    /**
     * Display status bar element for stopped plugin.
     */
    async onHostedPluginStopped() {
        this.pluginState = hosted_plugin_manager_client_1.HostedInstanceState.STOPPED;
        this.entry = {
            text: 'Hosted Plugin: Stopped $(angle-up)',
            alignment: browser_1.StatusBarAlignment.LEFT,
            priority: 100,
            onclick: e => {
                this.showMenu(e.clientX, e.clientY);
            }
        };
        this.entry.className = HostedPluginController_1.HOSTED_PLUGIN;
        await this.statusBar.setElement(HostedPluginController_1.HOSTED_PLUGIN, this.entry);
    }
    /**
     * Display status bar element for starting plugin.
     */
    async onHostedPluginStarting() {
        this.pluginState = hosted_plugin_manager_client_1.HostedInstanceState.STARTING;
        this.hostedPluginLogViewer.showLogConsole();
        this.entry = {
            text: '$(cog~spin) Hosted Plugin: Starting',
            alignment: browser_1.StatusBarAlignment.LEFT,
            priority: 100
        };
        this.entry.className = HostedPluginController_1.HOSTED_PLUGIN;
        await this.statusBar.setElement(HostedPluginController_1.HOSTED_PLUGIN, this.entry);
    }
    /**
     * Display status bar element for running plugin.
     */
    async onHostedPluginRunning() {
        this.pluginState = hosted_plugin_manager_client_1.HostedInstanceState.RUNNING;
        let entryText;
        if (this.hostedPluginPreferences['hosted-plugin.watchMode'] && this.watcherSuccess) {
            entryText = '$(cog~spin) Hosted Plugin: Watching $(angle-up)';
        }
        else {
            entryText = '$(cog~spin) Hosted Plugin: Running $(angle-up)';
        }
        this.entry = {
            text: entryText,
            alignment: browser_1.StatusBarAlignment.LEFT,
            priority: 100,
            onclick: e => {
                this.showMenu(e.clientX, e.clientY);
            }
        };
        this.entry.className = HostedPluginController_1.HOSTED_PLUGIN;
        await this.statusBar.setElement(HostedPluginController_1.HOSTED_PLUGIN, this.entry);
    }
    /**
     * Display status bar element for failed plugin.
     */
    async onHostedPluginFailed() {
        this.pluginState = hosted_plugin_manager_client_1.HostedInstanceState.FAILED;
        this.entry = {
            text: 'Hosted Plugin: Stopped $(angle-up)',
            alignment: browser_1.StatusBarAlignment.LEFT,
            priority: 100,
            onclick: e => {
                this.showMenu(e.clientX, e.clientY);
            }
        };
        this.entry.className = HostedPluginController_1.HOSTED_PLUGIN_FAILED;
        await this.statusBar.setElement(HostedPluginController_1.HOSTED_PLUGIN, this.entry);
    }
    async onPreferencesChanged(preference) {
        if (preference.preferenceName === 'hosted-plugin.watchMode') {
            if (await this.hostedPluginServer.isHostedPluginInstanceRunning()) {
                const pluginLocation = await this.hostedPluginServer.getHostedPluginURI();
                const isWatchCompilationRunning = await this.hostedPluginServer.isWatchCompilationRunning(pluginLocation);
                if (preference.newValue === true) {
                    if (!isWatchCompilationRunning) {
                        await this.runWatchCompilation(pluginLocation.toString());
                    }
                }
                else {
                    if (isWatchCompilationRunning) {
                        await this.hostedPluginServer.stopWatchCompilation(pluginLocation.toString());
                    }
                }
                // update status bar
                this.onHostedPluginRunning();
            }
        }
    }
    /**
     * Starts / stops watchers on hosted instance state change.
     *
     * @param event hosted instance state change event
     */
    async handleWatchers(event) {
        if (event.state === hosted_plugin_manager_client_1.HostedInstanceState.RUNNING) {
            if (this.hostedPluginPreferences['hosted-plugin.watchMode']) {
                await this.runWatchCompilation(event.pluginLocation.toString());
                // update status bar
                this.onHostedPluginRunning();
            }
        }
        else if (event.state === hosted_plugin_manager_client_1.HostedInstanceState.STOPPING) {
            if (this.hostedPluginPreferences['hosted-plugin.watchMode']) {
                const isRunning = await this.hostedPluginServer.isWatchCompilationRunning(event.pluginLocation.toString());
                if (isRunning) {
                    try {
                        await this.hostedPluginServer.stopWatchCompilation(event.pluginLocation.toString());
                    }
                    catch (error) {
                        this.messageService.error(this.getErrorMessage(error));
                    }
                }
            }
        }
    }
    async runWatchCompilation(pluginLocation) {
        try {
            await this.hostedPluginServer.runWatchCompilation(pluginLocation);
            this.watcherSuccess = true;
        }
        catch (error) {
            this.messageService.error(this.getErrorMessage(error));
            this.watcherSuccess = false;
        }
    }
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    getErrorMessage(error) {
        var _a;
        return ((_a = error === null || error === void 0 ? void 0 : error.message) === null || _a === void 0 ? void 0 : _a.substring(error.message.indexOf(':') + 1)) || '';
    }
    /**
     * Updating status bar element when changing connection status.
     */
    onConnectionStatusChanged() {
        if (this.connectionStatusService.currentStatus === connection_status_service_1.ConnectionStatus.OFFLINE) {
            // Re-set the element only if it's visible on status bar
            if (this.entry) {
                const offlineElement = {
                    text: 'Hosted Plugin: Stopped',
                    alignment: browser_1.StatusBarAlignment.LEFT,
                    priority: 100
                };
                this.entry.className = HostedPluginController_1.HOSTED_PLUGIN_OFFLINE;
                this.statusBar.setElement(HostedPluginController_1.HOSTED_PLUGIN, offlineElement);
            }
        }
        else {
            // ask state of hosted plugin when switching to Online
            if (this.entry) {
                this.hostedPluginServer.isHostedPluginInstanceRunning().then(running => {
                    if (running) {
                        this.onHostedPluginRunning();
                    }
                    else {
                        this.onHostedPluginStopped();
                    }
                });
            }
        }
    }
    /**
     * Show menu containing actions to start/stop/restart hosted plugin.
     */
    showMenu(x, y) {
        const commands = new commands_1.CommandRegistry();
        const menu = new widgets_1.Menu({
            commands
        });
        if (this.pluginState === hosted_plugin_manager_client_1.HostedInstanceState.RUNNING) {
            this.addCommandsForRunningPlugin(commands, menu);
        }
        else if (this.pluginState === hosted_plugin_manager_client_1.HostedInstanceState.STOPPED || this.pluginState === hosted_plugin_manager_client_1.HostedInstanceState.FAILED) {
            this.addCommandsForStoppedPlugin(commands, menu);
        }
        menu.open(x, y);
    }
    /**
     * Adds commands to the menu for running plugin.
     */
    addCommandsForRunningPlugin(commands, menu) {
        commands.addCommand(hosted_plugin_manager_client_1.HostedPluginCommands.STOP.id, {
            label: 'Stop Instance',
            icon: 'fa fa-stop',
            execute: () => setTimeout(() => this.hostedPluginManagerClient.stop(), 100)
        });
        menu.addItem({
            type: 'command',
            command: hosted_plugin_manager_client_1.HostedPluginCommands.STOP.id
        });
        commands.addCommand(hosted_plugin_manager_client_1.HostedPluginCommands.RESTART.id, {
            label: 'Restart Instance',
            icon: 'fa fa-repeat',
            execute: () => setTimeout(() => this.hostedPluginManagerClient.restart(), 100)
        });
        menu.addItem({
            type: 'command',
            command: hosted_plugin_manager_client_1.HostedPluginCommands.RESTART.id
        });
    }
    /**
     * Adds command to the menu for stopped plugin.
     */
    addCommandsForStoppedPlugin(commands, menu) {
        commands.addCommand(hosted_plugin_manager_client_1.HostedPluginCommands.START.id, {
            label: 'Start Instance',
            icon: 'fa fa-play',
            execute: () => setTimeout(() => this.hostedPluginManagerClient.start(), 100)
        });
        menu.addItem({
            type: 'command',
            command: hosted_plugin_manager_client_1.HostedPluginCommands.START.id
        });
        commands.addCommand(hosted_plugin_manager_client_1.HostedPluginCommands.DEBUG.id, {
            label: 'Debug Instance',
            icon: 'fa fa-bug',
            execute: () => setTimeout(() => this.hostedPluginManagerClient.debug(), 100)
        });
        menu.addItem({
            type: 'command',
            command: hosted_plugin_manager_client_1.HostedPluginCommands.DEBUG.id
        });
    }
};
HostedPluginController.HOSTED_PLUGIN = 'hosted-plugin';
HostedPluginController.HOSTED_PLUGIN_OFFLINE = 'hosted-plugin-offline';
HostedPluginController.HOSTED_PLUGIN_FAILED = 'hosted-plugin-failed';
__decorate([
    inversify_1.inject(status_bar_1.StatusBar),
    __metadata("design:type", Object)
], HostedPluginController.prototype, "statusBar", void 0);
__decorate([
    inversify_1.inject(frontend_application_state_1.FrontendApplicationStateService),
    __metadata("design:type", frontend_application_state_1.FrontendApplicationStateService)
], HostedPluginController.prototype, "frontendApplicationStateService", void 0);
__decorate([
    inversify_1.inject(plugin_dev_protocol_1.HostedPluginServer),
    __metadata("design:type", Object)
], HostedPluginController.prototype, "hostedPluginServer", void 0);
__decorate([
    inversify_1.inject(hosted_plugin_manager_client_1.HostedPluginManagerClient),
    __metadata("design:type", hosted_plugin_manager_client_1.HostedPluginManagerClient)
], HostedPluginController.prototype, "hostedPluginManagerClient", void 0);
__decorate([
    inversify_1.inject(connection_status_service_1.ConnectionStatusService),
    __metadata("design:type", Object)
], HostedPluginController.prototype, "connectionStatusService", void 0);
__decorate([
    inversify_1.inject(hosted_plugin_log_viewer_1.HostedPluginLogViewer),
    __metadata("design:type", hosted_plugin_log_viewer_1.HostedPluginLogViewer)
], HostedPluginController.prototype, "hostedPluginLogViewer", void 0);
__decorate([
    inversify_1.inject(hosted_plugin_preferences_1.HostedPluginPreferences),
    __metadata("design:type", Object)
], HostedPluginController.prototype, "hostedPluginPreferences", void 0);
__decorate([
    inversify_1.inject(browser_1.PreferenceServiceImpl),
    __metadata("design:type", browser_1.PreferenceServiceImpl)
], HostedPluginController.prototype, "preferenceService", void 0);
__decorate([
    inversify_1.inject(common_1.MessageService),
    __metadata("design:type", common_1.MessageService)
], HostedPluginController.prototype, "messageService", void 0);
HostedPluginController = HostedPluginController_1 = __decorate([
    inversify_1.injectable()
], HostedPluginController);
exports.HostedPluginController = HostedPluginController;
//# sourceMappingURL=hosted-plugin-controller.js.map