/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/
var __values = (this && this.__values) || function(o) {
    var s = typeof Symbol === "function" && Symbol.iterator, m = s && o[s], i = 0;
    if (m) return m.call(o);
    if (o && typeof o.length === "number") return {
        next: function () {
            if (o && i >= o.length) o = void 0;
            return { value: o && o[i++], done: !o };
        }
    };
    throw new TypeError(s ? "Object is not iterable." : "Symbol.iterator is not defined.");
};
import * as platform from '../../registry/common/platform';
import { ThemeIcon } from './themeService';
import { Emitter } from '../../../base/common/event';
import { localize } from '../../../nls';
import { Extensions as JSONExtensions } from '../../jsonschemas/common/jsonContributionRegistry';
import { RunOnceScheduler } from '../../../base/common/async';
import * as Codicons from '../../../base/common/codicons';
//  ------ API types
// icon registry
export var Extensions = {
    IconContribution: 'base.contributions.icons'
};
var IconRegistry = /** @class */ (function () {
    function IconRegistry() {
        this._onDidChange = new Emitter();
        this.onDidChange = this._onDidChange.event;
        this.iconSchema = {
            definitions: {
                icons: {
                    type: 'object',
                    properties: {
                        fontId: { type: 'string', description: localize('iconDefintion.fontId', 'The id of the font to use. If not set, the font that is defined first is used.') },
                        fontCharacter: { type: 'string', description: localize('iconDefintion.fontCharacter', 'The font character associated with the icon definition.') }
                    },
                    additionalProperties: false,
                    defaultSnippets: [{ body: { fontCharacter: '\\\\e030' } }]
                }
            },
            type: 'object',
            properties: {}
        };
        this.iconReferenceSchema = { type: 'string', pattern: "^" + Codicons.CSSIcon.iconNameExpression + "$", enum: [], enumDescriptions: [] };
        this.iconsById = {};
        this.iconFontsById = {};
    }
    IconRegistry.prototype.registerIcon = function (id, defaults, description, deprecationMessage) {
        var existing = this.iconsById[id];
        if (existing) {
            if (description && !existing.description) {
                existing.description = description;
                this.iconSchema.properties[id].markdownDescription = description + " $(" + id + ")";
                var enumIndex = this.iconReferenceSchema.enum.indexOf(id);
                if (enumIndex !== -1) {
                    this.iconReferenceSchema.enumDescriptions[enumIndex] = description;
                }
                this._onDidChange.fire();
            }
            return existing;
        }
        var iconContribution = { id: id, description: description, defaults: defaults, deprecationMessage: deprecationMessage };
        this.iconsById[id] = iconContribution;
        var propertySchema = { $ref: '#/definitions/icons' };
        if (deprecationMessage) {
            propertySchema.deprecationMessage = deprecationMessage;
        }
        if (description) {
            propertySchema.markdownDescription = description + ": $(" + id + ")";
        }
        this.iconSchema.properties[id] = propertySchema;
        this.iconReferenceSchema.enum.push(id);
        this.iconReferenceSchema.enumDescriptions.push(description || '');
        this._onDidChange.fire();
        return { id: id };
    };
    IconRegistry.prototype.deregisterIcon = function (id) {
        delete this.iconsById[id];
        delete this.iconSchema.properties[id];
        var index = this.iconReferenceSchema.enum.indexOf(id);
        if (index !== -1) {
            this.iconReferenceSchema.enum.splice(index, 1);
            this.iconReferenceSchema.enumDescriptions.splice(index, 1);
        }
        this._onDidChange.fire();
    };
    IconRegistry.prototype.getIcons = function () {
        var _this = this;
        return Object.keys(this.iconsById).map(function (id) { return _this.iconsById[id]; });
    };
    IconRegistry.prototype.getIcon = function (id) {
        return this.iconsById[id];
    };
    IconRegistry.prototype.getIconSchema = function () {
        return this.iconSchema;
    };
    IconRegistry.prototype.getIconReferenceSchema = function () {
        return this.iconReferenceSchema;
    };
    IconRegistry.prototype.registerIconFont = function (id, definition) {
        var existing = this.iconFontsById[id];
        if (existing) {
            return existing;
        }
        var iconFontContribution = { id: id, definition: definition };
        this.iconFontsById[id] = iconFontContribution;
        this._onDidChange.fire();
        return iconFontContribution;
    };
    IconRegistry.prototype.deregisterIconFont = function (id) {
        delete this.iconFontsById[id];
    };
    IconRegistry.prototype.getIconFonts = function () {
        var _this = this;
        return Object.keys(this.iconFontsById).map(function (id) { return _this.iconFontsById[id]; });
    };
    IconRegistry.prototype.getIconFont = function (id) {
        return this.iconFontsById[id];
    };
    IconRegistry.prototype.toString = function () {
        var e_1, _a, e_2, _b;
        var _this = this;
        var sorter = function (i1, i2) {
            return i1.id.localeCompare(i2.id);
        };
        var classNames = function (i) {
            while (ThemeIcon.isThemeIcon(i.defaults)) {
                i = _this.iconsById[i.defaults.id];
            }
            return "codicon codicon-" + (i ? i.id : '');
        };
        var reference = [];
        reference.push("| preview     | identifier                        | default codicon ID                | description");
        reference.push("| ----------- | --------------------------------- | --------------------------------- | --------------------------------- |");
        var contributions = Object.keys(this.iconsById).map(function (key) { return _this.iconsById[key]; });
        try {
            for (var _c = __values(contributions.filter(function (i) { return !!i.description; }).sort(sorter)), _d = _c.next(); !_d.done; _d = _c.next()) {
                var i = _d.value;
                reference.push("|<i class=\"" + classNames(i) + "\"></i>|" + i.id + "|" + (ThemeIcon.isThemeIcon(i.defaults) ? i.defaults.id : i.id) + "|" + (i.description || '') + "|");
            }
        }
        catch (e_1_1) { e_1 = { error: e_1_1 }; }
        finally {
            try {
                if (_d && !_d.done && (_a = _c.return)) _a.call(_c);
            }
            finally { if (e_1) throw e_1.error; }
        }
        reference.push("| preview     | identifier                        ");
        reference.push("| ----------- | --------------------------------- |");
        try {
            for (var _e = __values(contributions.filter(function (i) { return !ThemeIcon.isThemeIcon(i.defaults); }).sort(sorter)), _f = _e.next(); !_f.done; _f = _e.next()) {
                var i = _f.value;
                reference.push("|<i class=\"" + classNames(i) + "\"></i>|" + i.id + "|");
            }
        }
        catch (e_2_1) { e_2 = { error: e_2_1 }; }
        finally {
            try {
                if (_f && !_f.done && (_b = _e.return)) _b.call(_e);
            }
            finally { if (e_2) throw e_2.error; }
        }
        return reference.join('\n');
    };
    return IconRegistry;
}());
var iconRegistry = new IconRegistry();
platform.Registry.add(Extensions.IconContribution, iconRegistry);
export function registerIcon(id, defaults, description, deprecationMessage) {
    return iconRegistry.registerIcon(id, defaults, description, deprecationMessage);
}
export function getIconRegistry() {
    return iconRegistry;
}
function initialize() {
    var e_3, _a;
    try {
        for (var _b = __values(Codicons.iconRegistry.all), _c = _b.next(); !_c.done; _c = _b.next()) {
            var icon = _c.value;
            iconRegistry.registerIcon(icon.id, icon.definition, icon.description);
        }
    }
    catch (e_3_1) { e_3 = { error: e_3_1 }; }
    finally {
        try {
            if (_c && !_c.done && (_a = _b.return)) _a.call(_b);
        }
        finally { if (e_3) throw e_3.error; }
    }
    Codicons.iconRegistry.onDidRegister(function (icon) { return iconRegistry.registerIcon(icon.id, icon.definition, icon.description); });
}
initialize();
export var iconsSchemaId = 'vscode://schemas/icons';
var schemaRegistry = platform.Registry.as(JSONExtensions.JSONContribution);
schemaRegistry.registerSchema(iconsSchemaId, iconRegistry.getIconSchema());
var delayer = new RunOnceScheduler(function () { return schemaRegistry.notifySchemaChanged(iconsSchemaId); }, 200);
iconRegistry.onDidChange(function () {
    if (!delayer.isScheduled()) {
        delayer.schedule();
    }
});
//setTimeout(_ => console.log(iconRegistry.toString()), 5000);
// common icons
export var widgetClose = registerIcon('widget-close', Codicons.Codicon.close, localize('widgetClose', 'Icon for the close action in widgets.'));
export var gotoPreviousLocation = registerIcon('goto-previous-location', Codicons.Codicon.arrowUp, localize('previousChangeIcon', 'Icon for goto previous editor location.'));
export var gotoNextLocation = registerIcon('goto-next-location', Codicons.Codicon.arrowDown, localize('nextChangeIcon', 'Icon for goto next editor location.'));
export var syncing = ThemeIcon.modify(Codicons.Codicon.sync, 'spin');
